/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Sbastien Minguet (Mia-Software) - initial API and implementation
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 *******************************************************************************/

package org.eclipse.gmt.modisco.j2se5;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.common.core.utils.Tools;
import org.eclipse.gmt.modisco.common.core.utils.UriUtils;
import org.eclipse.gmt.modisco.j2se5.io.java.JavaReader;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.m2m.atl.common.ATLLaunchConstants;
import org.eclipse.m2m.atl.core.ATLCoreException;
import org.eclipse.m2m.atl.core.IExtractor;
import org.eclipse.m2m.atl.core.IInjector;
import org.eclipse.m2m.atl.core.IModel;
import org.eclipse.m2m.atl.core.IReferenceModel;
import org.eclipse.m2m.atl.core.ModelFactory;
import org.eclipse.m2m.atl.core.launch.ILauncher;
import org.eclipse.m2m.atl.core.service.CoreService;
import org.eclipse.m2m.atl.core.ui.vm.asm.ASMModelWrapper;
import org.eclipse.m2m.atl.drivers.emf4atl.ASMEMFModel;
import org.eclipse.m2m.atl.engine.vm.nativelib.ASMModel;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;

public class J2SE5Discoverer implements J2se5DiscovererInterface {
	//resources location
	private final String kdmMMUri ="http://www.eclipse.org/MoDisco/kdm/action";
	private final String j2se5MMUri ="http://www.eclipse.org/MoDisco/J2SE5";
	
	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.j2se5.J2se5DiscovererInterface#getJ2SE5ModelFromJavaSource(org.eclipse.jdt.core.IJavaProject)
	 */
	public Resource getJ2SE5ModelFromJavaSource(IJavaProject javaProject) { 
		return JavaReader.readModel(javaProject);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.j2se5.J2se5DiscovererInterface#getKDMModelFromJavaSource(org.eclipse.jdt.core.IJavaProject)
	 */
	public Resource getKDMModelFromJavaSource(IJavaProject javaProject) throws ATLCoreException { 		
		Resource javaModel = this.getJ2SE5ModelFromJavaSource(javaProject);
		Resource kdmModel = null;
		try {
//			IPath path  = javaProject.getCorrespondingResource().getLocation().addTrailingSeparator().append(javaProject.getElementName()).addFileExtension("j2se5");
			IPath path  = javaProject.getCorrespondingResource().getFullPath().append(javaProject.getElementName()).addFileExtension("j2se5");
			Tools.saveModel(javaModel, URI.createPlatformResourceURI(path.toString(), false));
			//Testing success of javaModel saving before transforming to kdm would ensure integrity
			//but there are still minor problems with anonymous classes which need to be solved
			//so continue execution no matter what
			//if (Tools.saveModel(javaModel, path.toString())){
				kdmModel = this.getKDMModelFromJ2SE5Model(javaModel);
			//}				
		} catch (JavaModelException e) {			
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return kdmModel;
	}
	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.j2se5.J2se5DiscovererInterface#getKDMModelFromJavaSource(org.eclipse.jdt.core.IJavaProject, org.eclipse.emf.common.util.URI)
	 */
	public Resource getKDMModelFromJavaSource(IJavaProject javaProject, URI kdmModelUri) throws ATLCoreException { 		
		Resource javaModel = this.getJ2SE5ModelFromJavaSource(javaProject);
		Resource kdmModel = null;
		try {
			Tools.saveModel(javaModel, kdmModelUri.trimFileExtension().appendFileExtension("j2se5"));
			//Testing success of javaModel saving before transforming to kdm would ensure integrity
			//but there are still minor problems with anonymous classes which need to be solved
			//so continue execution no matter what
			//if (Tools.saveModel(javaModel, path.toString())){
				kdmModel = getKDMModelFromJ2SE5Model(javaModel.getURI(), kdmModelUri);
			//}				
		} catch (IOException e) {
			e.printStackTrace();
		}
		return kdmModel;
	}
	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.j2se5.J2se5DiscovererInterface#getKDMModelFromJ2SE5Model(org.eclipse.emf.ecore.resource.Resource)
	 */
	public Resource getKDMModelFromJ2SE5Model(Resource J2SE5Model) throws IOException, ATLCoreException {
		return this.getKDMModelFromJ2SE5Model(J2SE5Model.getURI(), null);
	}
	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.j2se5.J2se5DiscovererInterface#getKDMModelFromJ2SE5Model(org.eclipse.emf.ecore.resource.Resource, org.eclipse.emf.common.util.URI)
	 */
	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.j2se5.J2se5DiscovererInterface#getKDMModelFromJ2SE5Model(org.eclipse.emf.common.util.URI, org.eclipse.emf.common.util.URI)
	 */
	public Resource getKDMModelFromJ2SE5Model(URI j2se5SourceModelUri, URI kdmModelUri) throws IOException, ATLCoreException {
		//get transformation rules URL
		URL transformation = this.getClass().getResource("resources/transformations/J2SE5ToKDM.asm");
		return this.getKDMModelFromJ2SE5ModelWithCustomTransformation(j2se5SourceModelUri, transformation, kdmModelUri);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.j2se5.J2se5DiscovererInterface#exportJ2se5ToKdmTransformation(org.eclipse.core.runtime.IPath)
	 */
	public void exportJ2se5ToKdmTransformation(final IPath pathParameter) {
		final InputStream transfoFileStream = this.getClass()
				.getResourceAsStream("resources/transformations/J2SE5ToKDM.atl");
		IPath path = pathParameter;
		if (path.getFileExtension() == null
				|| !path.getFileExtension().equals("atl")) {
			path = path.addFileExtension("atl");
		}
		final IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(
				path);
		final IProject destinationProject = file.getProject();
		Job job = new Job("exporting ATL transformation") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				try {
					/*
					 *  quel que soit la mthode utilise,
					 *   le flux transfoFileStream sera correctement ferm
					 */
					if (file.exists()) {
						file.setContents(transfoFileStream, IResource.FORCE,
								monitor);
					} else {
						file.create(transfoFileStream, IResource.FORCE, monitor);
					}
				} catch (CoreException e) {
					return Status.CANCEL_STATUS;
				}
				return Status.OK_STATUS;
			}
		};
		// when copy job is done, open the file
		job.addJobChangeListener(new JobChangeAdapter() {
			@Override
			public void done(IJobChangeEvent event) {
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						try {
							destinationProject.refreshLocal(
									IResource.DEPTH_INFINITE,
									new NullProgressMonitor());
							IWorkbenchPage page = PlatformUI.getWorkbench()
									.getActiveWorkbenchWindow().getActivePage();
							IDE.openEditor(page, file);
						} catch (Exception e) {
							e.printStackTrace();
						}
					}
				});
			}
		});
		job.schedule();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.j2se5.J2se5DiscovererInterface#getKDMModelFromJ2SE5ModelWithCustomTransformation(org.eclipse.emf.common.util.URI, java.net.URL, org.eclipse.emf.common.util.URI)
	 */
	public Resource getKDMModelFromJ2SE5ModelWithCustomTransformation(
			URI j2se5SourceModelUri, final URL transformation, URI kdmTargetModelUri) throws IOException, ATLCoreException {
		Map<String, String> modelHandlers = new HashMap<String, String>();
		String j2se5MetaModelName = "j2se5";
		modelHandlers.put(j2se5MetaModelName, "EMF");
		String kdmMetaModelName = "kdm";
		modelHandlers.put(kdmMetaModelName, "EMF");
    	
    	Map<String, Object> launcherOptions = new HashMap<String, Object>();
    	launcherOptions.put(ATLLaunchConstants.OPTION_MODEL_HANDLER, modelHandlers);
		
		String launcherName = ATLLaunchConstants.REGULAR_VM_NAME;
		final ILauncher launcher = CoreService.getLauncher(launcherName);
		launcher.initialize(launcherOptions);

		ModelFactory factory = CoreService.createModelFactory(launcher.getDefaultModelFactoryName());

		IInjector injector = CoreService.getInjector(factory.getDefaultInjectorName());
		IExtractor extractor = CoreService.getExtractor(factory.getDefaultExtractorName());

		// chargement du mta modle Java
		Map<String, Object> referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(j2se5MetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", j2se5MetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", this.j2se5MMUri); //$NON-NLS-1$
		IReferenceModel j2se5MM = factory.newReferenceModel(referenceModelOptions);
		injector.inject(j2se5MM, this.j2se5MMUri);
		
		// chargement du modle Java
		Map<String, Object> modelOptions = new HashMap<String, Object>();
		String inModelName = "IN";
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", UriUtils.toString(j2se5SourceModelUri)); //$NON-NLS-1$
		modelOptions.put("newModel", false); //$NON-NLS-1$
		IModel input = factory.newModel(j2se5MM, modelOptions);
		injector.inject(input, UriUtils.toString(j2se5SourceModelUri));
		launcher.addInModel(input, inModelName, j2se5MetaModelName);

		// Chargement du mta modle KDM
		referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(kdmMetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", kdmMetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", this.kdmMMUri); //$NON-NLS-1$
		IReferenceModel kdmMM = factory.newReferenceModel(referenceModelOptions);
		injector.inject(kdmMM, this.kdmMMUri);
		// Chargement du modle KDM
		modelOptions = new HashMap<String, Object>();
		inModelName = "OUT";
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		if (kdmTargetModelUri != null) {
			modelOptions.put("path", UriUtils.toString(kdmTargetModelUri)); //$NON-NLS-1$
		} else {
			modelOptions.put("path", UriUtils.toString(j2se5SourceModelUri.trimFileExtension().appendFileExtension("kdm")));
		}
		modelOptions.put("newModel", true); //$NON-NLS-1$
		IModel outputInstance = factory.newModel(kdmMM, modelOptions);
		launcher.addOutModel(outputInstance, inModelName, kdmMetaModelName);

		/*
		 * encapsulation de la transformation dans un nouveau Thread pour
		 * obtenir une pile d'excution vide.
		 */
		final Map<String, Object> options = new HashMap<String, Object>();
		options.put("continueAfterError", "true");
		options.put("printExecutionTime", "true");

		Job transformationThread = new Job("Convert J2SE5 model to KDM") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				IStatus result = Status.OK_STATUS;
				try {
					launcher.launch(ILauncher.RUN_MODE, monitor, options,
							transformation.openStream());
				} catch (IOException e) {
					result = Status.CANCEL_STATUS;
					e.printStackTrace();
				}

				return result;
			}
		};
		transformationThread.schedule();
		try {
			transformationThread.join();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		
		if (kdmTargetModelUri != null) {
			extractor.extract(outputInstance, UriUtils.toString(kdmTargetModelUri));
		}
		
		Resource output = null;
		if (outputInstance instanceof ASMModelWrapper) {
			ASMModelWrapper wrapper = (ASMModelWrapper) outputInstance;
			ASMModel model = wrapper.getAsmModel();
			if (model instanceof ASMEMFModel) {
				output = ((ASMEMFModel) model).getExtent();
			}
		}
		return output;
	}
}
