/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 *******************************************************************************/

package org.eclipse.gmt.modisco.j2se5.actions;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.common.core.logging.AtlLogHandler;
import org.eclipse.gmt.modisco.common.core.utils.Tools;
import org.eclipse.gmt.modisco.discoverersmanager.Discoverer;
import org.eclipse.gmt.modisco.j2se5.J2SE5Discoverer;
import org.eclipse.gmt.modisco.j2se5.J2se5DiscovererInterface;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.m2m.atl.common.ATLLogger;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;

/**
 * @author Gabriel Barbier
 * 
 */
public class DiscoverJ2se5ModelFromJavaProject implements Discoverer {

	private final String targetExtension = "j2se5";
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.gmt.modisco.discoverersmanager.DiscovererInterface#applyTo
	 * (java.lang.Object)
	 */
	public boolean applyTo(final Object source) {
		boolean result = false;
		if (source instanceof IJavaProject) {
			result = true;
		} else if (source instanceof IProject) {
			IProject project = (IProject) source;
			try {
				if (project.hasNature(JavaCore.NATURE_ID)) {
					result = true;
				}
			} catch (CoreException e) {
				e.printStackTrace();
			}
		}
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seeorg.eclipse.gmt.modisco.discoverersmanager.DiscovererInterface#
	 * discoverElement(java.lang.Object, org.eclipse.emf.common.util.URI)
	 */
	public Resource discoverElement(final Object source, final URI target) {
		Resource resource = null;
		final Logger logger = Logger.getLogger(ATLLogger.LOGGER_ID);
		// retrieval of Java project
		IJavaProject tempProject = null;
		if (source instanceof IJavaProject) {
			tempProject = (IJavaProject) source;
		} else if (source instanceof IProject) {
			IProject project = (IProject) source;
			try {
				if (project.hasNature(JavaCore.NATURE_ID)) {
					tempProject = JavaCore.create(project);
				}
			} catch (CoreException e) {
				e.printStackTrace();
			}
		}
		final IJavaProject project = tempProject;

		// retrieval of target path
		IPath tempPath = project.getProject().getLocation()
				.addTrailingSeparator().append(project.getElementName())
				.addFileExtension(this.targetExtension);
		if (target != null) {
			if (target.isRelative()) {
				tempPath = ResourcesPlugin.getWorkspace().getRoot()
						.getFullPath().append(target.path());
			} else {
				tempPath = new Path(target.toFileString());
			}
		}

		final IPath path = tempPath;

		final AtlLogHandler logHandler = new AtlLogHandler(path.toString()
				+ ".log");
		logger.addHandler(logHandler);
		try {
			final J2se5DiscovererInterface disco = new J2SE5Discoverer();
			resource = disco.getJ2SE5ModelFromJavaSource(project);
			if (target != null) {
				Tools.saveModel(resource, target);
			} else {
				Tools.saveModel(resource, path.toString());
			}
		} catch (Exception e) {
			logger.log(Level.WARNING, "Error in discovery of java project", e);
		} finally {
			logger.removeHandler(logHandler);
			logHandler.close();
			try {
				project.getCorrespondingResource().refreshLocal(1, null);
			} catch (Exception e1) {
				// give up
			}
		}
		// access UI resources so that we can obtain the current eclipse
		// workbench activePage
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				try {
					IWorkspaceRoot root = ResourcesPlugin.getWorkspace()
							.getRoot();
					IFile ifile = root.getFileForLocation(path);
					if ((ifile != null) && ifile.exists()) {
						IWorkbenchPage page = PlatformUI.getWorkbench()
								.getActiveWorkbenchWindow().getActivePage();
						// programmatically opens the j2se5 model using the
						// default editor if one is present
						IDE.openEditor(page, ifile);
					}
				} catch (PartInitException e) {
					e.printStackTrace();
				}
			}
		});

		return resource;
	}

	public String toString() {
		return "Discover J2SE5 model from Java project";
	}

	public DiscoverJ2se5ModelFromJavaProject() {
		for (String key : this.keys) {
			this.parameterKeys.add(key);
		}
	}

	private final Set<String> parameterKeys = new HashSet<String>();
	private final String[] keys = {"TargetModel", "TargetUri"};
	
	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.discoverersmanager.DiscovererInterface#getParameterKeys()
	 */
	public Set<String> getParameterKeys() {
		return this.parameterKeys;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.discoverersmanager.Discoverer#discoverElement(java.lang.Object, java.util.Map)
	 */
	public void discoverElement(final Object source, final Map<String, Object> parameters) {
		URI targetUri = null;
		if (parameters.containsKey(this.keys[1])) {
			Object value = parameters.get(this.keys[1]);
			if (value instanceof URI) {
				targetUri = (URI) value;
			}
		}
		Resource result = this.discoverElement(source, targetUri);
		parameters.put(this.keys[0], result);	
	}
}
