/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Sbastien Minguet (Mia-Software) - initial API and implementation
 *    Frdric Madiot (Mia-Software) - initial API and implementation
 *    Fabien Giquel (Mia-Software) - initial API and implementation
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 *    Erwan Breton (Sodifrance) - initial API and implementation
 *******************************************************************************/
package org.eclipse.gmt.modisco.j2se5.io.java;


import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;

import org.eclipse.gmt.modisco.j2se5.ASTNode;
import org.eclipse.gmt.modisco.j2se5.BodyDeclaration;
import org.eclipse.gmt.modisco.j2se5.Comment;



/**
 * The aim of this class is to resolve one limitation with JDT use :
 * JDT does not attach comments (line  and block comments) to model nodes.
 * The difficulty is to find in the model graph the right container node.
 * On a node, there are potentially 3 positions for a comment : before, enclosed, after
 * but in most cases "before" is the position (since a comment "after" a node is a comment "before" next one)
 * In some specific cases, "enclosed", "after" are necessary :
 * - comments in last line of a file
 * - comments in last line of a block
 * - comments between "package" and "class" sections
 * 
 * @author fgiquel
 */
public class CommentsManager {

	public static void resolveCommentPositions(JDTVisitor visitor) {
		org.eclipse.jdt.core.dom.CompilationUnit cuNode = visitor.getCuNode();
		List<Comment> commentsList = new ArrayList<Comment>(visitor.getCommentsBinding().getValues());
		/*
		 * Algorithme de positionnement des commentaires
		 * 
		 * Il faut tout d'abord une liste ordonne de commentaires,
		 * cette liste est fournie par l'objet CompilationUnit (cuNode.getCommentList())
		 * 
		 * Ensuite, il faut faire une passe pour ordonner les noeuds
		 * portant des commentaires. En effet, comme seul les index 
		 * de dbut et de fin sont donns, il y a des recouvrements
		 * puisque les noeuds sont emboits.
		 * Par exemple, pour la classe racine, l'index de dbut sera 2
		 * et l'index de fin sera 11. Pour une mthode dfinie dans cette
		 * classe racine, l'index de dbut sera 4 et l'index de fin sera 6.
		 * Le tri se fera de manire  obtenir en tte de liste 
		 * le noeud dont l'index de dbut de commentaires est 
		 * le plus grand. Et en fin de liste, le noeud dont l'index
		 * de dbut de commentaires est le plus petit.
		 */
		SortedMap<Integer, org.eclipse.jdt.core.dom.ASTNode> nodesMap =
			new TreeMap<Integer, org.eclipse.jdt.core.dom.ASTNode>(new Comparator<Integer>() {
				public int compare(Integer o1, Integer o2) {
					return - o1.compareTo(o2);
				}				
			});
		for (org.eclipse.jdt.core.dom.ASTNode node : visitor.getBijectiveMap().getKeys()) {
			int index = cuNode.firstLeadingCommentIndex(node);
			if (index != -1) {
				nodesMap.put(index, node);
			}
		}
		for (Integer indexMap : nodesMap.keySet()) {
			int index = indexMap;
			org.eclipse.jdt.core.dom.ASTNode node = nodesMap.get(indexMap);
			int lastIndex = cuNode.lastTrailingCommentIndex(node);
			ASTNode element = visitor.getBijectiveMap().getValue(node);
			
			while (index != -1) {
				org.eclipse.jdt.core.dom.ASTNode commentNode = (org.eclipse.jdt.core.dom.ASTNode) cuNode.getCommentList().get(index);
				Comment comment = visitor.getCommentsBinding().get(commentNode);
				if ((comment != null) && (commentsList.contains(comment))) {
					if (index < lastIndex) {
						index ++;
						// positionnement du commentaire
						element.getComments().add(comment);
						// suppression de la liste temporaire
						commentsList.remove(comment);
					} else {
						index = -1;
					}
				} else {
					index = -1;
				}
			}
		}
		
		/*
		 *  vrification de la prise en compte de tous les commentaires
		 *  En effet, les commentaires restants seront rattachs par dfaut au root type.
		 */
		for (Comment comment : commentsList) {
			if (comment.getOriginalStartPosition() 
					< visitor.getRootTypeOrEnum().getOriginalStartPosition()) {
				visitor.getRootTypeOrEnum().getCommentsBeforeBody().add(comment);
			} else if (comment.getOriginalStartPosition() 
					< (visitor.getRootTypeOrEnum().getOriginalStartPosition() 
							+ visitor.getRootTypeOrEnum().getOriginalLength())) {
				visitor.getRootTypeOrEnum().getComments().add(comment);
			} else {
				visitor.getRootTypeOrEnum().getCommentsAfterBody().add(comment);
			}
		}
	}

	/*
	 * Adjust bodyDeclaration starts and length, to avoid beginning at the javadoc start position
	 */
	public static void adjustBodyPosition(org.eclipse.jdt.core.dom.BodyDeclaration node, BodyDeclaration element,
						Collection<org.eclipse.jdt.core.dom.Comment> comments, String javaFileContent) {
		if (node.getJavadoc() != null && element.getOriginalStartPosition() == node.getJavadoc().getStartPosition()) {
			//
			// here we imagine we have the config as e.g :
			// --- /**
			// ---  * a Java Doc
			// ---  */
			// --- private String anAttribute;
			element.setOriginalStartPosition(element.getOriginalStartPosition() + node.getJavadoc().getLength());
			element.setOriginalLength(element.getOriginalLength() - node.getJavadoc().getLength());
			
			// Another adjustement is necessary if some comments between javadoc and body node
			for (Iterator<org.eclipse.jdt.core.dom.Comment> itComments = comments.iterator(); itComments.hasNext();) {
				org.eclipse.jdt.core.dom.Comment aComment = itComments.next();
				if (aComment != node.getJavadoc() && 
						aComment.getStartPosition() >= element.getOriginalStartPosition() 
						&& aComment.getStartPosition() < (element.getOriginalStartPosition() + element.getOriginalLength())) {
					String interComments = javaFileContent.substring(element.getOriginalStartPosition(), aComment.getStartPosition());
					if (interComments.trim().length() == 0) {
						//
						// here we have the config as e.g :
						// --- /**
						// ---  * a Java Doc
						// ---  */
						// --- // a comment
						// --- private String anAttribute;
						int previousOriginalPosition = element.getOriginalStartPosition();
						element.setOriginalStartPosition(aComment.getStartPosition() + aComment.getLength());
						element.setOriginalLength(element.getOriginalLength() - aComment.getLength()
								- (aComment.getStartPosition() - previousOriginalPosition));
					}
				}
			}
		}
	}

	/**
	 * @param comment
	 * @param originalFileContent
	 * @return content of comment
	 */
	public static String extractCommentContent(org.eclipse.jdt.core.dom.Comment comment,
			String originalFileContent) {
		String result = originalFileContent.substring(comment.getStartPosition(), 
				comment.getStartPosition() + comment.getLength());
		return result;
	}

}