/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;

/** Utility functions related to EMF */
public class EMFUtil {

	/** @return whether the given EObject is in the first resource of its resource set */
	public static boolean isInFirstResource(EObject eObject) {
		Resource resource = eObject.eResource();
		if (resource == null)
			return false;
		ResourceSet resourceSet = resource.getResourceSet();
		if (resourceSet == null)
			return false;
		EList<Resource> resources = resourceSet.getResources();
		if (resources.size() > 0) {
			if (resourceSet.getResources().get(0) == resource)
				return true;
		}
		return false;
	}

	/** @return the qualified name of the given metaclass */
	public static String getMetaclassQualifiedName(EClass eClass) {
		ArrayList<String> qualifiedNameParts = new ArrayList<String>();
		StringBuilder builder = new StringBuilder();

		EPackage ePackage = eClass.getEPackage();
		while (ePackage != null) {
			qualifiedNameParts.add(ePackage.getName());
			ePackage = ePackage.getESuperPackage();
		}

		for (int i = qualifiedNameParts.size() - 1; i >= 0; i--) {
			builder.append(qualifiedNameParts.get(i) + ".");
		}

		builder.append(eClass.getName());

		return builder.toString();
	}

	/**
	 * Search for instances of the given EClass
	 * 
	 * @param eClass
	 *            the EClass of the elements that are looked for
	 * @param resource
	 *            the resource to search in
	 * @return elements that are instances of the given EClass
	 */
	public Collection<EObject> findInstancesOf(EClass eClass, Resource resource) {
		ArrayList<EObject> instances = new ArrayList<EObject>();
		TreeIterator<EObject> allContents = resource.getAllContents();
		while (allContents.hasNext()) {
			EObject eObject = allContents.next();
			if (eObject.eClass() == eClass) {
				instances.add(eObject);
			}
		}
		return instances;
	}

	/**
	 * Find and return all the metaclasses of the given resource set.
	 * 
	 * @return a list of all the metaclasses of elements contained in the resource set, and
	 *         metaclasses in resources in which metaclasses of elements in the resource set have
	 *         been found
	 */
	public static Collection<EClass> findAllClasses(ResourceSet resourceSet) {

		// TODO: find EClasses in EPackages

		/*
		 * the list of metaclasses is a tree set so that it is always sorted and each class is
		 * guaranteed to appear at most once.
		 */
		TreeSet<EClass> classes = new TreeSet<EClass>(new Comparator<EClass>() {
			public int compare(EClass c1, EClass c2) {
				return EMFUtil.getMetaclassQualifiedName(c1).compareTo(
						EMFUtil.getMetaclassQualifiedName(c2));
			}
		});

		/** The resources in the resource set */
		EList<Resource> resourcesList = resourceSet.getResources();
		/** The resources, in a HashSet to get constant time access */
		HashSet<Resource> resources = new HashSet<Resource>();
		/**
		 * Resources that were discovered outside of the resource set, and which contain metaclasses
		 * used in the model
		 */
		ArrayList<Resource> newResources = new ArrayList<Resource>();

		resources.addAll(resourcesList);

		for (Resource resource : resourcesList) {
			TreeIterator<EObject> allContents = resource.getAllContents();
			while (allContents.hasNext()) {
				EObject eObject = allContents.next();
				if (eObject instanceof EClass) {
					EClass eClass = (EClass) eObject;
					classes.add(eClass);
				} else {
					// EPackage ePackage = eObject.eClass().getEPackage();
					// System.out.println("package : " + ePackage.getNsURI());

					EClass eClass = eObject.eClass();
					if (eClass != null) {
						// make sure all model elements have their metaclass in the list
						classes.add(eClass);

						// if we discovered a new resource, then memorize it
						Resource res = eClass.eResource();
						if (res != null && !resources.contains(res)) {
							resources.add(res);
							newResources.add(res);
						}
					}
				}
			}
		}

		/*
		 * Now, add the classes that reside in resources that we discovered and are not part of the
		 * resource set. For example, the UML models have metaclasses in
		 * 'http://www.eclipse.org/uml2/2.1.0/UML', which is not in the resource set.
		 */
		addAll(classes, newResources);
		newResources.clear();

		ArrayList<EClass> newClasses = new ArrayList<EClass>();

		for (EClass eClass : classes) {
			EList<EClass> allSuperTypes = eClass.getEAllSuperTypes();
			for (EClass superclass : allSuperTypes) {
				newClasses.add(superclass);

				// if we discovered a new resource, then memorize it
				Resource resource = superclass.eResource();
				if (resource != null && !resources.contains(resource)) {
					resources.add(resource);
					newResources.add(resource);
				}
			}
		}

		for (EClass eClass : newClasses) {
			classes.add(eClass);
		}

		// add classes in newly discovered resources (again)
		addAll(classes, newResources);

		// for (Resource resource : resources) {
		// System.out.println(resource.getURI().toString());
		// }

		return classes;
	}

	/** Add all the classes in <code>resources</code> to the <code>classes</code> set */
	private static void addAll(Set<EClass> classes, List<Resource> resources) {
		for (Resource resource : resources) {
			TreeIterator<EObject> allContents = resource.getAllContents();
			while (allContents.hasNext()) {
				EObject eObject = allContents.next();
				if (eObject instanceof EClass) {
					EClass eClass = (EClass) eObject;
					classes.add(eClass);
				}
			}
		}
	}

	/**
	 * Find the most specific metaclass that is common to all of the given elements.
	 * 
	 * @return the common metaclass or <code>null</code> if none was found
	 */
	public static EClass findCommonClass(List<EObject> elements) {
		// FIXME this algorithm can make a choice about a branch early and not find
		// a solution, when a solution does exist
		EClass eClass = null;
		for (EObject element : elements) {
			EClass elementEClass = element.eClass();
			if (eClass == null) {
				eClass = elementEClass;
			} else {
				if (eClass.isSuperTypeOf(elementEClass)) {
					// do nothing
				} else if (elementEClass.isSuperTypeOf(eClass)) {
					eClass = elementEClass;
				} else {
					// elements are on two different branches
					// try to find a common metaclass
					eClass = findSuperType(eClass, elementEClass);
					if (eClass == null)
						return null;
				}
			}
		}
		return eClass;
	}

	/** Find the first superclass of <code>eClass</code> that is a superclass of <code>element</code> */
	private static EClass findSuperType(EClass eClass, EClass element) {
		// do a breadth-first search (with a bottom-up tree)
		LinkedList<EClass> breadthFirstList = new LinkedList<EClass>();
		breadthFirstList.addFirst(eClass);

		while (!breadthFirstList.isEmpty()) {
			EClass candidateClass = breadthFirstList.poll();

			if (candidateClass.isSuperTypeOf(element))
				return candidateClass;

			// add all the direct super-types of this class
			for (EClass supertype : candidateClass.getESuperTypes()) {
				breadthFirstList.addLast(supertype);
			}
		}

		return null;
	}
}
