/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.adapters;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.edit.provider.IEditingDomainItemProvider;
import org.eclipse.emf.edit.provider.IItemColorProvider;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.emf.edit.provider.IItemPropertyDescriptor;
import org.eclipse.emf.edit.provider.IItemPropertySource;
import org.eclipse.emf.edit.provider.IStructuredItemContentProvider;
import org.eclipse.emf.edit.provider.ITreeItemContentProvider;
import org.eclipse.emf.edit.provider.ItemPropertyDescriptorDecorator;
import org.eclipse.emf.edit.provider.ViewerNotification;
import org.eclipse.gmt.modisco.common.editor.editors.EditorConfiguration;
import org.eclipse.gmt.modisco.common.editor.util.ImageProvider;
import org.eclipse.gmt.modisco.common.editor.util.Util;

/**
 * An item provider for displaying attributes of a model element as virtual elements in the tree.
 */
public class AttributeItemProvider extends TransientItemProvider implements
		IEditingDomainItemProvider, IStructuredItemContentProvider, ITreeItemContentProvider,
		IItemLabelProvider, IItemPropertySource, IItemColorProvider {

	/** The attribute */
	protected EAttribute attribute;
	/** The model element that owns the attribute */
	protected EObject parent;
	/** The configuration of the editor in which the model is displayed */
	protected final EditorConfiguration editorConfiguration;

	/**
	 * Instantiate a new {@link AttributeItemProvider}.
	 * 
	 * @param adapterFactory
	 *            the adapter factory used to create adapters
	 * @param parent
	 *            the model element that owns the attribute
	 * @param attribute
	 *            the attribute
	 * @param editorConfiguration
	 *            the configuration of the editor in which the model is displayed
	 */
	public AttributeItemProvider(AdapterFactory adapterFactory, EObject parent,
			EAttribute attribute, EditorConfiguration editorConfiguration) {
		super(adapterFactory, parent);
		this.attribute = attribute;
		this.parent = parent;
		this.editorConfiguration = editorConfiguration;
		parent.eAdapters().add(this);
	}

	public EAttribute getAttribute() {
		return this.attribute;
	}

	@Override
	public void notifyChanged(Notification notification) {
		// fire the notification only if it matches the reference
		if (this.attribute == notification.getFeature()) {
			// update the attribute
			fireNotifyChanged(new ViewerNotification(notification, this, true, true));
			// update the label of the parent when the attribute is modified
			fireNotifyChanged(new ViewerNotification(notification, this.parent, true, true));
		}
	}

	/** @return the number of elements referenced by this attribute */
	public int getCount() {
		int count;
		Object value = this.parent.eGet(this.attribute);
		if (this.attribute.isMany()) {
			count = ((List<?>) value).size();
		} else {
			count = (value != null) ? 1 : 0;
		}
		return count;
	}

	@Override
	public String getText(Object object) {

		String multiplicity = "";

		if (this.editorConfiguration.isShowMultiplicity()) {
			String multLow = (this.attribute.isRequired() ? "1" : "0");
			String multHigh = (this.attribute.isMany() ? "*" : "1");
			multiplicity = " [" + multLow + ".." + multHigh + "]";
		}

		String prefix = (this.attribute.isDerived() ? "/" : "");

		String strCount;
		if (this.attribute.isMany()) {
			strCount = " (" + getCount() + ")";
		} else {
			strCount = "";
		}

		String strValue;
		if (!this.attribute.isMany()) {
			Object value = this.parent.eGet(this.attribute);
			if (value != null) {
				strValue = " = " + value.toString();
			} else {
				strValue = "";
			}
		} else {
			strValue = "";
		}

		strValue = Util.truncateBeforeNewline(strValue);

		return prefix + this.attribute.getName() + multiplicity + strCount + strValue;
	}

	@Override
	public Object getImage(Object object) {
		return ImageProvider.getInstance().getAttributeIcon();
	}

	@Override
	public Object getParent(Object object) {
		return this.parent;
	}

	@Override
	protected String getFeatureText(Object feature) {
		return this.attribute.getName();
	}

	@Override
	public boolean hasChildren(Object object) {
		return this.attribute.isMany() && getCount() > 0;
	}

	@Override
	public Collection<?> getChildren(Object object) {

		if (this.attribute.isMany()) {
			List<Object> childrenList = new ArrayList<Object>();
			Object value = this.parent.eGet(this.attribute);
			@SuppressWarnings("unchecked")
			List<Object> elements = (List<Object>) value;
			childrenList.addAll(elements);
			return childrenList;
		} else
			return Collections.EMPTY_LIST;
	}

	@Override
	public Object getForeground(Object object) {
		/* Show the attribute in gray if it is empty */
		if (getCount() == 0)
			return IItemColorProvider.GRAYED_OUT_COLOR;
		return super.getForeground(object);
	}

	/**
	 * Return property descriptors so that the attribute appears in the Properties view when it is
	 * selected.
	 */
	@Override
	public List<IItemPropertyDescriptor> getPropertyDescriptors(Object object) {
		if (this.itemPropertyDescriptors == null) {
			this.itemPropertyDescriptors = new ArrayList<IItemPropertyDescriptor>();

			// get the property descriptors from the parent
			MiaReflectiveItemProvider parentItemProvider = (MiaReflectiveItemProvider) this.adapterFactory
					.adapt(this.parent, IItemPropertySource.class);

			List<IItemPropertyDescriptor> parentDescriptors = parentItemProvider
					.getPropertyDescriptors(this.parent);

			for (IItemPropertyDescriptor descriptor : parentDescriptors) {
				// if the parent's descriptor matches the link's reference, then add it
				if (descriptor.getFeature(this.parent) == this.attribute) {
					this.itemPropertyDescriptors.add(new ItemPropertyDescriptorDecorator(
							this.parent, descriptor) {
						@Override
						public String getCategory(Object thisObject) {
							// no category (show at root)
							return null;
						}

						@Override
						public String getId(Object thisObject) {
							return AttributeItemProvider.this.attribute.getName()
									+ getDisplayName(thisObject);
						}

					});
				}
			}
		}
		return this.itemPropertyDescriptors;
	}
}