/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.editors;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.common.ui.viewer.IViewerProvider;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.domain.IEditingDomainProvider;
import org.eclipse.emf.edit.ui.action.CreateChildAction;
import org.eclipse.emf.edit.ui.action.CreateSiblingAction;
import org.eclipse.emf.edit.ui.action.EditingDomainActionBarContributor;
import org.eclipse.emf.edit.ui.action.ValidateAction;
import org.eclipse.gmt.modisco.common.editor.MoDiscoEditorPlugin;
import org.eclipse.gmt.modisco.common.editor.adapters.LinkItemProvider;
import org.eclipse.gmt.modisco.common.editor.editors.table.TableEditor;
import org.eclipse.gmt.modisco.common.editor.util.EMFUtil;
import org.eclipse.gmt.modisco.common.editor.util.ImageProvider;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IContributionManager;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.action.SubContributionItem;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.PartInitException;

/** Action bar contributor for the MoDisco Ecore model editor. */
public class EcoreActionBarContributor extends EditingDomainActionBarContributor implements
		ISelectionChangedListener {

	/** The ID of the "Properties" view */
	private static final String PROPERTY_SHEET_ID = "org.eclipse.ui.views.PropertySheet";

	/** This keeps track of the active editor. */
	protected IEditorPart activeEditorPart;

	/** This keeps track of the current selection provider. */
	protected ISelectionProvider selectionProvider;

	/** This action opens the "Properties" view. */
	protected IAction showPropertiesViewAction = new Action(MoDiscoEditorPlugin.INSTANCE
			.getString("_UI_ShowPropertiesView_menu_item")) {
		@Override
		public void run() {
			try {
				getPage().showView(PROPERTY_SHEET_ID);
			} catch (PartInitException exception) {
				MoDiscoEditorPlugin.INSTANCE.log(exception);
			}
		}
	};

	/** This action opens a table editor on the current selection. */
	protected IAction openTableEditorAction = new Action(MoDiscoEditorPlugin.INSTANCE
			.getString("_UI_OpenTableEditorOnSelection_menu_item")) {
		@Override
		public void run() {
			if (EcoreActionBarContributor.this.activeEditor instanceof EcoreEditor) {
				EcoreEditor editor = (EcoreEditor) EcoreActionBarContributor.this.activeEditor;
				EditorConfiguration editorConfiguration = editor.getEditorConfiguration();

				ISelection selection = EcoreActionBarContributor.this.lastSelectionChangedEvent
						.getSelection();

				if (selection instanceof IStructuredSelection) {
					IStructuredSelection structuredSelection = (IStructuredSelection) selection;

					if (structuredSelection.size() == 1
							&& structuredSelection.getFirstElement() instanceof LinkItemProvider) {
						LinkItemProvider linkItemProvider = (LinkItemProvider) structuredSelection
								.getFirstElement();

						openOn(editorConfiguration, linkItemProvider);
					} else {
						openOn(editorConfiguration, structuredSelection);
					}
				}
			}
		}

		private void openOn(EditorConfiguration editorConfiguration,
				LinkItemProvider linkItemProvider) {
			Collection<EObject> childrenElements = linkItemProvider.getChildrenElements();
			List<EObject> elementsList = new ArrayList<EObject>(childrenElements);

			String description = TableEditor.getEditorDescriptionFor((EObject) linkItemProvider
					.getParent(null), linkItemProvider.getReference(), editorConfiguration);
			TableEditor.openOn(editorConfiguration, elementsList, description);
		}

		private void openOn(EditorConfiguration editorConfiguration,
				IStructuredSelection structuredSelection) {

			List<EObject> elements = new ArrayList<EObject>();
			Iterator<?> iterator = structuredSelection.iterator();
			while (iterator.hasNext()) {
				Object selectedElement = iterator.next();
				if (selectedElement instanceof EObject) {
					EObject eObject = (EObject) selectedElement;
					elements.add(eObject);
				} else {
					MoDiscoEditorPlugin.INSTANCE.log("Cannot open table editor: "
							+ "selected objects are not all model elements");
					return;
				}
			}

			String description;
			String metamodelName = editorConfiguration.getMetamodelName();
			if (metamodelName != null) {
				description = metamodelName + " elements selection";
			} else {
				description = "elements selection (unknown metamodel)";
			}
			TableEditor.openOn(editorConfiguration, elements, description);
		}
	};

	/**
	 * This action refreshes the viewer of the current editor if the editor implements
	 * {@link org.eclipse.emf.common.ui.viewer.IViewerProvider}.
	 */
	protected IAction refreshViewerAction = new Action(MoDiscoEditorPlugin.INSTANCE
			.getString("_UI_RefreshViewer_menu_item")) {
		@Override
		public boolean isEnabled() {
			return EcoreActionBarContributor.this.activeEditorPart instanceof IViewerProvider;
		}

		@Override
		public void run() {
			if (EcoreActionBarContributor.this.activeEditorPart instanceof IViewerProvider) {
				Viewer viewer = ((IViewerProvider) EcoreActionBarContributor.this.activeEditorPart)
						.getViewer();
				if (viewer != null) {
					viewer.refresh();
				}
			}
		}
	};

	/**
	 * This will contain one {@link org.eclipse.emf.edit.ui.action.CreateChildAction} corresponding
	 * to each descriptor generated for the current selection by the item provider.
	 */
	protected Collection<IAction> createChildActions;

	/**
	 * This will contain one {@link org.eclipse.emf.edit.ui.action.CreateSiblingAction}
	 * corresponding to each descriptor generated for the current selection by the item provider.
	 */
	protected Collection<IAction> createSiblingActions;

	/** The last "selection changed" event received from the selection provider */
	protected SelectionChangedEvent lastSelectionChangedEvent;

	/** An action to show the selected element in the list of instances of its metaclass */
	private Action browseAction = new Action("Browse") {
		@Override
		public void run() {
			IEditorPart activeEditor = getActiveEditor();
			if (activeEditor instanceof EcoreEditor) {
				EcoreEditor editor = (EcoreEditor) activeEditor;
				editor.browse();
			}
		}
	};

	/** An action to increase font size */
	private Action increaseFontSizeAction = new Action("Increase Font Size", IAction.AS_PUSH_BUTTON) {
		@Override
		public void run() {
			IEditorPart activeEditor = getActiveEditor();
			if (activeEditor instanceof EcoreEditor) {
				EcoreEditor editor = (EcoreEditor) activeEditor;
				editor.setFontSizeDelta(editor.getFontSizeDelta() + 1);
			}
		}
	};

	/** An action to decrease font size */
	private Action decreaseFontSizeAction = new Action("Decrease Font Size", IAction.AS_PUSH_BUTTON) {
		@Override
		public void run() {
			IEditorPart activeEditor = getActiveEditor();
			if (activeEditor instanceof EcoreEditor) {
				EcoreEditor editor = (EcoreEditor) activeEditor;
				editor.setFontSizeDelta(editor.getFontSizeDelta() - 1);
			}
		}
	};

	/** An action to restore font size */
	private Action restoreFontSizeAction = new Action("Restore Font Size", IAction.AS_PUSH_BUTTON) {
		@Override
		public void run() {
			IEditorPart activeEditor = getActiveEditor();
			if (activeEditor instanceof EcoreEditor) {
				EcoreEditor editor = (EcoreEditor) activeEditor;
				editor.setFontSizeDelta(0);
			}
		}
	};

	/** This creates an instance of the contributor. */
	public EcoreActionBarContributor() {
		super(ADDITIONS_LAST_STYLE);
		this.validateAction = new ValidateAction();
	}

	/** This adds {@link Separator}s (for editor additions) and {@link Action}s to the tool bar. */
	@Override
	public void contributeToToolBar(IToolBarManager toolBarManager) {
		toolBarManager.add(new Separator("ecore-settings"));
		toolBarManager.add(new Separator("ecore-additions"));

		this.increaseFontSizeAction.setImageDescriptor(ImageProvider.getInstance()
				.getIncreaseFontSizeIcon());
		toolBarManager.add(this.increaseFontSizeAction);
		this.decreaseFontSizeAction.setImageDescriptor(ImageProvider.getInstance()
				.getDecreaseFontSizeIcon());
		toolBarManager.add(this.decreaseFontSizeAction);
		this.restoreFontSizeAction.setImageDescriptor(ImageProvider.getInstance()
				.getRestoreFontSizeIcon());
		toolBarManager.add(this.restoreFontSizeAction);
	}

	protected IMenuManager createSubmenuManager() {
		return new MenuManager(MoDiscoEditorPlugin.getPlugin().getString("_UI_EcoreEditor_menu"),
				"org.eclipse.emf.ecoreMenuID");
	}

	/**
	 * When the active editor changes, this remembers the change and registers with it as a
	 * selection provider.
	 */
	@Override
	public void setActiveEditor(IEditorPart part) {
		super.setActiveEditor(part);
		this.activeEditorPart = part;

		// Switch to the new selection provider.
		if (this.selectionProvider != null) {
			this.selectionProvider.removeSelectionChangedListener(this);
		}

		if (part == null) {
			this.selectionProvider = null;
		} else {
			this.selectionProvider = part.getSite().getSelectionProvider();
			this.selectionProvider.addSelectionChangedListener(this);

			// Fake a selection changed event to update the menus.
			if (this.selectionProvider.getSelection() != null) {
				selectionChanged(new SelectionChangedEvent(this.selectionProvider,
						this.selectionProvider.getSelection()));
			}
		}

		updateActions(part);
	}

	/**
	 * Update actions depending on the editor which is activated
	 * 
	 * @param editorPart
	 *            the editor which is activated
	 */
	private void updateActions(IEditorPart editorPart) {
		// if (editorPart instanceof EcoreEditor) {
		// EcoreEditor editor = (EcoreEditor) editorPart;
		// }
	}

	/**
	 * This implements {@link org.eclipse.jface.viewers.ISelectionChangedListener}, memorizing
	 * {@link org.eclipse.jface.viewers.SelectionChangedEvent}s to be able to build the menu when
	 * they are about to be shown (see {@link EcoreActionBarContributor#updateMenu()}).
	 */
	public void selectionChanged(SelectionChangedEvent event) {
		this.lastSelectionChangedEvent = event;
	}

	/** Update actions state depending on the selection in the editor */
	private void updateActions(ISelection selection) {
		this.browseAction.setEnabled(enableBrowse(selection));
		this.openTableEditorAction.setEnabled(TableEditor.canBeOpenedOnSelection(selection));
	}

	/**
	 * Whether to enable the browse action. The browse action is enabled only if the selected
	 * element is an EObject which is part of the first resource of the resource set.
	 */
	private boolean enableBrowse(ISelection selection) {
		if (selection instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			if (structuredSelection.size() != 1)
				return false;
			Object element = structuredSelection.getFirstElement();
			if (element instanceof EObject) {
				return EMFUtil.isInFirstResource((EObject) element);
			}
		}
		return false;
	}

	/**
	 * This generates a {@link org.eclipse.emf.edit.ui.action.CreateChildAction} for each object in
	 * <code>descriptors</code>, and returns the collection of these actions.
	 */
	protected Collection<IAction> generateCreateChildActions(Collection<?> descriptors,
			ISelection selection) {
		Collection<IAction> actions = new ArrayList<IAction>();
		if (descriptors != null) {
			for (Object descriptor : descriptors) {
				actions.add(new CreateChildAction(this.activeEditorPart, selection, descriptor));
			}
		}
		return actions;
	}

	/**
	 * This generates a {@link org.eclipse.emf.edit.ui.action.CreateSiblingAction} for each object
	 * in <code>descriptors</code>, and returns the collection of these actions.
	 */
	protected Collection<IAction> generateCreateSiblingActions(Collection<?> descriptors,
			ISelection selection) {
		Collection<IAction> actions = new ArrayList<IAction>();
		if (descriptors != null) {
			for (Object descriptor : descriptors) {
				actions.add(new CreateSiblingAction(this.activeEditorPart, selection, descriptor));
			}
		}
		return actions;
	}

	/**
	 * This populates the specified <code>manager</code> with
	 * {@link org.eclipse.jface.action.ActionContributionItem}s based on the
	 * {@link org.eclipse.jface.action.IAction}s contained in the <code>actions</code> collection,
	 * by inserting them before the specified contribution item <code>contributionID</code>. If
	 * <code>contributionID</code> is <code>null</code>, they are simply added.
	 * 
	 */
	protected void populateManager(IContributionManager manager,
			Collection<? extends IAction> actions, String contributionID) {
		if (actions != null) {
			for (IAction action : actions) {
				if (contributionID != null) {
					manager.insertBefore(contributionID, action);
				} else {
					manager.add(action);
				}
			}
		}
	}

	/**
	 * This removes from the specified <code>manager</code> all
	 * {@link org.eclipse.jface.action.ActionContributionItem}s based on the
	 * {@link org.eclipse.jface.action.IAction}s contained in the <code>actions</code> collection.
	 */
	protected void depopulateManager(IContributionManager manager,
			Collection<? extends IAction> actions) {
		if (actions != null) {
			IContributionItem[] items = manager.getItems();
			for (IContributionItem contributionItem : items) {
				// look into SubContributionItems
				while (contributionItem instanceof SubContributionItem) {
					contributionItem = ((SubContributionItem) contributionItem).getInnerItem();
				}

				// delete the ActionContributionItems with matching action.
				if (contributionItem instanceof ActionContributionItem) {
					IAction action = ((ActionContributionItem) contributionItem).getAction();
					if (actions.contains(action)) {
						manager.remove(contributionItem);
					}
				}
			}
		}
	}

	/** This populates the pop-up menu before it appears. */
	@Override
	public void menuAboutToShow(IMenuManager menuManager) {
		super.menuAboutToShow(menuManager);

		updateMenu();

		MenuManager submenuManager = null;

		submenuManager = new MenuManager(MoDiscoEditorPlugin.INSTANCE
				.getString("_UI_CreateChild_menu_item"));
		populateManager(submenuManager, this.createChildActions, null);
		menuManager.insertBefore("edit", submenuManager);

		submenuManager = new MenuManager(MoDiscoEditorPlugin.INSTANCE
				.getString("_UI_CreateSibling_menu_item"));
		populateManager(submenuManager, this.createSiblingActions, null);
		menuManager.insertBefore("edit", submenuManager);

		menuManager.insertBefore("edit", this.browseAction);
		menuManager.insertBefore("edit", this.openTableEditorAction);
	}

	/**
	 * De-populate and re-populate the menu based on the current selection. Query for the children
	 * and siblings that can be added to the selected object and update the menus accordingly.
	 */
	private void updateMenu() {
		// Query the new selection for appropriate new child/sibling descriptors
		Collection<?> newChildDescriptors = null;
		Collection<?> newSiblingDescriptors = null;

		ISelection selection = this.lastSelectionChangedEvent.getSelection();
		if (selection instanceof IStructuredSelection
				&& ((IStructuredSelection) selection).size() == 1) {
			Object object = ((IStructuredSelection) selection).getFirstElement();

			EditingDomain domain = ((IEditingDomainProvider) this.activeEditorPart)
					.getEditingDomain();

			newChildDescriptors = domain.getNewChildDescriptors(object, null);
			newSiblingDescriptors = domain.getNewChildDescriptors(null, object);
		}

		// Generate actions for selection; populate and redraw the menus.
		this.createChildActions = generateCreateChildActions(newChildDescriptors, selection);
		this.createSiblingActions = generateCreateSiblingActions(newSiblingDescriptors, selection);

		updateActions(selection);
	}

	/** This inserts global actions before the "additions-end" separator. */
	@Override
	protected void addGlobalActions(IMenuManager menuManager) {
		menuManager.insertAfter("additions-end", new Separator("ui-actions"));
		menuManager.insertAfter("ui-actions", this.showPropertiesViewAction);

		this.refreshViewerAction.setEnabled(this.refreshViewerAction.isEnabled());
		menuManager.insertAfter("ui-actions", this.refreshViewerAction);

		super.addGlobalActions(menuManager);
	}

	/** This ensures that a delete action will clean up all references to deleted objects. */
	@Override
	protected boolean removeAllReferencesOnDelete() {
		return true;
	}

}
