/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Payneau (Mia-Software) - initial API and implementation
 */
package org.eclipse.gmt.modisco.workflow.controller;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Platform;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.internal.ui.launchConfigurations.LaunchGroupExtension;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmt.modisco.workflow.Activator;
import org.eclipse.gmt.modisco.workflow.Constant;
import org.eclipse.gmt.modisco.workflow.WorkflowExecutionInterface;
import org.eclipse.gmt.modisco.workflow.WorkflowModelInterface;
import org.eclipse.gmt.modisco.workflow.actions.LaunchDriversWizard;
import org.eclipse.gmt.modisco.workflow.modiscowork.MoDiscoWork;
import org.eclipse.gmt.modisco.workflow.modiscowork.ModiscoworkFactory;
import org.eclipse.gmt.modisco.workflow.modiscowork.impl.MoDiscoWorkImpl;
import org.eclipse.gmt.modisco.workflow.ui.common.AdditionalButtonsComposite;
import org.eclipse.gmt.modisco.workflow.ui.common.Common;
import org.eclipse.gmt.modisco.workflow.ui.common.MoDiscoWorkParameter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.TableItem;

@SuppressWarnings("restriction")
public class ControllerWorkflowTransformation {

	private AdditionalButtonsComposite additionalButton;
	private Map<String,WorkflowModelInterface> workflowInterfaceMap;
	private Map<String,WorkflowExecutionInterface> workflowExecutionInterfaceMap;
	private LaunchDriversWizard wizardDrivers;
	private ControllerDriverPage controllerDriver;
	private String transformationNameNotValid;
	private int editionIndice;
	private LaunchGroupExtension group;
	private String currentIdWorkflow;
	
	/**
	 * Class constructor
	 */
	public ControllerWorkflowTransformation(String idWorkflow){
		this.workflowExecutionInterfaceMap = new HashMap<String, WorkflowExecutionInterface>();
		this.workflowInterfaceMap = new HashMap<String, WorkflowModelInterface>();
		this.controllerDriver = new ControllerDriverPage();
		this.wizardDrivers = new LaunchDriversWizard(this,this.controllerDriver);
		this.currentIdWorkflow = idWorkflow;
		this.initializeWorkflow();
	}
	
	/**
	 * Open the wizard to add a new MoDiscoWork in the MoDiscoWorkflow
	 */
	public void newMoDiscoWork() {
		this.wizardDrivers.run(null,Constant.addMode);
		refresh(-1);
	}
	
	/**
	 * Open the wizard to edit a present MoDiscoWork in the MoDiscoWorkflow
	 * @param index MoDiscoWork selected in the table
	 * @return true if the MoDiscoWork found otherwise false.
	 */
	public boolean editMoDiscoWork(int index) {
		boolean transformationFound = true;
		this.editionIndice = index;
		MoDiscoWorkImpl modiscoWork = getComponent(index);
		if(modiscoWork!=null){
			ILaunchConfiguration conf = getLaunchConfigurationFromName(modiscoWork.getName());
			ILaunchConfigurationWorkingCopy config = null;
			try {
				config = conf.getWorkingCopy();
			} catch (CoreException e) {
				e.printStackTrace();
			}
			if(config!=null){
				this.controllerDriver.setConfigurationEdition(config);
				this.controllerDriver.setTypeEdition(modiscoWork.getType());
				this.wizardDrivers.run(null,Constant.editMode);
			}else{
				transformationFound = false;
			}
		}
		refresh(index);
		return transformationFound;
	}
	
	/**
	 * Remove a MoDiscoWork in the MoDiscoWorkflow
	 * @param tableIndex index MoDiscoWork selected in the table
	 */
	public void removeFromWorkflow(int tableIndex){
		List<MoDiscoWorkImpl> modiscoWorkList = getAllMoDiscoWork();
		Collections.sort(modiscoWorkList);
		MoDiscoWorkImpl modisco = modiscoWorkList.get(tableIndex);
		int modiscoWorkflowIndex=0;
		for(MoDiscoWorkImpl component : getAllMoDiscoWork()){
			if(component.getIndex()==modisco.getIndex()){
				break;
			}
			modiscoWorkflowIndex++;
		}
		this.workflowInterfaceMap.get(this.currentIdWorkflow).removeComponent(modiscoWorkflowIndex);
		updateAllMoDiscoWorkIndex(modisco.getIndex());
		refresh(-1);
	}
	
	/**
	 * Allows to update all MoDiscoWork index after have deleted one MoDiscoWork
	 * @param modiscoWorkIndex index of the MoDiscoWork deleted
	 */
	private void updateAllMoDiscoWorkIndex(int modiscoWorkIndexDeleted){
		int i=0;
		for(MoDiscoWorkImpl component : getAllMoDiscoWork()){
			if(component.getIndex()>modiscoWorkIndexDeleted){
				MoDiscoWorkImpl modisco = (MoDiscoWorkImpl)this.workflowInterfaceMap.get(this.currentIdWorkflow).getComponent(i);
				modisco.setIndex(modisco.getIndex()-1);
				setComponent(i,modisco);
			}
			i++;
		}
	}
	
	/**
	 * Remove all occurrences of MoDiscoWork selected and its launcher
	 * @param indexList List of index. It corresponds to the selected MoDiscoWork and all these occurrences in the table
	 */
	public void removeFromLauncher(List<Integer> indexList){
		try {
			//0 because they have all the same of ILaunchConfiguration
			MoDiscoWorkImpl modiscoWork = getComponent(indexList.get(0));
			ILaunchConfiguration config = getLaunchConfigurationFromName(modiscoWork.getName());
			if(config.exists()){
				ILaunchConfigurationWorkingCopy wc = config.getWorkingCopy();
				if(wc!=null){
					wc.getOriginal().delete();
				}
			}
		} catch (CoreException e) {
			e.printStackTrace();
		}
		for(int index : indexList){
			removeFromWorkflow(index);
		}
		refresh(-1);
	}
	
	/**
	 * Allows to update ModiscoWork index to up the MoDiscoWork selected in the MoDiscoWorkflow
	 * @param tableIndex index of table selected
	 */
	public void upMoDiscoWork(int tableIndex) {
		int modiscoWorkflowIndex=0;
		int indexModisco1 = 0;
		int indexModisco2 = 0;
		MoDiscoWorkImpl modisco1 = null;
		MoDiscoWorkImpl modisco2 = null;
		for(MoDiscoWorkImpl component : getAllMoDiscoWork()){
			if(component.getIndex()==tableIndex){
				modisco1 = component;
				indexModisco1 = modiscoWorkflowIndex;
			}else if(component.getIndex()==tableIndex-1){
				modisco2 = component;
				indexModisco2 = modiscoWorkflowIndex;
			}else if(modisco1!=null&&modisco2!=null){
				break;
			}
			modiscoWorkflowIndex++;
		}
		modisco1.setIndex(modisco1.getIndex()-1);
		modisco2.setIndex(modisco2.getIndex()+1);
		setComponent(indexModisco1, modisco1);
		setComponent(indexModisco2, modisco2);
		refresh(tableIndex-1);
	}
	
	/**
	 * Allows to update ModiscoWork index to down the MoDiscoWork selected in the MoDiscoWorkflow
	 * @param tableIndex index of table selected
	 */
	public void downMoDiscoWork(int tableIndex) {
		int modiscoWorkflowIndex=0;
		int indexModisco1 = 0;
		int indexModisco2 = 0;
		MoDiscoWorkImpl modisco1 = null;
		MoDiscoWorkImpl modisco2 = null;
		for(MoDiscoWorkImpl component : getAllMoDiscoWork()){
			if(component.getIndex()==tableIndex){
				modisco1 = component;
				indexModisco1 = modiscoWorkflowIndex;
			}else if(component.getIndex()==tableIndex+1){
				modisco2 = component;
				indexModisco2 = modiscoWorkflowIndex;
			}else if(modisco1!=null&&modisco2!=null){
				break;
			}
			modiscoWorkflowIndex++;
		}
		modisco1.setIndex(modisco1.getIndex()+1);
		modisco2.setIndex(modisco2.getIndex()-1);
		setComponent(indexModisco1, modisco1);
		setComponent(indexModisco2, modisco2);
		refresh(tableIndex+1);
	}
	
	/**
	 * Open the wizard to load a MoDiscoWork in the MoDiscoWorkflow
	 */
	public void loadAdditionalTransformation() {
		this.wizardDrivers.run(null,Constant.loadMode);
		refresh(-1);
	}
	
	/**
	 * Allows to add a MoDiscoWork in the MoDisco workflow
	 * @param type Launcher type
	 * @param index	Index in the MoDiscoWorkflow
	 * @param name Transformation name
	 * @param configuration The configuration
	 * @param refresh true if it is necessary to refresh otherwise false
	 */
	public void addMoDiscoWork(String type, final int index, String name, boolean refresh){
		int indexLocal = index;
		if(indexLocal==-1){
			indexLocal = this.workflowInterfaceMap.get(this.currentIdWorkflow).getAllComponents().size();
		}
		//ModiscoWork modiscoWork = new ModiscoWork(name,index,type, configuration);
		MoDiscoWork modisco = ModiscoworkFactory.eINSTANCE.createMoDiscoWork();
		modisco.setIndex(indexLocal);
		//modisco.setLaunchConfiguration(configuration.getName());
		modisco.setName(name);
		modisco.setType(type);
		this.workflowInterfaceMap.get(this.currentIdWorkflow).addComponentWorkflow(name,modisco,this.controllerDriver.getDriverFromLaunchType(type).getParameters(this.getLaunchConfigurationFromName(name)));
		if(refresh){
			refresh(-1);
		}
	}
	
	/**
	 * Load the plug-in to refer the point extension Constant.nameExtensionPointWorkflow
	 */
	private void initializeWorkflow(){
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint point = registry.getExtensionPoint(Constant.nameExtensionPointWorkflow); //$NON-NLS-1$
		
		for (IExtension extension : point.getExtensions()) {
			for (IConfigurationElement element : extension.getConfigurationElements()) {
				//We take into account the first extension
				if(element.getName().equals("workflow")){
					try {
						this.workflowInterfaceMap.put(extension.getUniqueIdentifier(),(WorkflowModelInterface)element.createExecutableExtension("class"));
					} catch (Throwable e) {
						e.printStackTrace();
					}
				}
				if(element.getName().equals("executionWorkflow")){
					try {
						this.workflowExecutionInterfaceMap.put(extension.getUniqueIdentifier(),(WorkflowExecutionInterface)element.createExecutableExtension("class"));
					} catch (Throwable e) {
						e.printStackTrace();
					}
				}
			}
		}
	}
	
	/**
	 * Refresh the view (Table of MoDiscoWork)
	 * @param index index for the next selection
	 */
	public void refresh(int index){
		this.additionalButton.getTable().removeAll();
		List<MoDiscoWorkImpl> modiscoWorkList = getAllMoDiscoWork();
		Collections.sort(modiscoWorkList);
		for(MoDiscoWorkImpl component : modiscoWorkList){
			TableItem item = new TableItem(this.additionalButton.getTable(),SWT.NULL);
			item.setText(component.getName());
			item.setImage(Common.retrieveImageDriver(component.getType()));
		}
		if(index!=-1){
			this.additionalButton.getTable().select(index);
		}else{
			this.additionalButton.getTable().select(this.additionalButton.getTable().getItemCount()-1);
		}
		this.additionalButton.getTable().notifyListeners(SWT.Selection, null);
	}
	
	/**
	 * Allows to retrieve an instance of {@link MoDiscoWork} corresponding at the item selected in the table
	 * @param tableIndex Index of item selected
	 * @return an instance of {@link MoDiscoWork} corresponding at the item selected in the table
	 */
	public MoDiscoWorkImpl getComponent(int tableIndex){
		List<MoDiscoWorkImpl> modiscoWorkList = getAllMoDiscoWork();
		Collections.sort(modiscoWorkList);
		MoDiscoWorkImpl modisco = modiscoWorkList.get(tableIndex);
		int modiscoWorkflowIndex=0;
		for(MoDiscoWorkImpl component : getAllMoDiscoWork()){
			if(component.getIndex()==modisco.getIndex()){
				break;
			}
			modiscoWorkflowIndex++;
		}
		return (MoDiscoWorkImpl)this.workflowInterfaceMap.get(this.currentIdWorkflow).getComponent(modiscoWorkflowIndex);
	}
	
	/**
	 * Change a component of MoDiscoWorkflow
	 * @param index Component index
	 * @param modiscoWork an instance of {@link MoDiscoWork}
	 */
	private void setComponent(int index, MoDiscoWorkImpl modiscoWork){
		this.workflowInterfaceMap.get(this.currentIdWorkflow).setComponent(index,modiscoWork);
	}
	
	/**
	 * Allows to retrieve all components in the MoDiscoWorkflow.
	 * @return a list of MoDiscoWork
	 */
	@SuppressWarnings("unchecked")
	public List<MoDiscoWorkImpl> getAllMoDiscoWork(){
		List<? extends EObject> listComponent = this.workflowInterfaceMap.get(this.currentIdWorkflow).getAllComponents();
		return  (List<MoDiscoWorkImpl>)listComponent;
	}

	/**
	 * Allows to save the modifications to do in the edition of the MoDiscoWork
	 * @param transformationName The new transformation name
	 * @param configuration The new configuration
	 */
	public void performFinishEditTransformation(String transformationName) {
		MoDiscoWorkImpl modiscoWork = getComponent(this.editionIndice);
		modiscoWork.setName(transformationName);
		//modiscoWork.setLaunchConfiguration(configuration.getName());
		int modiscoWorkflowIndex=0;
		for(MoDiscoWorkImpl component : getAllMoDiscoWork()){
			if(component.getIndex()==modiscoWork.getIndex()){
				break;
			}
			modiscoWorkflowIndex++;
		}
		setComponent(modiscoWorkflowIndex,modiscoWork);
	}
	
	/**
	 * Verify that all the MoDiscoWork have a configuration
	 * @return the MoDiscoWork name which didn't have a configuration
	 */
	public String allComponentsHaveConfiguration() {
		String transformationName = null;
		for(TableItem item :this.additionalButton.getTable().getItems()){
			MoDiscoWorkImpl modiscoWork = getComponent(this.additionalButton.getTable().indexOf(item));
			if(modiscoWork!=null){
				ILaunchConfiguration config = getLaunchConfigurationFromName(modiscoWork.getName());
				if(!config.exists()){
					transformationName = modiscoWork.getName();
					break;
				}
			}
		}
		return transformationName;
	}

	/**
	 * Allows to retrieve the MoDiscoWork name which is not valid
	 * @return the MoDiscoWork name which is not valid
	 */
	public String getTransformationNameNotValid() {
		return this.transformationNameNotValid;
	}
	
	/**
	 * Allows to save the resource
	 */
	public void saveWorkflowResource(ILaunchConfigurationWorkingCopy configuration) {
		this.workflowInterfaceMap.get(this.currentIdWorkflow).save(configuration);		
	}
	
	/**
	 * Allows to move the resource
	 * @param path Workflow resource path
	 */
	public void moveWorkflowResource(String path) {
		this.workflowInterfaceMap.get(this.currentIdWorkflow).moveResource(path);
	}
	
	/**
	 * Allows to initialize or load the workflow
	 * @param path workflow path
	 */
	public void initializationWorkflowResource(ILaunchConfiguration configuration, String path) {
		WorkflowModelInterface model = this.workflowInterfaceMap.get(this.currentIdWorkflow);
		model.initialization(configuration,path);		
	}
	
	/**
	 * Allows to remove resource
	 */
	public void removeResource(String resourcePath) {
		this.workflowInterfaceMap.get(this.currentIdWorkflow).removeResource(resourcePath);
		
	}
	
	/**
	 * Allows to retrieve all parameters of selected driver in the table
	 * @param tableIndex Index selected of the table
	 * @return All parameters of driver
	 */
	public List<MoDiscoWorkParameter> getAllParameterOfDriver(int tableIndex){
		MoDiscoWorkImpl modisco = getComponent(tableIndex);
		ILaunchConfiguration config = getLaunchConfigurationFromName(modisco.getName());
		ILaunchConfigurationWorkingCopy copy = null;
		try {
			copy = config.getWorkingCopy();
		} catch (CoreException e) {
			System.err.println("getAllParameterOfDriver: Not copy found for LaunchConfiguration");
			e.printStackTrace();
		}
		return this.controllerDriver.getDriverFromLaunchType(modisco.getType()).getParameters(copy);
	}
	
	/**
	 * Allows to find the good instance of LaunchConfiguration corresponding at path
	 * @param configName Name of LaunchConfiguration
	 * @return An instance of launchConfiguration corresponding at path
	 */
	private ILaunchConfiguration getLaunchConfigurationFromName(String configName){
		ILaunchConfiguration configuration = null;
		try {
			for(ILaunchConfiguration config : DebugPlugin.getDefault().getLaunchManager().getLaunchConfigurations()){
				if(config.getName().equals(configName)){
					configuration = config;
					break;
				}
			}
		} catch (CoreException e) {
			e.printStackTrace();
		}
		return configuration;
	}
	
	/**
	 * Allows to change workflow type (Serialize to minimalist or minimalist to serialize)
	 * @param newId
	 * @param configuration
	 */
	public void changeWorkflowType(String newId, ILaunchConfigurationWorkingCopy configuration) {
		if(!newId.equals(this.currentIdWorkflow)){
			List<MoDiscoWorkImpl> listMoDisco = getAllMoDiscoWork();
			Collections.sort(listMoDisco);
			String resourcePath = null;
			if (!configuration.isLocal()) {				
				IPath path = configuration.getFile().getLocation().removeLastSegments(1).append(configuration.getName()).addFileExtension(Constant.extension);
				resourcePath = path.toString();
			} else {
				resourcePath = ResourcesPlugin.getWorkspace().getRoot().getLocation().append(".metadata").append(".plugins").append(Activator.PLUGIN_ID).append(configuration.getName()).addFileExtension(Constant.extension).toString();
			}
			//Serialize to minimalist.
			if(newId.equals(Constant.idMinimaliste)){
				//Delete the resource
				removeResource(resourcePath);
				configuration.removeAttribute(Constant.resourcePath);
				configuration.removeAttribute(Constant.moDiscoWorkNumber);
				configuration.removeAttribute(Constant.moDiscoListSerialize);
			}else{
				this.currentIdWorkflow = newId;
				configuration.removeAttribute(Constant.moDiscoListMinimalist);
			}
			this.currentIdWorkflow = newId;
			initializationWorkflowResource(configuration.getOriginal(), resourcePath);
			for(MoDiscoWorkImpl modisco : listMoDisco){
				addMoDiscoWork(modisco.getType(), modisco.getIndex(), modisco.getName(), false);
			}
			saveWorkflowResource(configuration);
		}
	}
	
	/**
	 * Allows to create a discoverer project from a MoDiscoWorkflow
	 */
	public void exportMoDiscoWorkflowToDiscovererProject(String projectName) {
		try {
			CreateJavaProject javaProjectCreator = new CreateJavaProject(projectName);
		} catch (CoreException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
//		try {
//			IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
//			IProject project = root.getProject(projectName);
//			if(project.exists()){
//				project.delete(true, true, new NullProgressMonitor());
//			}
//			project.create(null);
//			project.open(null);
//			IProjectDescription description = project.getDescription();
//			description.setNatureIds(new String[] { JavaCore.NATURE_ID });
//			project.setDescription(description, null);
//			IJavaProject javaProject = JavaCore.create(project); 
//			IFolder binFolder = project.getFolder("bin");
//			binFolder.create(false, true, null);
//			List<IClasspathEntry> entries = new ArrayList<IClasspathEntry>();
//			IVMInstall vmInstall = JavaRuntime.getDefaultVMInstall();
//			LibraryLocation[] locations = JavaRuntime.getLibraryLocations(vmInstall);
//			for (LibraryLocation element : locations) {
//			 entries.add(JavaCore.newLibraryEntry(element.getSystemLibraryPath(), null, null));
//			}
//			//add libs to project class path
//			javaProject.setRawClasspath(entries.toArray(new IClasspathEntry[entries.size()]), null);
//			IFolder sourceFolder = project.getFolder("src");
//			sourceFolder.create(false, true, null);
//			IPackageFragmentRoot rootPackage = javaProject.getPackageFragmentRoot(sourceFolder);
//			IClasspathEntry[] oldEntries = javaProject.getRawClasspath();
//			IClasspathEntry[] newEntries = new IClasspathEntry[oldEntries.length + 1];
//			System.arraycopy(oldEntries, 0, newEntries, 0, oldEntries.length);
//			newEntries[oldEntries.length] = JavaCore.newSourceEntry(rootPackage.getPath());
//			javaProject.setRawClasspath(newEntries, null);
//			
//			IPackageFragment packageProject = rootPackage.createPackageFragment(projectName.trim(), false, null);
//		} catch (CoreException e) {
//			e.printStackTrace();
//		}	
	}
	
	/*
	 * Setter
	 */
	public void setAdditionnalButton(AdditionalButtonsComposite additionalButton){
		this.additionalButton = additionalButton;
	}

	public LaunchGroupExtension getGroup() {
		return this.group;
	}

	public void setGroup(LaunchGroupExtension group) {
		this.group = group;
	}

	public WorkflowExecutionInterface getWorkflowExecutionInterface(String id) {
		return this.workflowExecutionInterfaceMap.get(id);
	}

	public String getIdWorkflowExecution() {
		return this.currentIdWorkflow;
	}
}
