/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 *******************************************************************************/

package org.eclipse.gmt.modisco.discoverersmanager;

import java.util.Map;
import java.util.Set;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;


/**
 * A discoverer is a component that injects an element in a model
 * which represents this element.
 * 
 */
public interface Discoverer {

	/**
	 * To determine if source object could be handled
	 * by the discoverer.
	 * Each discoverer has to implements this method
	 * with its own criteria to filter selected object,
	 * If the source object is managed by this discoverer,
	 * the discoverer menu will be available in popup menu
	 * when users click with contextual button (right-click)
	 * 
	 * @param source the selected object.
	 * @return true if selected object is managed by this discoverer, false other else.
	 */
	public boolean applyTo(final Object source);
	
	/**
	 * Execute discovery task with selected object.
	 * This is a method adapted to common use of discoverers.
	 * 
	 * For a more generic method, see discoverElement(Object source, Map<String, Object> parameters)
	 * @see org.eclipse.gmt.modisco.discoverersmanager.Discoverer#discoverElement(java.lang.Object, java.util.Map)
	 * 
	 * @param source the selected object.
	 * @param target the target uri if provided, null other else.
	 * @return the EMF resource of target model discovered from source.
	 */
	public Resource discoverElement(final Object source, final URI target);
	
	/**
	 * Generic method to discover a source element, it allows more different use cases.
	 * Because there is no constraints to expected result, we have a way to drive
	 * elements discovery with a wide configuration panel.
	 * This method has to be used in coordination with "getParameterKeys" method
	 * to know which parameters are available.
	 *
	 * @param source the selected object.
	 * @param parameters list of parameters pair (key, value)
	 */
	public void discoverElement(final Object source, final Map<String, Object> parameters);
	
	/**
	 * To have a complete list of all parameters available in this discoverer.
	 * Further informations should be provided to indicate features of each parameter,
	 * for example, direction kind (in, out, in-out), if parameter is optional or not ...
	 *
	 * @return set of parameters names provided by this discoverer.
	 */
	public Set<String> getParameterKeys();
	
}
