/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 */

package org.eclipse.gmt.modisco.common.core.examples;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.m2m.atl.common.ATLLaunchConstants;
import org.eclipse.m2m.atl.core.ATLCoreException;
import org.eclipse.m2m.atl.core.IExtractor;
import org.eclipse.m2m.atl.core.IInjector;
import org.eclipse.m2m.atl.core.IModel;
import org.eclipse.m2m.atl.core.IReferenceModel;
import org.eclipse.m2m.atl.core.ModelFactory;
import org.eclipse.m2m.atl.core.launch.ILauncher;
import org.eclipse.m2m.atl.core.service.CoreService;
import org.eclipse.m2m.atl.core.service.LauncherService;
import org.eclipse.m2m.atl.core.ui.vm.asm.ASMModelWrapper;
import org.eclipse.m2m.atl.drivers.emf4atl.ASMEMFModel;
import org.eclipse.m2m.atl.engine.vm.nativelib.ASMModel;

/**
 * @author Gabriel Barbier
 *
 */
public class AtlTransformation {

	//resources location
	private final String kdmMMUri ="http://www.eclipse.org/MoDisco/kdm/action";
	private final String j2se5MMUri ="http://www.eclipse.org/MoDisco/J2SE5";

	public static final String TEST_IN_MODEL
	= "platform:/resource/test/test.in";


	public Resource testDetaillePourRegularVm(final URI j2se5SourceModelUri, final URL transformation, final URI kdmTargetModelUri) throws ATLCoreException {
		/*
		 * Notes globales :
		 * Attention les uri utilises pour indiquer que le modle
		 * est enregistr dans le EMFRegistry ne doivent plus
		 * comporter le prfixe "uri:"
		 */

		/*
		 * Attention cette collection est trs importante,
		 * elle associe le nom d'un mtamodle au nom du model handler  utiliser
		 * 
		 * par exemple, j2se5 avec EMF
		 * String j2se5MetaModelName = "j2se5";
		 * modelHandlers.put(j2se5MetaModelName, "EMF");
		 */
		Map<String, String> modelHandlers = new HashMap<String, String>();
		String j2se5MetaModelName = "j2se5";
		modelHandlers.put(j2se5MetaModelName, "EMF");
		String kdmMetaModelName = "kdm";
		modelHandlers.put(kdmMetaModelName, "EMF");

		/*
		 * initialisation du launcher,
		 * maintenant toutes les instances sont fournies par la classe CoreService
		 * et les diffrences se font  travers des chaines de caractres.
		 * Heureusement, la classe ATLLaunchConstants regroupe la plupart
		 * de ces constantes.
		 */
		Map<String, Object> launcherOptions = new HashMap<String, Object>();
		launcherOptions.put(ATLLaunchConstants.OPTION_MODEL_HANDLER, modelHandlers);

		String launcherName = ATLLaunchConstants.REGULAR_VM_NAME;
		final ILauncher launcher = CoreService.getLauncher(launcherName);
		launcher.initialize(launcherOptions);
		//launcher.initialize(Collections.<String, Object> emptyMap());

		/*
		 * Initialisation des anciens model handlers,
		 * ils sont maintenant appel injecteurs/extracteurs
		 */
		ModelFactory factory = CoreService.createModelFactory(launcher.getDefaultModelFactoryName());

		IInjector injector = CoreService.getInjector(factory.getDefaultInjectorName());
		IExtractor extractor = CoreService.getExtractor(factory.getDefaultExtractorName());

		/*
		 * Toujours dans le cadre de l'ancienne vm,
		 * il faut complter les options de chaque modle
		 * avec le nom du modle, son chemin, etc.
		 * il y a un peu redondance d'informations mais c'est requis !
		 * Il faut donc fournir ces informations pour tous les modles.
		 */
		// chargement du mta modle Java
		Map<String, Object> referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(j2se5MetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", j2se5MetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", this.j2se5MMUri); //$NON-NLS-1$
		IReferenceModel j2se5MM = factory.newReferenceModel(referenceModelOptions);
		injector.inject(j2se5MM, this.j2se5MMUri);

		// chargement du modle Java
		Map<String, Object> modelOptions = new HashMap<String, Object>();
		String inModelName = "IN";
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", j2se5SourceModelUri.toString()); //$NON-NLS-1$
		modelOptions.put("newModel", false); //$NON-NLS-1$
		IModel input = factory.newModel(j2se5MM, modelOptions);
		injector.inject(input, j2se5SourceModelUri.toString());
		launcher.addInModel(input, inModelName, j2se5MetaModelName);

		// Chargement du mta modle KDM
		referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(kdmMetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", kdmMetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", this.kdmMMUri); //$NON-NLS-1$
		IReferenceModel kdmMM = factory.newReferenceModel(referenceModelOptions);
		injector.inject(kdmMM, this.kdmMMUri);
		// Chargement du modle KDM
		modelOptions = new HashMap<String, Object>();
		inModelName = "OUT";
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		if (kdmTargetModelUri != null) {
			modelOptions.put("path", kdmTargetModelUri.toString()); //$NON-NLS-1$
		} else {
			modelOptions.put("path", j2se5SourceModelUri.trimFileExtension().appendFileExtension("kdm").toString()); //$NON-NLS-1$
		}
		modelOptions.put("newModel", true); //$NON-NLS-1$
		IModel outputInstance = factory.newModel(kdmMM, modelOptions);
		launcher.addOutModel(outputInstance, inModelName, kdmMetaModelName);

		/*
		 * encapsulation de la transformation dans un nouveau Thread pour
		 * obtenir une pile d'excution vide.
		 */
		final Map<String, Object> options = new HashMap<String, Object>();
		options.put("continueAfterError", "true");
		options.put("printExecutionTime", "true");

		Job transformationThread = new Job("Convert J2SE5 model to KDM") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				IStatus result = Status.OK_STATUS;
				try {
					launcher.launch(ILauncher.RUN_MODE, monitor, options,
							transformation.openStream());
				} catch (IOException e) {
					result = Status.CANCEL_STATUS;
					e.printStackTrace();
				}

				return result;
			}
		};
		transformationThread.schedule();
		try {
			transformationThread.join();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		
		if (kdmTargetModelUri != null) {
			extractor.extract(outputInstance, kdmTargetModelUri.toString());
		}
		

		Resource output = null;
		if (outputInstance instanceof ASMModelWrapper) {
			ASMModelWrapper wrapper = (ASMModelWrapper) outputInstance;
			ASMModel model = wrapper.getAsmModel();
			if (model instanceof ASMEMFModel) {
				output = ((ASMEMFModel) model).getExtent();
			}
		}
		return output;

	}

	private Resource resourceTemp;
	
	public Resource testSimplifieUtilisantLauncherServicePourRegularVm(final URI j2se5SourceModelUri, final URL transformation, final URI kdmTargetModelUri) throws ATLCoreException {
		/*
		 * Cette version semble plus simple,
		 * mais elle ne permet pas de rcuprer facilement le rsultat
		 * de la transformation.
		 * Il va falloir passer par une collection, ou l'initialisation
		 * d'une variable d'instance ...
		 * 
		 * Attention, avec cette version, la resource cible est forcment
		 * sauvegarde par ATL.
		 * 
		 */
		/**
		 * Launches a transformation using the given parameters. Process the injections, extractions.
		 * 
		 * @param mode
		 *            the launching mode
		 * @param monitor
		 *            the progress monitor
		 * @param launcher
		 *            the {@link ILauncher}
		 * @param inModels
		 *            a map defining the source models names: [model name, reference model name]
		 * @param inoutModels
		 *            a map defining the inout models names: [model name, reference model name]
		 * @param outModels
		 *            a map defining the target models names: [model name, reference model name]
		 * @param paths
		 *            a map defining the location of each model name: [model name, model location]
		 * @param options
		 *            a map of transformation options: [key, value]
		 * @param libraries
		 *            a map of libraries modules: [library name, module]
		 * @param modules
		 *            a list of modules to superimpose. The first module overrides the next ones.
		 * @return the transformation result
		 */

		/*
		 * Cette collection est trs importante,
		 * elle associe le nom d'un mtamodle au nom du model handler  utiliser
		 * 
		 * par exemple, j2se5 avec EMF
		 * String j2se5MetaModelName = "j2se5";
		 * modelHandlers.put(j2se5MetaModelName, "EMF");
		 */
		Map<String, String> modelHandlers = new HashMap<String, String>();
		String j2se5MetaModelName = "j2se5";
		modelHandlers.put(j2se5MetaModelName, "EMF");
		String kdmMetaModelName = "kdm";
		modelHandlers.put(kdmMetaModelName, "EMF");

		/*
		 * initialisation du launcher,
		 * maintenant toutes les instances sont fournies par la classe CoreService
		 * et les diffrences se font  travers des chaines de caractres.
		 * Heureusement, la classe ATLLaunchConstants regroupe la plupart
		 * de ces constantes.
		 */
		final Map<String, Object> options = new HashMap<String, Object>();
		options.put(ATLLaunchConstants.OPTION_MODEL_HANDLER, modelHandlers);

		String launcherName = ATLLaunchConstants.REGULAR_VM_NAME;
		final ILauncher launcher = CoreService.getLauncher(launcherName);
		launcher.initialize(options);
		//launcher.initialize(Collections.<String, Object> emptyMap());

		

		/*
		 * Gestion des modles : 
		 * uniquement les modles dclars dans la transo
		 */
		final Map<String, String> sourceModels = new HashMap<String, String>();
		sourceModels.put("IN", "j2se5");
		final Map<String, String> targetModels = new HashMap<String, String>();
		targetModels.put("OUT", "kdm");
		/*
		 * Gestion des chemins des modles ...
		 * tous les modles relatifs  la transfo !
		 */
		final Map<String, String> modelPaths = new HashMap<String, String>();
		modelPaths.put("j2se5", this.j2se5MMUri);
		modelPaths.put("IN", j2se5SourceModelUri.toString());
		modelPaths.put("kdm", this.kdmMMUri);
		if (kdmTargetModelUri != null) {
			modelPaths.put("OUT", kdmTargetModelUri.toString());
		} else {
			modelPaths.put("OUT", j2se5SourceModelUri.trimFileExtension().appendFileExtension("kdm").toString());
		}
		/*
		 * encapsulation de la transformation dans un nouveau Thread pour
		 * obtenir une pile d'excution vide.
		 */
		options.put("continueAfterError", "true");
		options.put("printExecutionTime", "true");

		Job transformationThread = new Job("Convert J2SE5 model to KDM") {
			@SuppressWarnings({ "unchecked", "unqualified-field-access" })
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				IStatus result = Status.OK_STATUS;
				try {
					final InputStream[] modules = {transformation.openStream()};
					final Map<String, InputStream> libraries = Collections.emptyMap();
					Object resultModel = LauncherService.launch(ILauncher.RUN_MODE, monitor, launcher, sourceModels, Collections.EMPTY_MAP,
							targetModels, modelPaths, options, libraries, modules);
					if (resultModel instanceof ASMModelWrapper) {
						ASMModelWrapper wrapper = (ASMModelWrapper) resultModel;
						ASMModel model = wrapper.getAsmModel();
						if (model instanceof ASMEMFModel) {
							resourceTemp = ((ASMEMFModel) model).getExtent();
						}
					}
				} catch (IOException e) {
					result = Status.CANCEL_STATUS;
					e.printStackTrace();
				} catch (ATLCoreException e) {
					result = Status.CANCEL_STATUS;
					e.printStackTrace();
				}

				return result;
			}
		};
		transformationThread.schedule();
		try {
			transformationThread.join();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		

		Resource output = this.resourceTemp;
		
		return output;

	}
	
	public void testDetaillePourEmfVm(URL asmURL, IFile file) throws ATLCoreException, IOException {
		// Getting launcher
		ILauncher launcher = null;
		launcher = CoreService.getLauncher(ATLLaunchConstants.EMF_VM_NAME); //$NON-NLS-1$
		launcher.initialize(Collections.<String, Object> emptyMap());
		// Defaults
		ModelFactory factory = CoreService.createModelFactory(launcher.getDefaultModelFactoryName()); //$NON-NLS-1$
		IInjector injector = CoreService.getInjector(factory.getDefaultInjectorName());
		IExtractor extractor = CoreService.getExtractor(factory.getDefaultExtractorName());

		// Metamodels
		IReferenceModel umlMetamodel = factory.newReferenceModel();
		injector.inject(umlMetamodel, "http://www.eclipse.org/uml2/2.1.0/UML"); //$NON-NLS-1$
		IReferenceModel refiningTraceMetamodel = factory.getBuiltInResource("RefiningTrace.ecore"); //$NON-NLS-1$

		// Creating models
		IModel refiningTraceModel = factory.newModel(refiningTraceMetamodel);
		IModel umlModel = factory.newModel(umlMetamodel);

		// Loading existing model
		injector.inject(umlModel, file.getFullPath().toString());

		// Launching
		launcher.addOutModel(refiningTraceModel, "refiningTrace", "RefiningTrace"); //$NON-NLS-1$ //$NON-NLS-2$
		launcher.addInOutModel(umlModel, "IN", "UML"); //$NON-NLS-1$ //$NON-NLS-2$

		launcher.launch(ILauncher.RUN_MODE, new NullProgressMonitor(), Collections
				.<String, Object> emptyMap(), asmURL.openStream());

		// Saving model
		extractor.extract(umlModel, file.getFullPath().toString());

	}
}
