/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.editors;

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.edit.EMFEditPlugin;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.emf.edit.ui.provider.ExtendedImageRegistry;
import org.eclipse.gmt.modisco.common.editor.core.InstancesForMetaclass;
import org.eclipse.gmt.modisco.common.editor.core.InstancesForMetaclasses;
import org.eclipse.gmt.modisco.common.editor.editors.EditorConfiguration.MetaclassesSortMode;
import org.eclipse.gmt.modisco.common.editor.util.EMFUtil;
import org.eclipse.gmt.modisco.common.editor.util.ImageProvider;
import org.eclipse.jface.viewers.IColorProvider;
import org.eclipse.jface.viewers.IFontProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;

/**
 * The viewer for displaying the list of metaclasses that appears in the model editor. The input of
 * this viewer is an instance of the {@link InstancesForMetaclasses} class and the elements are
 * instances of {@link InstancesForMetaclass}.
 */
public class MetaclassViewer {

	private TreeViewer treeViewer;

	/** The resource set containing all the resources of the model */
	private final ResourceSet resourceSet;
	/** The configuration of the model editor */
	private final EditorConfiguration editorConfiguration;

	/**
	 * @param parent
	 *            the composite in which this viewer must be created
	 * @param editorConfiguration
	 *            the configuration of the model editor, which is passed on to the
	 *            {@link MetaclassViewer}s.
	 */
	public MetaclassViewer(Composite parent, EditorConfiguration editorConfiguration) {
		this.resourceSet = editorConfiguration.getResourceSet();
		this.editorConfiguration = editorConfiguration;

		this.treeViewer = new TreeViewer(parent, SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER
				| SWT.MULTI);

		initViewer();

		InstancesForMetaclasses instancesForMetaclasses = this.editorConfiguration
				.getInstancesForMetaclasses();
		this.treeViewer.setInput(instancesForMetaclasses);

		createContextMenu(this.treeViewer.getTree());
	}

	/** Create a context menu on the tree of metaclasses */
	private void createContextMenu(Control control) {
		MetaclassViewerMenuManager menuManager = new MetaclassViewerMenuManager(this,
				this.editorConfiguration);
		Menu menu = menuManager.createContextMenu(control);
		control.setMenu(menu);
	}

	/** Select the class corresponding to the root element (the 'model' element) */
	public void selectRootElement() {
		this.treeViewer.setSelection(null);
		EList<Resource> resources = this.resourceSet.getResources();
		if (resources.size() > 0) {
			EList<EObject> contents = resources.get(0).getContents();
			if (contents.size() > 0) {
				EObject object = contents.get(0);
				String classFullQualifiedName = EMFUtil.getMetaclassQualifiedName(object.eClass());
				selectMetaclass(classFullQualifiedName);
			}
		}
	}

	/** Select the metaclass with the given full qualified name */
	public void selectMetaclass(String classFullQualifiedName) {
		InstancesForMetaclasses instancesForMetaclasses = (InstancesForMetaclasses) this.treeViewer
				.getInput();
		InstancesForMetaclass instancesForMetaclass = instancesForMetaclasses
				.getInstancesForMetaclass(classFullQualifiedName);
		if (instancesForMetaclass != null) {
			this.treeViewer.setSelection(new StructuredSelection(instancesForMetaclass), true);
		}
	}

	/** Return a list of qualified names of selected metaclasses, or an empty list if none */
	public String[] getSelectedMetaclassesQualifiedNames() {
		ArrayList<String> selectedMetaclasses = new ArrayList<String>();
		ISelection selection = this.treeViewer.getSelection();
		if (selection instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			Iterator<?> iterator = structuredSelection.iterator();
			while (iterator.hasNext()) {
				Object selectedElement = iterator.next();
				if (selectedElement instanceof InstancesForMetaclass) {
					InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) selectedElement;
					selectedMetaclasses.add(instancesForMetaclass.getClassQualifiedName());
				}
			}
		}
		return selectedMetaclasses.toArray(new String[selectedMetaclasses.size()]);
	}

	/** Return a list of selected metaclasses, or an empty list if none */
	public EClass[] getSelectedMetaclasses() {
		ArrayList<EObject> selectedMetaclasses = new ArrayList<EObject>();
		ISelection selection = this.treeViewer.getSelection();
		if (selection instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			Iterator<?> iterator = structuredSelection.iterator();
			while (iterator.hasNext()) {
				Object selectedElement = iterator.next();
				if (selectedElement instanceof InstancesForMetaclass) {
					InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) selectedElement;
					selectedMetaclasses.add(instancesForMetaclass.getEClass());
				}
			}
		}
		return selectedMetaclasses.toArray(new EClass[selectedMetaclasses.size()]);
	}

	/** Return the qualified name of the first selected metaclass, or null if none */
	public String getFirstSelectedMetaclassQualifiedName() {
		String[] selectedMetaclassesQualifiedNames = getSelectedMetaclassesQualifiedNames();

		String firstSelectedMetaclass = null;
		if (selectedMetaclassesQualifiedNames.length > 0) {
			firstSelectedMetaclass = selectedMetaclassesQualifiedNames[0];
		}
		return firstSelectedMetaclass;
	}

	protected class EmptyMetaclassesFilter extends ViewerFilter {
		@Override
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			if (!MetaclassViewer.this.editorConfiguration.isShowEmptyMetaclasses()) {
				if (element instanceof InstancesForMetaclass) {
					InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) element;
					int count = getInstanceCountFor(instancesForMetaclass);
					if (count == 0) {
						if (MetaclassViewer.this.editorConfiguration.isShowDerivationTree()) {
							// don't hide a node on a branch leading to a non-empty class
							return isOneSubclassNonEmpty(instancesForMetaclass);
						}

						return false;
					}
				}
				// don't show empty packages
				else if (element instanceof PackageGroup) {
					return !isEmpty((PackageGroup) element);
				}
			}
			return true;
		}

		/** @return whether the given package is empty */
		private boolean isEmpty(PackageGroup packageGroup) {
			for (Object element : packageGroup) {
				if (element instanceof PackageGroup) {
					PackageGroup p = (PackageGroup) element;
					if (!isEmpty(p)) {
						return false;
					}
				}

				else if (element instanceof InstancesForMetaclass) {
					InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) element;
					if (getInstanceCountFor(instancesForMetaclass) != 0) {
						return false;
					}
				}
			}

			return true;
		}

		private boolean isOneSubclassNonEmpty(InstancesForMetaclass instancesForMetaclass) {
			InstancesForMetaclass[] subclasses = instancesForMetaclass.getSubclasses();
			for (InstancesForMetaclass subclass : subclasses) {
				if (getInstanceCountFor(subclass) > 0 || isOneSubclassNonEmpty(subclass))
					return true;
			}
			return false;
		}

		private int getInstanceCountFor(InstancesForMetaclass instancesForMetaclass) {
			int count = 0;
			if (MetaclassViewer.this.editorConfiguration.isDisplayInstancesOfSubclasses()) {
				count = instancesForMetaclass.totalSize();
			} else {
				count = instancesForMetaclass.size();
			}
			return count;
		}
	}

	protected class MetaclassLabelProvider extends LabelProvider implements ILabelProvider,
			IColorProvider, IFontProvider {

		private final Color colorGrayedOut;

		public MetaclassLabelProvider() {
			this.colorGrayedOut = new Color(Display.getDefault(), 128, 128, 128);
		}

		@Override
		public String getText(Object element) {
			if (element instanceof InstancesForMetaclass) {
				InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) element;

				int count = 0;
				if (MetaclassViewer.this.editorConfiguration.isDisplayInstancesOfSubclasses()) {
					count = instancesForMetaclass.totalSize();
				} else {
					count = instancesForMetaclass.size();
				}
				String name;

				if (MetaclassViewer.this.editorConfiguration.isShowMetaclassesFullQualifiedNames()) {
					name = instancesForMetaclass.getClassQualifiedName();
				} else {
					// only show the short name of the class (instead of the full qualified name)
					name = instancesForMetaclass.getEClass().getName();
				}

				return name + " (" + count + ")";
			}

			if (element instanceof PackageGroup) {
				PackageGroup packageGroup = (PackageGroup) element;
				return packageGroup.getName();
			}

			return element.toString();
		}

		@Override
		public Image getImage(Object element) {
			// return the same image as for instances of this metaclass

			if (element instanceof InstancesForMetaclass) {
				InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) element;
				ArrayList<EObject> instances = instancesForMetaclass.getElements();
				if (instances.size() > 0) {
					// take the image of the first instance found
					EObject anInstance = instances.get(0);

					IItemLabelProvider itemLabelProvider = (IItemLabelProvider) MetaclassViewer.this.editorConfiguration
							.getAdapterFactoryWithRegistry().adapt(anInstance,
									IItemLabelProvider.class);

					if (itemLabelProvider != null) {
						return ExtendedImageRegistry.getInstance().getImage(
								itemLabelProvider.getImage(anInstance));
					}
				}

				String className = instancesForMetaclass.getEClass().getName();

				URI imageURI = URI.createURI(EMFEditPlugin.INSTANCE.getImage("full/obj16/Item")
						.toString()
						+ "#" + className);
				return ExtendedImageRegistry.getInstance().getImage(imageURI);
			}

			if (element instanceof PackageGroup) {
				return ImageProvider.getInstance().getPackageIcon();
			}
			return null;
		}

		public Color getBackground(Object element) {
			return null;
		}

		public Color getForeground(Object element) {
			if (element instanceof InstancesForMetaclass) {
				InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) element;
				int count = 0;
				if (MetaclassViewer.this.editorConfiguration.isDisplayInstancesOfSubclasses()) {
					count = instancesForMetaclass.totalSize();
				} else {
					count = instancesForMetaclass.size();
				}
				if (count == 0) {
					return this.colorGrayedOut;
				}
			}

			// default color
			return null;
		}

		public Font getFont(Object element) {
			if (element instanceof InstancesForMetaclass) {
				InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) element;
				if (instancesForMetaclass.getEClass().isAbstract()) {
					return MetaclassViewer.this.editorConfiguration.getCustomItalicFont();
				}
			}

			// default font (possibly customized)
			return MetaclassViewer.this.editorConfiguration.getCustomFont();
		}

		@Override
		public void dispose() {
			this.colorGrayedOut.dispose();
			super.dispose();
		}

	}

	/**
	 * A list corresponding to a group of metaclasses belonging to the same package, or
	 * sub-packages.
	 */
	@SuppressWarnings("serial")
	protected class PackageGroup extends ArrayList<Object> {
		/** The name of the package */
		private final String name;
		private final Object parent;

		public PackageGroup(Object parent, String name) {
			this.parent = parent;
			this.name = name;
		}

		public Object getParent() {
			return this.parent;
		}

		public String getName() {
			return this.name;
		}
	}

	protected class MetaclassContentProvider implements ITreeContentProvider {
		public Object[] getElements(Object inputElement) {
			InstancesForMetaclasses instancesForMetaclasses = (InstancesForMetaclasses) inputElement;
			InstancesForMetaclass[] instancesByMetaclass = instancesForMetaclasses
					.getInstancesForMetaclasses();

			if (MetaclassViewer.this.editorConfiguration.isShowDerivationTree()) {
				instancesByMetaclass = instancesForMetaclasses.getRootMetaclasses();
			}

			if (MetaclassViewer.this.editorConfiguration.isGroupByPackage()
					&& !MetaclassViewer.this.editorConfiguration.isShowDerivationTree()) {
				return groupByPackage(instancesByMetaclass);
			} else {
				return instancesByMetaclass;
			}
		}

		public void dispose() {
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}

		public Object[] getChildren(Object parentElement) {
			if (parentElement instanceof PackageGroup) {
				PackageGroup packageGroup = (PackageGroup) parentElement;
				return packageGroup.toArray(new Object[packageGroup.size()]);
			}

			if (MetaclassViewer.this.editorConfiguration.isShowDerivationTree()) {
				if (parentElement instanceof InstancesForMetaclass) {
					InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) parentElement;
					return instancesForMetaclass.getSubclasses();
				}
			}

			return new Object[0];
		}

		public Object getParent(Object element) {
			if (element instanceof PackageGroup) {
				PackageGroup packageGroup = (PackageGroup) element;
				return packageGroup.getParent();
			}

			if (element instanceof InstancesForMetaclass) {

				InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) element;
				if (MetaclassViewer.this.editorConfiguration.isShowDerivationTree()) {
					EList<EClass> superTypes = instancesForMetaclass.getEClass().getESuperTypes();
					if (superTypes.size() > 0) {
						// return the first parent found
						String superclassQualifiedName = EMFUtil
								.getMetaclassQualifiedName(superTypes.get(0));
						InstancesForMetaclass superclass = MetaclassViewer.this.editorConfiguration
								.getInstancesForMetaclasses().getInstancesForMetaclass(
										superclassQualifiedName);
						return superclass;
					}
				} else if (MetaclassViewer.this.editorConfiguration.isGroupByPackage()) {
					return instancesForMetaclass.getParent();
				}
			}

			return null;
		}

		public boolean hasChildren(Object element) {
			if (element instanceof PackageGroup) {
				PackageGroup packageGroup = (PackageGroup) element;
				return packageGroup.size() > 0;
			}

			if (MetaclassViewer.this.editorConfiguration.isShowDerivationTree()) {
				if (element instanceof InstancesForMetaclass) {
					InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) element;
					return instancesForMetaclass.getSubclasses().length > 0;
				}
			}

			return false;
		}
	}

	/** Initialize the viewer with a content and label provider */
	private void initViewer() {

		this.treeViewer.setContentProvider(new MetaclassContentProvider());
		this.treeViewer.setLabelProvider(new MetaclassLabelProvider());

		// A filter for empty metaclasses (which don't have any instance in the model)
		this.treeViewer.addFilter(new EmptyMetaclassesFilter());

		setSortMode(this.editorConfiguration.getMetaclassesSortMode());
	}

	/**
	 * Group the given metaclasses by packages.
	 * 
	 * @return a list of packages containing metaclasses or other packages recursively
	 */
	public Object[] groupByPackage(InstancesForMetaclass[] instancesByMetaclass) {

		ArrayList<Object> toplevelItems = new ArrayList<Object>();

		for (InstancesForMetaclass instancesForMetaclass : instancesByMetaclass) {
			EClass eClass = instancesForMetaclass.getEClass();
			// path segments in reverse order
			ArrayList<String> pathSegments = new ArrayList<String>();
			pathSegments.add(eClass.getName());

			EPackage ePackage = eClass.getEPackage();
			while (ePackage != null) {
				pathSegments.add(ePackage.getName());
				ePackage = ePackage.getESuperPackage();
			}

			// copy path segments in reverse in a string table
			String[] path = new String[pathSegments.size()];
			int i = pathSegments.size() - 1;
			for (String segment : pathSegments) {
				path[i--] = segment;
			}

			addToPackage(instancesForMetaclass, path, 0, toplevelItems, null);
		}

		return toplevelItems.toArray(new Object[toplevelItems.size()]);
	}

	/**
	 * A recursive function that creates a hierarchy of packages containing metaclasses
	 * 
	 * @param instancesForMetaclass
	 *            the metaclass to add to a package
	 * @param path
	 *            a list of package segments
	 * @param index
	 *            the index of the first package segment to consider
	 * @param items
	 *            the items corresponding to the level from which the path is defined (index)
	 * @param parent
	 *            the parent of the object in the tree
	 */
	private void addToPackage(InstancesForMetaclass instancesForMetaclass, String[] path,
			int index, ArrayList<Object> items, Object parent) {

		if (path.length - index == 1) {
			instancesForMetaclass.setParent(parent);
			items.add(instancesForMetaclass);
		} else if (path.length - index > 1) {
			// see if the package already exists
			for (Object item : items) {
				if (item instanceof PackageGroup) {
					PackageGroup packageGroup = (PackageGroup) item;
					if (packageGroup.getName().equals(path[index])) {
						// add the element to an existing package
						addToPackage(instancesForMetaclass, path, index + 1, packageGroup,
								packageGroup);
						return;
					}
				}
			}

			// create a new package for the element
			PackageGroup packageGroup = new PackageGroup(parent, path[index]);
			items.add(packageGroup);
			addToPackage(instancesForMetaclass, path, index + 1, packageGroup, packageGroup);
		}
	}

	public void refresh() {
		if (!this.treeViewer.getTree().isDisposed()) {
			try {
				this.treeViewer.getTree().setRedraw(false);
				this.treeViewer.refresh();
			} finally {
				this.treeViewer.getTree().setRedraw(true);
			}
		}
	}

	public void addSelectionChangedListener(ISelectionChangedListener selectionChangedListener) {
		this.treeViewer.addSelectionChangedListener(selectionChangedListener);
	}

	public void clearSelection() {
		this.treeViewer.setSelection(null);
	}

	public ISelection getSelection() {
		return this.treeViewer.getSelection();
	}

	public void setSortMode(MetaclassesSortMode mode) {
		this.editorConfiguration.setMetaclassesSortMode(mode);

		if (mode == MetaclassesSortMode.ByName) {
			this.treeViewer.setComparator(new ViewerComparator() {
				@Override
				public int compare(Viewer viewer, Object e1, Object e2) {
					String first = getDisplayedName(e1);
					String second = getDisplayedName(e2);
					return first.compareToIgnoreCase(second);
				}
			});
		} else if (mode == MetaclassesSortMode.ByCount) {
			this.treeViewer.setComparator(new ViewerComparator() {
				@Override
				public int compare(Viewer viewer, Object e1, Object e2) {
					int first = getCount(e1);
					int second = getCount(e2);
					return first - second;
				}
			});
		}
	}

	public void setShowMetaclassesFullQualifiedNames(boolean value) {
		this.editorConfiguration.setShowMetaclassesFullQualifiedNames(value);
		this.treeViewer.refresh();
	}

	public void setShowEmptyMetaclasses(boolean value) {
		this.editorConfiguration.setShowEmptyMetaclasses(value);
		refresh();
	}

	public void setGroupByPackage(boolean value) {
		this.editorConfiguration.setGroupByPackage(value);
		refresh();
	}

	public void setDisplayInstancesOfSubclasses(boolean value) {
		this.editorConfiguration.setDisplayInstancesOfSubclasses(value);
		refresh();
		// force the model tree to refresh
		if (!this.treeViewer.getSelection().isEmpty()) {
			this.treeViewer.setSelection(this.treeViewer.getSelection());
		}
	}

	public void setShowDerivationTree(boolean value) {
		this.editorConfiguration.setShowDerivationTree(value);
		refresh();
	}

	/** @return the name that is displayed for the following element in the viewer */
	private String getDisplayedName(Object object) {
		String name = "";
		if (object instanceof InstancesForMetaclass) {
			InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) object;
			if (this.editorConfiguration.isShowMetaclassesFullQualifiedNames()) {
				name = instancesForMetaclass.getClassQualifiedName();
			} else {
				name = instancesForMetaclass.getEClass().getName();
			}
		}
		return name;
	}

	/** @return the instance count for the given element (metaclass) in the viewer */
	private int getCount(Object object) {
		int count = 0;
		if (object instanceof InstancesForMetaclass) {
			InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) object;
			if (this.editorConfiguration.isDisplayInstancesOfSubclasses()) {
				count = instancesForMetaclass.totalSize();
			} else {
				count = instancesForMetaclass.size();
			}
		}
		return count;
	}

	public void setFont(Font font) {
		this.treeViewer.getTree().setFont(font);
	}

	public Viewer getViewer() {
		return this.treeViewer;
	}
}