/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 *******************************************************************************/

package org.eclipse.gmt.modisco.kdm.uml2converter.actions;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceImpl;
import org.eclipse.gmt.modisco.common.core.logging.AtlLogHandler;
import org.eclipse.gmt.modisco.discoverersmanager.Discoverer;
import org.eclipse.gmt.modisco.kdm.uml2converter.KDMtoUML2Converter;
import org.eclipse.gmt.modisco.kdm.uml2converter.KdmToUmlConverterInterface;
import org.eclipse.m2m.atl.common.ATLLogger;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;

/**
 * @author Gabriel Barbier
 * 
 */
public class DiscoverUmlModelFromKdmModel implements Discoverer {

	private final String sourceExtension = "kdm";
	private final String targetExtension = "uml";
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.gmt.modisco.discoverersmanager.DiscovererInterface#applyTo
	 * (java.lang.Object)
	 */
	public boolean applyTo(final Object source) {
		boolean result = false;
		if (source instanceof IFile) {
			IFile iFile = (IFile) source;
			String extension = iFile.getFileExtension();
			if ((extension != null) && (extension.equals(this.sourceExtension))) {
				result = true;
			}
		}
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seeorg.eclipse.gmt.modisco.discoverersmanager.DiscovererInterface#
	 * discoverElement(java.lang.Object, org.eclipse.emf.common.util.URI)
	 */
	public Resource discoverElement(final Object source, final URI target) {
		Resource resource = null;
		final Logger logger = Logger.getLogger(ATLLogger.LOGGER_ID);
		// retrieval of Java project
		final IFile iFile = (IFile) source;
		final String sourcePath = iFile.getLocation().toString();
		
		// retrieval of target path
		IPath tempPath = iFile.getLocation().removeFileExtension().addFileExtension(this.targetExtension);
		if (target != null) {
			if (target.isRelative()) {
				tempPath = ResourcesPlugin.getWorkspace().getRoot()
						.getFullPath().append(target.path());
			} else {
				tempPath = new Path(target.toFileString());
			}
		}

		final IPath path = tempPath;

		final AtlLogHandler logHandler = new AtlLogHandler(path.toString()
				+ ".log");
		logger.addHandler(logHandler);
		try {
			final KdmToUmlConverterInterface transfo = new KDMtoUML2Converter();
			Resource kdmModel = new XMIResourceImpl();
			kdmModel.setURI(URI.createFileURI(sourcePath));
			URI outputURI = URI.createFileURI(path.toString());
			
			transfo.getUML2ModelFromKDMModel(kdmModel, false, outputURI);
			
		} catch (Exception e) {
			logger.log(Level.WARNING, "Error in discovery of java project", e);
		} finally {
			logger.removeHandler(logHandler);
			logHandler.close();
			try {
				iFile.getParent().refreshLocal(1, null);
			} catch (Exception e1) {
				// give up
			}
		}
		// access UI resources so that we can obtain the current eclipse
		// workbench activePage
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				try {
					IWorkspaceRoot root = ResourcesPlugin.getWorkspace()
							.getRoot();
					IFile ifile = root.getFileForLocation(path);
					if ((ifile != null) && ifile.exists()) {
						IWorkbenchPage page = PlatformUI.getWorkbench()
								.getActiveWorkbenchWindow().getActivePage();
						// programmatically opens the j2se5 model using the
						// default editor if one is present
						IDE.openEditor(page, ifile);
					}
				} catch (PartInitException e) {
					e.printStackTrace();
				}
			}
		});

		return resource;
	}

	public String toString() {
		return "Discover UML model from KDM model";
	}

	public DiscoverUmlModelFromKdmModel() {
		for (String key : this.keys) {
			this.parameterKeys.add(key);
		}
	}

	private final Set<String> parameterKeys = new HashSet<String>();
	private final String[] keys = {"TargetModel", "TargetUri"};
	
	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.discoverersmanager.DiscovererInterface#getParameterKeys()
	 */
	public Set<String> getParameterKeys() {
		return this.parameterKeys;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.discoverersmanager.Discoverer#discoverElement(java.lang.Object, java.util.Map)
	 */
	public void discoverElement(final Object source, final Map<String, Object> parameters) {
		URI targetUri = null;
		if (parameters.containsKey(this.keys[1])) {
			Object value = parameters.get(this.keys[1]);
			if (value instanceof URI) {
				targetUri = (URI) value;
			}
		}
		Resource result = this.discoverElement(source, targetUri);
		parameters.put(this.keys[0], result);	
	}
}
