/**
 * Copyright (c) 2008 Mia-Software. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributor: 
 *     Fabien Giquel
 */

package org.eclipse.gmt.modisco.discoverer.kdm.source.popup.actions;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.common.core.utils.Tools;
import org.eclipse.gmt.modisco.discoverer.kdm.source.KDMSourceDiscoverer;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IObjectActionDelegate;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;

/**
 * This action launches the KDM source discovering for user workbench selected
 * elements.
 * 
 */
public class FromContainerToKDMSource implements IObjectActionDelegate {

	/**
	 * Default file extension for kdm models.
	 */
	private static final String KDM_EXTENSION = "kdm";

	/**
	 * Memorize use selection from the workbench.
	 */
	private List<IContainer> selectedContainers = new ArrayList<IContainer>();

	/**
	 * 
	 * @see org.eclipse.ui.IActionDelegate
	 *      #run(org.eclipse.jface.action.IAction).
	 * @param action
	 *            an action.
	 */
	public final void run(final IAction action) {
		
		final List<IPath> newModelsPaths = new ArrayList<IPath>();
		Job job = new Job("Discovering KDM source model") {
			protected IStatus run(final IProgressMonitor monitor) {
				try {
					KDMSourceDiscoverer disco = new KDMSourceDiscoverer();
					Resource kdmResource = null;
					if (FromContainerToKDMSource.this
							.getSelectedContainers() != null) {
						for (IContainer aContainer : FromContainerToKDMSource
								.this.getSelectedContainers()) {
							kdmResource = disco
									.getKDMModelFromContainer(aContainer);
							if (kdmResource != null) {
								IPath path = aContainer.getLocation()
										.addTrailingSeparator().append(
												aContainer.getName())
										.addFileExtension(KDM_EXTENSION);
								Tools.saveModel(kdmResource, path.toString());
								aContainer.refreshLocal(1, monitor);
								newModelsPaths.add(path);
							}
						}
					}
					return Status.OK_STATUS;
				} catch (Exception e) {
					e.printStackTrace();
					return Status.CANCEL_STATUS;
				}
			}
		};

		// triggers opening of the models as soon as saved
		job.addJobChangeListener(new JobChangeAdapter() {
			public void done(final IJobChangeEvent event) {
				FromContainerToKDMSource.this.openEditor(newModelsPaths);
			}
		});

		job.schedule();
	}

	/**
	 * Memorize user selected containers.
	 * 
	 * @param action
	 *            a user action.
	 * @param selection
	 *            a user selection.
	 */
	public final void selectionChanged(final IAction action,
			final ISelection selection) {
		getSelectedContainers().clear();

		for (Object selectedElement : ((IStructuredSelection) selection)
				.toList()) {
			if (selectedElement instanceof IJavaProject) {
				getSelectedContainers().add(
						(IContainer) ((IJavaProject) selectedElement)
								.getResource());
			} else if (selectedElement instanceof IContainer) {
				getSelectedContainers().add((IContainer) selectedElement);
			}
		}
	}

	/**
	 * Open default workbench editor on specified files.
	 * 
	 * @param newModelsPaths
	 *            models on which open editors.
	 */
	private void openEditor(final List<IPath> newModelsPaths) {
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				try {
					// access UI resources so that we can obtain the current
					// eclipse
					// workbench activePage
					IWorkspaceRoot root = ResourcesPlugin.getWorkspace()
							.getRoot();
					for (IPath aNewModelPath : newModelsPaths) {
						IFile ifile = root.getFileForLocation(aNewModelPath);
						if (ifile.exists()) {
							IWorkbenchPage page = PlatformUI.getWorkbench()
									.getActiveWorkbenchWindow().getActivePage();
							// opens the model using the default editor
							// if one is present
							IDE.openEditor(page, ifile);
						}
					}
				} catch (PartInitException e) {
					e.printStackTrace();
				}
			}
		});
	}

	/**
	 * 
	 * @see org.eclipse.ui.IObjectActionDelegate#setActivePart
	 *      (org.eclipse.jface.action.IAction, org.eclipse.ui.IWorkbenchPart)
	 * @param action
	 *            see super method
	 * @param targetPart
	 *            see super method
	 */
	public void setActivePart(final IAction action,
			final IWorkbenchPart targetPart) {
	}

	/**
	 * 
	 * @return the user selected containers
	 */
	public final List<IContainer> getSelectedContainers() {
		return this.selectedContainers;
	}
}
