package org.eclipse.gmt.modisco.discoverersmanager;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.gmt.modisco.discoverersmanager.handlers.DiscovererHandler;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

/**
 * The activator class controls the plug-in life cycle
 */
public class DiscoverersManager extends AbstractUIPlugin {

	// The plug-in ID
	public static final String PLUGIN_ID = "org.eclipse.gmt.modisco.discoverersmanager";

	// The shared instance
	private static DiscoverersManager plugin;
	
	/**
	 * The constructor
	 */
	public DiscoverersManager() {
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.plugin.AbstractUIPlugin#start(org.osgi.framework.BundleContext)
	 */
	public void start(BundleContext context) throws Exception {
		super.start(context);
		context.getBundle();
		plugin = this;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.plugin.AbstractUIPlugin#stop(org.osgi.framework.BundleContext)
	 */
	public void stop(BundleContext context) throws Exception {
		plugin = null;
		super.stop(context);
	}

	/**
	 * Returns the shared instance
	 *
	 * @return the shared instance
	 */
	public static DiscoverersManager getDefault() {
		return plugin;
	}

	/**
	 * Returns an image descriptor for the image file at the given
	 * plug-in relative path
	 *
	 * @param path the path
	 * @return the image descriptor
	 */
	public static ImageDescriptor getImageDescriptor(String path) {
		return imageDescriptorFromPlugin(PLUGIN_ID, path);
	}
	
	private static List<DiscovererHandler> discoverers;
	
	/**
	 * Accessor to all discoverers registered in DiscoverersManager
	 * through extensions.
	 * 
	 * @return the list of discoverers available in registry
	 */
	public static List<? extends Discoverer> getDiscoverers() {
		return DiscoverersManager.getDiscovererHandlers();
	}
	
	/**
	 * Technical accessor :
	 * Accessor to all discoverers registered in DiscoverersManager through extensions.
	 * Except that a discoverer handler provides also informations available in extension (name, path).
	 * 
	 * @return the list of discoverers available in registry
	 */
	public static List<DiscovererHandler> getDiscovererHandlers() {
		if (discoverers == null) {
			discoverers = new ArrayList<DiscovererHandler>();
			IExtensionRegistry registry = Platform.getExtensionRegistry();
			IExtensionPoint point = registry.getExtensionPoint("org.eclipse.gmt.modisco.discoverers"); //$NON-NLS-1$

			for (IExtension extension : point.getExtensions()) {
				for (IConfigurationElement element : extension.getConfigurationElements()) {
					String discovererClass = element.getAttribute("class");
					if (discovererClass != null) {
						try {
							
							Discoverer discoverer = (Discoverer) element.createExecutableExtension("class");
							DiscovererHandler handler = new DiscovererHandler(discoverer);
							
							String discovererName = element.getAttribute("name");
							if (discovererName == null) {
								handler.setName(discoverer.toString());
							} else {
								handler.setName(discovererName);
							}
							
							String discovererPath = element.getAttribute("path");
							if (discovererPath != null) {
								String[] paths = discovererPath.split("/");
								handler.setPaths(paths);
							}
							discoverers.add(handler);
						} catch (CoreException e) {
							e.printStackTrace();
						}
					}
				}
			}
		}
		return discoverers;
	}

	/**
	 * Accessor to one discoverer registered in DiscoverersManager
	 * through extensions.
	 * 
	 * @param discovererName name of a discoverer
	 * @return the discoverer with specified name, or null if there is no discoverer registered with specified name.
	 */
	public static Discoverer getDiscoverer(String discovererName) {
		Discoverer result = null;
		for (DiscovererHandler handler : DiscoverersManager.getDiscovererHandlers()) {
			if (handler.getName().equals(discovererName)) {
				result = handler;
			}
		}
		return result;
	}
}
