/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.editors.table;

import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Table;

/**
 * Allows sorting a table viewer's columns. Clicking on a column header cycles between ascending,
 * descending and no sorting.
 */
public abstract class TableSorter extends ViewerComparator {
	public static final int ASCENDING = -1;
	public static final int NONE = 0;
	public static final int DESCENDING = 1;

	/** Current sorting direction */
	private int direction = 0;

	/** The column that is managed by this {@link TableSorter} */
	private TableViewerColumn column;

	/** The table viewer containing the column to be sorted */
	private TableViewer viewer;

	/**
	 * @param viewer
	 *            the table viewer
	 * @param column
	 *            the column to provide sorting for
	 */
	public TableSorter(TableViewer viewer, TableViewerColumn column) {
		this.direction = NONE;
		this.column = column;
		this.viewer = viewer;

		column.getColumn().addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				columnClicked();
			}
		});
	}

	private void columnClicked() {
		// user changed sorting column : always start with descending sorting
		if (this.viewer.getComparator() == null || this.viewer.getComparator() != this) {
			setSorting(DESCENDING);
			return;
		}

		switch (this.direction) {
		case DESCENDING:
			setSorting(ASCENDING);
			break;
		case ASCENDING:
			setSorting(NONE);
			break;
		case NONE:
			setSorting(DESCENDING);
			break;
		}
	}

	public void setSorting(int direction) {
		this.direction = direction;
		Table table = this.viewer.getTable();
		switch (direction) {
		case DESCENDING:
			table.setSortColumn(this.column.getColumn());
			table.setSortDirection(SWT.UP);
			this.viewer.setComparator(this);
			// viewer.refresh();
			break;
		case ASCENDING:
			table.setSortColumn(this.column.getColumn());
			table.setSortDirection(SWT.DOWN);
			this.viewer.setComparator(this);
			this.viewer.refresh();
			break;
		case NONE:
			table.setSortColumn(null);
			table.setSortDirection(SWT.NONE);
			this.viewer.setComparator(null);
			// viewer.refresh();
			break;
		}
	}

	@Override
	public int compare(Viewer viewer, Object e1, Object e2) {
		return this.direction * doCompare(viewer, e1, e2);
	}

	/** Must be defined by subclasses to compare two objects */
	protected abstract int doCompare(Viewer viewer, Object e1, Object e2);

}
