/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.adapters;

import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.edit.provider.ITreeItemContentProvider;
import org.eclipse.gmt.modisco.common.editor.MoDiscoEditorPlugin;

/** An adapter to associate a parent to a model element. */
public class ParentAdapter extends AdapterImpl {

	/** The parent of the adapted model element */
	private Object parent;

	public ParentAdapter() {
	}

	/** Set the parent (null to unset it) */
	public void setParent(Object parent) {
		this.parent = parent;
	}

	/** Return the parent (can be null) */
	public Object getParent() {
		return this.parent;
	}

	@Override
	public boolean isAdapterForType(Object type) {
		return type == ParentAdapter.class;
	}

	/**
	 * Associate a parent with the given eObject, using a ParentAdapter.
	 * <p>
	 * Here, we must pay attention to the fact that an object can have several different parents in
	 * the tree viewer, due to the fact that we show backward links as well as forward links.
	 * <p>
	 * Problem is, JFace is expecting a single parent to be returned by the ContentProvider's
	 * getParent method. So, we always associate the parent that is closest to the root.
	 * <p>
	 * If the same eObject is found on a link following parents from this object to the root, then
	 * take the last one (going upward).
	 */
	public static void associateParent(EObject eObject, Object suggestedParent,
			AdapterFactory adapterFactory) {

		Object lastParent = suggestedParent;
		Object parent = suggestedParent;
		Object previous = null;

		// just in case it gets stuck in an infinite loop
		int maxIterations = 10000;

		while (parent != null) {

			Object adapter = adapterFactory.adapt(parent, ITreeItemContentProvider.class);
			if (adapter instanceof ITreeItemContentProvider) {
				ITreeItemContentProvider contentProvider = (ITreeItemContentProvider) adapter;
				parent = contentProvider.getParent(parent);

				// found the same object, make its parent the parent of the original one
				if (parent != null && previous == eObject) {
					lastParent = parent;
				}
			} else {
				parent = null;
			}

			previous = parent;

			if (maxIterations-- <= 0) {
				MoDiscoEditorPlugin.INSTANCE.log(new RuntimeException(
						"Infinite loop while trying to get parent").fillInStackTrace());
				lastParent = null;
				break;
			}
		}

		/*
		 * At this point, lastParent is the first parent of eObject coming from the root. Associate
		 * this parent with the object through the adapter.
		 */
		ParentAdapter adapter = (ParentAdapter) ParentAdapterFactory.getInstance().adapt(eObject,
				ParentAdapter.class);
		adapter.setParent(lastParent);
	}
}