/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.core;

import java.util.ArrayList;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmt.modisco.common.editor.MoDiscoEditorPlugin;
import org.eclipse.gmt.modisco.common.editor.util.EMFUtil;

/**
 * All the model elements for a metaclass. Each {@link InstancesForMetaclass} corresponds to a
 * metaclass, and has a list of all the instances of this metaclass in the model.
 */
public class InstancesForMetaclass extends AdapterImpl {

	/** All the model elements of the metaclass */
	private ArrayList<EObject> elements = new ArrayList<EObject>();

	/** All the model elements defined on derived metaclasses */
	private ArrayList<EObject> derivedElements = new ArrayList<EObject>();

	/** The list of subclasses of this metaclass */
	private ArrayList<InstancesForMetaclass> subclasses = new ArrayList<InstancesForMetaclass>();

	/** The EClass corresponding to this metaclass */
	private final EClass eClass;

	/** The list of all metaclasses. Used to find the instance list of super metaclasses. */
	private final InstancesForMetaclasses instancesForMetaclasses;

	/**
	 * @param eClass
	 *            the {@link EClass} corresponding to the metaclass.
	 */
	public InstancesForMetaclass(EClass eClass, InstancesForMetaclasses instancesForMetaclasses) {
		this.eClass = eClass;
		this.instancesForMetaclasses = instancesForMetaclasses;
	}

	public ArrayList<EObject> getElements() {
		return this.elements;
	}

	public ArrayList<EObject> getDerivedElements() {
		return this.derivedElements;
	}

	/** Add this instance to the list of instances of the metaclass */
	public void add(EObject object) {
		/* So that the object will notify us when it's changed */
		object.eAdapters().add(this);
		this.elements.add(object);

		// also add this object to super-classes
		addDerived(object);
	}

	/** Add this instance to the list of instances of metaclasses this metaclass is derived from */
	private void addDerived(EObject object) {
		EList<EClass> allSuperTypes = this.eClass.getEAllSuperTypes();
		for (EClass superclass : allSuperTypes) {
			InstancesForMetaclass instancesForSuperclass = this.instancesForMetaclasses
					.getInstancesForMetaclass(EMFUtil.getMetaclassQualifiedName(superclass));
			if (instancesForSuperclass != null) {
				instancesForSuperclass.derivedElements.add(object);
			} else {
				MoDiscoEditorPlugin.INSTANCE.log(EMFUtil.getMetaclassQualifiedName(superclass)
						+ " not found");
			}
		}
	}

	/** Remove this instance from the list of instances of the metaclass */
	public void remove(EObject object) {
		object.eAdapters().remove(this);
		this.elements.remove(object);

		// also remove this object from super-classes
		removeDerived(object, false);
	}

	/**
	 * Remove this instance from the list of instances of metaclasses this metaclass is derived from
	 */
	private void removeDerived(EObject object, boolean derived) {
		if (derived) {
			this.derivedElements.remove(object);
		}

		EList<EClass> allSuperTypes = this.eClass.getEAllSuperTypes();
		for (EClass superclass : allSuperTypes) {
			InstancesForMetaclass instancesForSuperclass = this.instancesForMetaclasses
					.getInstancesForMetaclass(EMFUtil.getMetaclassQualifiedName(superclass));
			instancesForSuperclass.removeDerived(object, true);
		}
	}

	/** A listener for change notifications */
	public interface ChangeListener {
		void notifyChanged(Notification msg);
	}

	/** A listener for change notifications */
	private ChangeListener listener = null;

	/** Cached class qualified name */
	private String classQualifiedName = null;

	/** A parent for the object when it is in a tree (when metaclasses are grouped by packages) */
	private Object parent;

	/** Set the listener that must be notified when the model changes */
	public void setListener(ChangeListener listener) {
		this.listener = listener;
	}

	@Override
	public void notifyChanged(Notification msg) {
		super.notifyChanged(msg);
		// pass on the message to the listener
		fireNotifyChanged(msg);

	}

	/** @return the number of instances of the metaclass */
	public int size() {
		return this.elements.size();
	}

	/** @return the number of instances of this metaclass or its subclasses */
	public int totalSize() {
		return this.elements.size() + this.derivedElements.size();
	}

	/** @return whether there is no instance of the metaclass */
	public boolean isEmpty() {
		return this.elements.isEmpty();
	}

	/** @return the EClass corresponding to this MetaclassListItemProvider */
	public EClass getEClass() {
		return this.eClass;
	}

	public String getClassQualifiedName() {
		if (this.classQualifiedName == null) {
			this.classQualifiedName = EMFUtil.getMetaclassQualifiedName(getEClass());
		}
		return this.classQualifiedName;
	}

	private void fireNotifyChanged(Notification msg) {
		if (this.listener != null) {
			this.listener.notifyChanged(msg);
		}
	}

	public void setParent(Object parent) {
		this.parent = parent;
	}

	public Object getParent() {
		return this.parent;
	}

	/** Add this class to the subclasses of its parents */
	public void buildParentsSubclasses() {
		EList<EClass> superTypes = this.eClass.getESuperTypes();
		for (EClass superclass : superTypes) {
			InstancesForMetaclass instancesForSuperclass = this.instancesForMetaclasses
					.getInstancesForMetaclass(EMFUtil.getMetaclassQualifiedName(superclass));
			if (instancesForSuperclass != null) {
				instancesForSuperclass.addSubclass(this);
			} else {
				MoDiscoEditorPlugin.INSTANCE.log(EMFUtil.getMetaclassQualifiedName(superclass)
						+ " not found");
			}
		}
	}

	/** Clears the list of subclasses of this class */
	public void clearSubclasses() {
		this.subclasses.clear();
	}

	public InstancesForMetaclass[] getSubclasses() {
		return this.subclasses.toArray(new InstancesForMetaclass[this.subclasses.size()]);
	}

	/** Add the given class to the subclasses of this class */
	private void addSubclass(InstancesForMetaclass instancesForMetaclass) {
		this.subclasses.add(instancesForMetaclass);
	}
}
