package org.eclipse.gmt.modisco.common.editor.editors.table;

import org.eclipse.gmt.modisco.common.editor.util.ImageProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;

public class TableEditorToolBar {

	private final TableEditor tableEditor;
	private final ToolBar toolBar;
	private final Label label;

	/** Hide columns which don't contain any elements */
	private ToolItem btnHideEmptyColumns;
	/**
	 * Whether to show columns which are specific to a few elements, or only columns which are
	 * common to all the elements
	 */
	private ToolItem btnOnlyShowCommonColumns;

	public TableEditorToolBar(final Composite parent, final TableEditor tableEditor) {
		this.tableEditor = tableEditor;

		addLinePainter(parent);

		GridData treeToolBarGridData = new GridData();
		treeToolBarGridData.grabExcessHorizontalSpace = true;
		treeToolBarGridData.horizontalAlignment = SWT.FILL;
		// since the layout is done manually, set fixed parent size
		treeToolBarGridData.heightHint = 25;
		parent.setLayoutData(treeToolBarGridData);

		this.label = new Label(parent, SWT.NONE);
		this.toolBar = new ToolBar(parent, SWT.NONE);

		/* do a manual layout to avoid the tool bar being pushed out of the screen by the label */
		parent.addControlListener(new ControlAdapter() {
			@Override
			public void controlResized(ControlEvent e) {
				Point toolbarSize = TableEditorToolBar.this.toolBar.computeSize(SWT.DEFAULT,
						SWT.DEFAULT);
				TableEditorToolBar.this.toolBar.setSize(toolbarSize);

				int splitPoint = parent.getSize().x - toolbarSize.x;

				TableEditorToolBar.this.toolBar.setLocation(splitPoint, 0);

				TableEditorToolBar.this.label.setLocation(3, 5);
				TableEditorToolBar.this.label.setSize(splitPoint - 6, parent.getSize().y - 7);
			}
		});

		createToolButtons();
		initialize();
	}

	/** Paints a line below the tool bar to delimit it neatly */
	private void addLinePainter(Composite composite) {
		composite.addPaintListener(new PaintListener() {
			public void paintControl(PaintEvent e) {
				e.gc.setForeground(Display.getDefault().getSystemColor(SWT.COLOR_GRAY));
				e.gc.drawLine(0, e.height - 1, e.width, e.height - 1);
			}
		});
	}

	private void createToolButtons() {
		this.btnHideEmptyColumns = new ToolItem(this.toolBar, SWT.CHECK);
		this.btnHideEmptyColumns.setToolTipText("Hide Empty Columns");
		this.btnHideEmptyColumns.setImage(ImageProvider.getInstance().getHideEmptyColumnsIcon());
		this.btnHideEmptyColumns.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				TableEditorToolBar.this.tableEditor
						.setHideEmptyColumns(TableEditorToolBar.this.btnHideEmptyColumns
								.getSelection());
			}
		});

		this.btnOnlyShowCommonColumns = new ToolItem(this.toolBar, SWT.CHECK);
		this.btnOnlyShowCommonColumns.setToolTipText("Only Show Common Columns");
		this.btnOnlyShowCommonColumns.setImage(ImageProvider.getInstance().getCommonColumnsIcon());
		this.btnOnlyShowCommonColumns.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				TableEditorToolBar.this.tableEditor
						.setOnlyShowCommonColumns(TableEditorToolBar.this.btnOnlyShowCommonColumns
								.getSelection());
			}
		});
	}

	/** Initialize the tool bar buttons with their initial value */
	private void initialize() {
		this.btnHideEmptyColumns.setSelection(false);
		this.btnOnlyShowCommonColumns.setSelection(false);
	}

	public void setLabelText(String text) {
		this.label.setText(text);
	}

}
