/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.adapters;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.UnexecutableCommand;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.edit.command.CommandParameter;
import org.eclipse.emf.edit.command.RemoveCommand;
import org.eclipse.emf.edit.command.SetCommand;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.provider.IEditingDomainItemProvider;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.emf.edit.provider.IStructuredItemContentProvider;
import org.eclipse.emf.edit.provider.ItemProviderAdapter;
import org.eclipse.gmt.modisco.common.editor.editors.EditorConfiguration;

/**
 * An item provider adapter for displaying in the model editor tree the list of instances of the
 * metaclass selected in the list of metaclasses.
 * <p>
 * Each MetaclassListItemProvider corresponds to a metaclass, and has a list of all the instances of
 * this metaclass in the model.
 */
public class MetaclassListItemProvider extends ItemProviderAdapter implements
		IStructuredItemContentProvider, IItemLabelProvider, IEditingDomainItemProvider {

	/** All the model elements of the same metaclass */
	private ArrayList<EObject> elements = new ArrayList<EObject>();
	/**
	 * The configuration of the editor used to display the list of metaclasses and the tree of model
	 * elements
	 */
	private final EditorConfiguration editorConfiguration;
	/** The EClass corresponding to this MetaclassListItemProvider */
	private final EClass eClass;

	public MetaclassListItemProvider(AdapterFactory adapterFactory,
			EditorConfiguration editorConfiguration, EClass eClass) {
		super(adapterFactory);
		this.editorConfiguration = editorConfiguration;
		this.eClass = eClass;
	}

	/**
	 * The elements are cached, so that the same elements are returned each time the getElements
	 * method is called, as long as the list of elements is unchanged. This is needed because the
	 * JFace viewers' selection system relies on the physical identity of the objects.
	 */
	private Collection<?> cachedElements = null;
	/** Whether the elements have to be rebuilt */
	private boolean dirty;

	/** Invalidate the cache */
	private void markDirty() {
		this.dirty = true;
	}

	private boolean oldSortInstances;

	@Override
	public Collection<?> getElements(Object object) {
		// Invalidate cache so that elements are recomputed
		if (this.editorConfiguration.isSortInstances() != this.oldSortInstances) {
			markDirty();
		}
		this.oldSortInstances = this.editorConfiguration.isSortInstances();

		if (this.cachedElements == null || this.dirty) {
			this.cachedElements = BigListItemProvider.splitElements(this, null, this.elements,
					this.adapterFactory, this.editorConfiguration, this.cachedElements);
			this.dirty = false;
		}
		return this.cachedElements;
	}

	@Override
	public Object getImage(Object object) {
		return null;
	}

	@Override
	public String getText(Object object) {
		// this is never displayed, since this element is the root
		return "Instances";
	}

	/** Add this instance to the list of instances of the metaclass */
	public void add(EObject object) {
		/* So that the object will notify us when it's changed */
		object.eAdapters().add(this);
		this.elements.add(object);
		markDirty();
	}

	/** Remove this instance from the list of instances of the metaclass */
	public void remove(EObject object) {
		object.eAdapters().remove(this);
		this.elements.remove(object);
		markDirty();
	}

	/** A listener for change notifications */
	public interface ChangeListener {
		void notifyChanged(Notification msg);
	}

	/** A listener for change notifications */
	private ChangeListener listener = null;

	/** Set the listener that must be notified when the model changes */
	public void setListener(ChangeListener listener) {
		this.listener = listener;
	}

	@Override
	public void notifyChanged(Notification msg) {
		super.notifyChanged(msg);
		/* pass on the message to the listener */
		if (this.listener != null) {
			this.listener.notifyChanged(msg);
		}
	}

	/** @return the number of instances of the metaclass */
	public int size() {
		return this.elements.size();
	}

	/** @return whether there is no instance of the metaclass */
	public boolean isEmpty() {
		return this.elements.isEmpty();
	}

	/** @return the EClass corresponding to this MetaclassListItemProvider */
	public EClass getEClass() {
		return this.eClass;
	}

	/**
	 * This method is overridden so that it is possible to delete elements that are children of a
	 * MetaclassListItemProvider.
	 */
	@Override
	public Command createCommand(Object object, EditingDomain domain,
			Class<? extends Command> commandClass, CommandParameter commandParameter) {

		// the real owner of the model objects impacted by the command
		EObject owner = null;
		// the feature that contains the model objects impacted by the command
		EStructuralFeature containingFeature = null;

		Collection<?> collection = commandParameter.getCollection();
		for (Object o : collection) {
			if (o instanceof EObject) {
				EObject eObject = (EObject) o;
				// all the objects don't have the same container => cannot create command
				if (owner != null && owner != eObject.eContainer() || containingFeature != null
						&& containingFeature != eObject.eContainingFeature())
					return UnexecutableCommand.INSTANCE;
				owner = eObject.eContainer();
				containingFeature = eObject.eContainingFeature();
			}
		}

		if (owner == null || containingFeature == null)
			return UnexecutableCommand.INSTANCE;

		// remove command only works on multiplicity many features
		if (commandClass == RemoveCommand.class && !containingFeature.isMany()) {
			return new SetCommand(domain, owner, containingFeature, commandParameter.value);
		}

		commandParameter.setOwner(owner);
		commandParameter.feature = containingFeature;
		return super.createCommand(owner, domain, commandClass, commandParameter);
	}

	@Override
	public Collection<?> getNewChildDescriptors(Object object, EditingDomain editingDomain,
			Object sibling) {
		return Collections.emptyList();
	}

	@Override
	public Collection<?> getChildren(Object object) {
		return Collections.emptyList();
	}

	@Override
	public Object getParent(Object object) {
		return null;
	}
}
