/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Payneau (Mia-Software) - initial API and implementation
 */
package org.eclipse.gmt.modisco.workflow.ui;

import java.util.HashMap;
import java.util.Map.Entry;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.gmt.modisco.workflow.Activator;
import org.eclipse.gmt.modisco.workflow.Constant;
import org.eclipse.gmt.modisco.workflow.DriverInterface;
import org.eclipse.gmt.modisco.workflow.controller.ControllerDriverPage;
import org.eclipse.gmt.modisco.workflow.ui.common.Common;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;

public class LoadLauncherPage extends WizardPage{

	private Table driversTable;
	private ControllerDriverPage controller;
	private HashMap<TableItem, ILaunchConfiguration> launchMap;
	private Composite container;
	private String transformationName;
	private ILaunchConfiguration launchConfigurationSelected;
	
	/**
	 * Class constructor
	 * @param pageName Name of the wizard page
	 * @param controller The controller managing the display
	 */
	protected LoadLauncherPage(String pageName, ControllerDriverPage controller) {
		super(pageName, pageName, null);
		this.setImageDescriptor(Activator.getImageDescriptor(Constant.pathBigImage));
		this.controller = controller;
		this.launchMap = new HashMap<TableItem, ILaunchConfiguration>();
		this.setDescription("This wizard represents the list of the launcher loaded.");
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		this.container = new Composite(parent, SWT.NULL);
		GridLayout layout = new GridLayout();
		this.container.setLayout(layout);
		layout.numColumns = 3;
		layout.verticalSpacing = 9;
	
		GridData gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		gridData.verticalAlignment = org.eclipse.swt.layout.GridData.FILL;
		gridData.grabExcessVerticalSpace = true;
		this.driversTable = new Table(this.container, SWT.BORDER | SWT.V_SCROLL);
		this.driversTable.setLayoutData(gridData);
		this.driversTable.addSelectionListener(new SelectionListener(){
			public void widgetDefaultSelected(SelectionEvent e) {}
			public void widgetSelected(SelectionEvent e) {
				TableItem itemSelected = LoadLauncherPage.this.driversTable.getItem(LoadLauncherPage.this.driversTable.getSelectionIndex());
				LoadLauncherPage.this.transformationName = itemSelected.getText();
				LoadLauncherPage.this.launchConfigurationSelected = LoadLauncherPage.this.launchMap.get(itemSelected);
				setPageComplete(true);
			}
		});
		this.driversTable.addMouseListener(new MouseAdapter(){
			@Override
			public void mouseDoubleClick(MouseEvent e) {
				LoadLauncherPage.this.getWizard().performFinish();
				LoadLauncherPage.this.getShell().close();
			}
		});
		loadLauncher();
		initializeDialogUnits(this.container);
		setControl(this.container);
		
	}
	
	/**
	 * Load all the launchers in the table which one driver is defined 
	 */
	private void loadLauncher(){
		ILaunchManager manager = DebugPlugin.getDefault().getLaunchManager();
		HashMap<String,DriverInterface> driverList = this.controller.getMapNameDriver();
		try {
			for(ILaunchConfiguration launch : manager.getLaunchConfigurations()){
				for(Entry<String,DriverInterface> entry : driverList.entrySet()){
					if(entry.getValue().getTypeLaunchConfig().equals(launch.getType().getIdentifier())){
						TableItem item = new TableItem(this.driversTable,SWT.NULL);
						item.setText(launch.getName());
						item.setImage(Common.retrieveImageDriver(entry.getValue().getTypeLaunchConfig()));
						this.launchMap.put(item,launch);
						break;
					}
				}
			}
		} catch (CoreException e) {
			e.printStackTrace();
		}
		setMessage("Select your launcher");
		setPageComplete(false);
	}

	/**
	 * Allows to retrieve the transformation name selected
	 * @return Transformation name selected
	 */
	public String getTransformationName() {
		return this.transformationName;
	}

	/**
	 * Allows to retrieve the type of the launcher selected in the table
	 * @return the type of the launcher selected in the table
	 */
	public String getLauncherType() {
		String type = null;
		try {
			type = this.launchConfigurationSelected.getType().getIdentifier();
		} catch (CoreException e) {
			e.printStackTrace();
		}
		return type;
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.DialogPage#setVisible(boolean)
	 */
	@Override
	public void setVisible(boolean visible) {
		if(visible){
			this.getShell().setSize(400, 400);
			this.getShell().setLocation(Common.centerShell(Display.getDefault(),this.getShell()));
		}
		super.setVisible(visible);
	}
}
