/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software)
 *******************************************************************************/
package org.eclipse.gmt.modisco.common.editor.extension.j2se5;

import java.util.HashMap;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.gmt.modisco.common.editor.extensions.naming.AbstractNameProvider;

/** Provides names to model elements of the J2SE5 metamodel */
public class NameProvider extends AbstractNameProvider {

	/**
	 * Maps metaclasses names with an enum constant for fast dispatching to the right handler
	 * (instead of comparing strings)
	 */
	HashMap<String, MetaClassName> mapping = new HashMap<String, MetaClassName>();

	private enum MetaClassName {
		NamedElementRef, ArrayType, ArrayAccess, CastExpression, CatchClause, BooleanLiteral,
		VariableDeclarationStatement, MethodInvocation, Annotation, ArrayCreation,
		ArrayLengthAccess, Block, ClassInstanceCreation, ConstructorInvocation,
		EnhancedForStatement, FieldAccess, ImportDeclaration, Modifier, ParameterizedType,
		SuperMethodInvocation, SuperConstructorInvocation, ThrowStatement,
		VariableDeclarationExpression
	};

	public NameProvider() {
		for (MetaClassName metaClassName : MetaClassName.values()) {
			this.mapping.put(metaClassName.name(), MetaClassName.valueOf(metaClassName.name()));
		}
	}

	public String getName(EObject eObject) {
		
		EClass eClass = eObject.eClass();
		String className = eClass.getName();

		MetaClassName metaClassName = this.mapping.get(className);

		if (metaClassName == null)
			return getDefaultName(eObject);

		switch (metaClassName) {
		case Annotation: {
			// Annotation -> type
			Object type = getFeature(eObject, "type");
			if (type instanceof EObject) {
				EObject eType = (EObject) type;
				return getNamedElementRefName(eType);
			}
			break;
		}

		case ArrayAccess: {
			// ArrayAccess -> array
			Object array = getFeature(eObject, "array");
			if (array instanceof EObject) {
				EObject eArray = (EObject) array;
				return getNamedElementRefName(eArray);
			}
			break;
		}
		case ArrayCreation: {
			// ArrayCreation -> type
			// get number of dimensions
			Object dimensions = getFeature(eObject, "dimensions");
			int dimension = 0;
			if (dimensions instanceof EList) {
				EList<?> eDimensions = (EList<?>) dimensions;
				dimension = eDimensions.size();
			}

			Object type = getFeature(eObject, "type");
			if (type instanceof EObject) {
				EObject eType = (EObject) type;
				String name = getNamedElementRefName(eType);
				for (int i = 0; i < dimension; i++) {
					name = name + "[]";
				}
				return name;
			}
			break;
		}
		case ArrayLengthAccess: {
			// ArrayLengthAccess -> array
			Object array = getFeature(eObject, "array");
			if (array instanceof EObject) {
				EObject eArray = (EObject) array;
				return getNamedElementRefName(eArray);
			}
			break;
		}
		case ArrayType: {
			// ArrayType -> elementType
			Object elementType = getFeature(eObject, "elementType");
			if (elementType instanceof EObject) {
				EObject eElementType = (EObject) elementType;
				return getNamedElementRefName(eElementType);
			}
			break;
		}
		case Block: {
			// Block -> statements.length
			// get number of statements
			Object statements = getFeature(eObject, "statements");
			int nStatements = 0;
			if (statements instanceof EList) {
				EList<?> eStatements = (EList<?>) statements;
				nStatements = eStatements.size();
			}
			return "" + nStatements;
		}
		case BooleanLiteral: {
			// BooleanLiteral -> value
			Object value = getFeature(eObject, "value");
			if (value instanceof Boolean) {
				return value.toString();
			}
			break;
		}
		case CastExpression: {
			// CastExpression -> type
			Object type = getFeature(eObject, "type");
			if (type instanceof EObject) {
				EObject eType = (EObject) type;
				return getNamedElementRefName(eType);
			}
			break;
		}
		case CatchClause: {
			// CatchClause -> exception -> type
			Object exception = getFeature(eObject, "exception");
			if (exception instanceof EObject) {
				EObject eException = (EObject) exception;
				Object type = getFeature(eException, "type");
				if (type instanceof EObject) {
					EObject eType = (EObject) type;
					return getNamedElementRefName(eType);
				}
			}
			break;
		}
		case ClassInstanceCreation: {
			// ClassInstanceCreation -> type
			Object type = getFeature(eObject, "type");
			if (type instanceof EObject) {
				EObject eType = (EObject) type;
				return getNamedElementRefName(eType);
			}
			break;
		}
		case ConstructorInvocation: {
			// ConstructorInvocation -> method
			Object method = getFeature(eObject, "method");
			if (method instanceof EObject) {
				EObject eMethod = (EObject) method;
				return getNamedElementRefName(eMethod);
			}
			break;
		}
		case EnhancedForStatement: {
			// EnhancedForStatement -> parameter -> type
			Object parameter = getFeature(eObject, "parameter");
			if (parameter instanceof EObject) {
				EObject eParameter = (EObject) parameter;
				Object type = getFeature(eParameter, "type");
				if (type instanceof EObject) {
					EObject eType = (EObject) type;
					return getNamedElementRefName(eType);
				}
			}
			break;
		}
		case FieldAccess: {
			// FieldAccess -> field
			Object field = getFeature(eObject, "field");
			if (field instanceof EObject) {
				EObject eField = (EObject) field;
				return getNamedElementRefName(eField);
			}
			break;
		}
		case ImportDeclaration: {
			// ImportDeclaration -> importedElement
			Object importedElement = getFeature(eObject, "importedElement");
			if (importedElement instanceof EObject) {
				EObject eImportedElement = (EObject) importedElement;
				String name = getNamedElementRefName(eImportedElement);

				Object element = getFeature(eImportedElement, "qualifier");
				while (element instanceof EObject) {
					name = getNamedElementRefName((EObject) element) + "." + name;
					element = getFeature((EObject) element, "qualifier");
				}

				return name;
			}
			break;
		}
		case MethodInvocation: {
			// MethodInvocation -> method
			Object method = getFeature(eObject, "method");
			if (method instanceof EObject) {
				EObject eMethod = (EObject) method;
				return getNamedElementRefName(eMethod);
			}
			break;
		}
		case Modifier: {
			// Modifier ->
			// visibility,isNative,isStatic,isStrictfp,isSynchronized,isTransient,isVolatile
			Object visibility = getFeature(eObject, "visibility");
			String name = "";
			if (visibility instanceof String) {
				name += visibility;
			}

			Object isNative = getFeature(eObject, "isNative");
			if (isNative instanceof Boolean && ((Boolean) isNative)) {
				name += " native";
			}
			Object isStatic = getFeature(eObject, "isStatic");
			if (isStatic instanceof Boolean && ((Boolean) isStatic)) {
				name += " static";
			}
			Object isStrictfp = getFeature(eObject, "isStrictfp");
			if (isStrictfp instanceof Boolean && ((Boolean) isStrictfp)) {
				name += " strictfp";
			}
			Object isSynchronized = getFeature(eObject, "isSynchronized");
			if (isSynchronized instanceof Boolean && ((Boolean) isSynchronized)) {
				name += " synchronized";
			}
			Object isTransient = getFeature(eObject, "isTransient");
			if (isTransient instanceof Boolean && ((Boolean) isTransient)) {
				name += " transient";
			}
			Object isVolatile = getFeature(eObject, "isVolatile");
			if (isVolatile instanceof Boolean && ((Boolean) isVolatile)) {
				name += " volatile";
			}

			return name;
		}
		case NamedElementRef: {
			// NamedElementRef -> element
			Object element = getFeature(eObject, "element");
			if (element instanceof EObject) {
				EObject eElement = (EObject) element;

				String name = getName(eElement);
				// Object name = getFeature(eElement, "name");

				EClass eElementClass = eElement.eClass();
				String strElementClass = "";
				if (eElementClass != null) {
					strElementClass = " (" + eElementClass.getName() + ")";
				}

				return "-> " + name + strElementClass;
			}

			return "";
		}
		case ParameterizedType: {
			// ParameterizedType -> type
			Object type = getFeature(eObject, "type");
			if (type instanceof EObject) {
				EObject eType = (EObject) type;
				return getNamedElementRefName(eType);
			}
			break;
		}
		case SuperConstructorInvocation: {
			// SuperConstructorInvocation -> method
			Object method = getFeature(eObject, "method");
			if (method instanceof EObject) {
				EObject eMethod = (EObject) method;
				return getNamedElementRefName(eMethod);
			}
			break;
		}
		case SuperMethodInvocation: {
			// SuperMethodInvocation -> method
			Object method = getFeature(eObject, "method");
			if (method instanceof EObject) {
				EObject eMethod = (EObject) method;
				return getNamedElementRefName(eMethod);
			}
			break;
		}
		case ThrowStatement: {
			// ThrowStatement -> expression
			// the thrown class is often in the expression
			Object expression = getFeature(eObject, "expression");
			if (expression instanceof EObject) {
				EObject eExpression = (EObject) expression;
				return getName(eExpression);
			}
			break;
		}
		case VariableDeclarationExpression: {
			// VariableDeclarationExpression -> type
			Object type = getFeature(eObject, "type");
			if (type instanceof EObject) {
				EObject eType = (EObject) type;
				return getNamedElementRefName(eType);
			}
			break;
		}
		case VariableDeclarationStatement: {
			// VariableDeclarationStatement -> type
			Object type = getFeature(eObject, "type");
			if (type instanceof EObject) {
				EObject eType = (EObject) type;
				return getNamedElementRefName(eType);
			}
			break;
		}
		}

		// // ArrayInitializer -> eContainer -> name
		// if (className.equals("ArrayInitializer")) {
		// EObject container = eObject.eContainer();
		// if(container != null) {
		// return getName(container);
		// }
		// }

		return getDefaultName(eObject);

	}

	private String getNamedElementRefName(EObject eObject) {
		Object element = getFeature(eObject, "element");
		if (element instanceof EObject) {
			EObject eElement = (EObject) element;
			return getName(eElement);
		}
		return "";
	}

	private Object getFeature(EObject eObject, String featureName) {
		if (eObject == null)
			return null;

		EClass eClass = eObject.eClass();
		EStructuralFeature structuralFeature = eClass.getEStructuralFeature(featureName);
		if (structuralFeature != null) {
			return eObject.eGet(structuralFeature);
		}

		return null;
	}

}
