/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Payneau (Mia-Software) - initial API and implementation
 *******************************************************************************/
package org.eclipse.gmt.modisco.workflow.controller;

import java.util.HashMap;
import java.util.Map.Entry;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.gmt.modisco.workflow.Constants;
import org.eclipse.gmt.modisco.workflow.DriverExecutionInterface;
import org.eclipse.gmt.modisco.workflow.DriverInterface;
import org.eclipse.gmt.modisco.workflow.ui.DriverPage;

/**
 * This class defines one controller which control the actions proceed to DriverPage (page of WizardDriver)
 * @author Nicolas Payneau
 */
public class ControllerDriverPage {

	private DriverPage driverPage;
	private HashMap <String,DriverInterface>  mapNameDrivers;
	private DriverInterface currentDriver;
	private ILaunchConfigurationWorkingCopy configurationEdition;
	private String typeEdition;
	
	/**
	 * Class constructor
	 */
	public ControllerDriverPage(){
		this.mapNameDrivers = new HashMap <String,DriverInterface>();
		initialization();
	}

	/***
	 * Allows to refresh the {@link DriverPage} page
	 * @param type Launcher type
	 */
	public void refreshDriverPage(String type){
		if(this.mapNameDrivers.size()!=0){
			this.currentDriver = this.mapNameDrivers.get(getDriverNameFromLaunchType(type));
			if(this.currentDriver!=null){
				this.driverPage.createTab(this.currentDriver.getTypeLaunchConfig());
			}
		}
		this.driverPage.setTitle(getDriverNameFromLaunchType(type));
		this.driverPage.refresh();
	}

	/**
	 * Load all plug-ins that refer to extension point Constant.nameExtensionPointDriver
	 */
	private void initialization(){
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint point = registry.getExtensionPoint(Constants.nameExtensionPointDriver); 

		for (IExtension extension : point.getExtensions()) {
			for (IConfigurationElement element : extension.getConfigurationElements()) {
				if(element.getName().equals(Constants.nameElementDriver)){
					try {
						this.mapNameDrivers.put(element.getAttribute("name"),(DriverInterface)element.createExecutableExtension("class"));
					} catch (CoreException e) {
						e.printStackTrace();
					}
				}
			}
		}
	}
	
	/**
	 * Perform apply on transformation specified.
	 * @return the configuration saved
	 */
	public ILaunchConfigurationWorkingCopy launchPerformFinish(){
		return this.driverPage.performApply();
	}
	
	/**
	 * Allows to retrieve the driver name from launcher type
	 * @param LaunchType Launcher type
	 * @return DriverName corresponding to launcher type specified
	 */
	public String getDriverNameFromLaunchType(String LaunchType) {
		String name = null;
		for(Entry<String,DriverInterface> entry : this.mapNameDrivers.entrySet()){
			if(entry.getValue().getTypeLaunchConfig().equals(LaunchType)){
				name = entry.getKey();
				break;
			}
		}
		return name;
	}
	
	/**
	 * Allows to retrieve an instance of {@link DriverInterface} from launcher type
	 * @param LaunchType Launcher type
	 * @return An instance of {@link DriverInterface}
	 */
	public DriverInterface getDriverFromLaunchType(String LaunchType){
		return this.mapNameDrivers.get(getDriverNameFromLaunchType(LaunchType));
	}
	
	/**
	 * Allows to retrieve a instance of {@link DriverExecutionInterface} for execute a
	 * configuration
	 * 
	 * @param type
	 *            The launcher type
	 * @return a instance of {@link DriverExecutionInterface} corresponding at type
	 *         specified
	 */
	public DriverExecutionInterface findPlugIn(String type) {
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint point = registry
				.getExtensionPoint(Constants.nameExtensionPointDriver); 
	
		for (IExtension extension : point.getExtensions()) {
			for (IConfigurationElement element : extension
					.getConfigurationElements()) {
				if (element.getName().equals(Constants.nameElementDriver)) {
					try {
						DriverInterface driver = (DriverInterface) element
								.createExecutableExtension("class");
						if (driver.getTypeLaunchConfig().equals(type)) {
							return findExecutionPlugIn(extension
									.getSimpleIdentifier());
						}
					} catch (CoreException e) {
						e.printStackTrace();
					}
				}
			}
		}
		return null;
	}
	
	/**
	 * Allows to retrieve a instance of {@link DriverExecutionInterface} for execute a
	 * configuration
	 * 
	 * @param id
	 *            The id of plug-in specified
	 * @return a instance of {@link DriverExecutionInterface} corresponding at id
	 *         specified
	 */
	private DriverExecutionInterface findExecutionPlugIn(String id) {
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint point = registry
				.getExtensionPoint(Constants.nameExtensionPointDriver); 
	
		for (IExtension extension : point.getExtensions()) {
			if (extension.getSimpleIdentifier().equals(id)) {
				for (IConfigurationElement element : extension
						.getConfigurationElements()) {
					if (element.getName().equals(Constants.nameElementExecution)) {
						try {
							return (DriverExecutionInterface) element
									.createExecutableExtension("class");
						} catch (CoreException e) {
							e.printStackTrace();
						}
					}
				}
			}
		}
		return null;
	}	
	
	/*
	 * Getter and setter	
	 */
	public ILaunchConfigurationWorkingCopy getConfigurationEdition() {
		return this.configurationEdition;
	}

	public String getTypeEdition() {
		return this.typeEdition;
	}

	public void setConfigurationEdition(ILaunchConfigurationWorkingCopy configuration){
		this.configurationEdition = configuration;
	}
	
	public void setTypeEdition(String type){
		this.typeEdition = type;
	}
	
	public DriverInterface getCurrentDriver(){
		return this.currentDriver;
	}

	public void setDriverPage(DriverPage driverPage) {
		this.driverPage = driverPage;		
	}
	
	public HashMap<String, DriverInterface> getMapNameDriver() {
		return this.mapNameDrivers;
	}
	
	public DriverPage getDriverPage() {
		return this.driverPage;
	}
}
