/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Hugues Dubourg (Mia-Software) - initial API and implementation
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 *******************************************************************************/

package org.eclipse.gmt.modisco.kdm.uml2converter;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.common.core.utils.Tools;
import org.eclipse.gmt.modisco.common.core.utils.UriUtils;
import org.eclipse.m2m.atl.common.ATLLaunchConstants;
import org.eclipse.m2m.atl.core.ATLCoreException;
import org.eclipse.m2m.atl.core.IExtractor;
import org.eclipse.m2m.atl.core.IInjector;
import org.eclipse.m2m.atl.core.IModel;
import org.eclipse.m2m.atl.core.IReferenceModel;
import org.eclipse.m2m.atl.core.ModelFactory;
import org.eclipse.m2m.atl.core.emf.EMFModel;
import org.eclipse.m2m.atl.core.launch.ILauncher;
import org.eclipse.m2m.atl.core.service.CoreService;
import org.eclipse.m2m.atl.core.ui.vm.asm.ASMModelWrapper;
import org.eclipse.m2m.atl.drivers.emf4atl.ASMEMFModel;
import org.eclipse.m2m.atl.engine.parser.AtlParser;
import org.eclipse.m2m.atl.engine.vm.nativelib.ASMModel;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;

public class KDMtoUML2Converter implements KdmToUmlConverterInterface {

	public final String mmLocation = "resources/metamodels";
	public final String transfoLocation = "resources/transformations";
	public final String kdmMMUri = "http://www.eclipse.org/MoDisco/kdm/action";
	public final String umlMMUri = "http://www.eclipse.org/uml2/2.1.0/UML";

	private final String atlMMPath = "http://www.eclipse.org/gmt/2005/ATL";
	
	public KDMtoUML2Converter() {

	}

	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.kdm.uml2converter.KdmToUmlConverterInterface#getUML2ModelFromKDMModel(org.eclipse.emf.ecore.resource.Resource, boolean, org.eclipse.emf.common.util.URI)
	 */
	public Resource[] getUML2ModelFromKDMModel(Resource kdmModelUri,
			boolean generateTraces, URI umlTargetModelUri) throws IOException, ATLCoreException {
		URL transformation;
		if (generateTraces) {
			transformation = this.getClass().getResource(
					this.transfoLocation + "/KDMtoUMLWithTraces.asm");
		} else {
			transformation = this.getClass().getResource(
					this.transfoLocation + "/KDMtoUML.asm");
		}

		return this.getUML2ModelFromKDMModelWithCustomTransformation(
				kdmModelUri.getURI(), generateTraces, transformation,
				umlTargetModelUri);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.kdm.uml2converter.KdmToUmlConverterInterface#getUML2ModelFromKDMModelWithCustomTransformation(org.eclipse.emf.common.util.URI, boolean, java.net.URL, org.eclipse.emf.common.util.URI)
	 */
	public Resource[] getUML2ModelFromKDMModelWithCustomTransformation(
			URI kdmSourceModelUri, boolean generateTraces,
			final URL transformation, URI umlTargetModelUri) throws IOException, ATLCoreException {

		Map<String, String> modelHandlers = new HashMap<String, String>();
		String umlMetaModelName = "uml";
		modelHandlers.put(umlMetaModelName, "UML2");
		String kdmMetaModelName = "kdm";
		modelHandlers.put(kdmMetaModelName, "EMF");
		String amwMetaModelName = "Trace";
		if (generateTraces) {
			modelHandlers.put(amwMetaModelName, "EMF");
//			modelHandlers.put(amwMetaModelName, "AMW");
		}
		
		Map<String, Object> launcherOptions = new HashMap<String, Object>();
		launcherOptions.put(ATLLaunchConstants.OPTION_MODEL_HANDLER, modelHandlers);

		String launcherName = ATLLaunchConstants.REGULAR_VM_NAME;
		final ILauncher launcher = CoreService.getLauncher(launcherName);
		launcher.initialize(launcherOptions);
		
		ModelFactory emfFactory = CoreService.createModelFactory(launcher.getDefaultModelFactoryName());
		IInjector emfInjector = CoreService.getInjector(emfFactory.getDefaultInjectorName());
		IExtractor emfExtractor = CoreService.getExtractor(emfFactory.getDefaultExtractorName());
		
//		ModelFactory umlFactory = CoreService.createModelFactory("UML2");
		ModelFactory umlFactory = emfFactory;
		IInjector umlInjector = CoreService.getInjector(umlFactory.getDefaultInjectorName());
		IExtractor umlExtractor = CoreService.getExtractor(umlFactory.getDefaultExtractorName());

		// Chargement du mta modle KDM
		Map<String, Object> referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(kdmMetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", kdmMetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", this.kdmMMUri); //$NON-NLS-1$
		IReferenceModel kdmMM = emfFactory.newReferenceModel(referenceModelOptions);
		emfInjector.inject(kdmMM, this.kdmMMUri);
		// Chargement du modle KDM
		Map<String, Object> modelOptions = new HashMap<String, Object>();
		String inModelName = "kdmInput";
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", UriUtils.toString(kdmSourceModelUri)); //$NON-NLS-1$
		modelOptions.put("newModel", false); //$NON-NLS-1$
		IModel input = emfFactory.newModel(kdmMM, modelOptions);
		emfInjector.inject(input, UriUtils.toString(kdmSourceModelUri));
		launcher.addInModel(input, inModelName, kdmMetaModelName);
		
		// Chargement du mta modle UML
		referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(umlMetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", umlMetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", this.umlMMUri); //$NON-NLS-1$
		IReferenceModel umlMM = umlFactory.newReferenceModel(referenceModelOptions);
		umlInjector.inject(umlMM, this.umlMMUri);
		// Chargement du modle UML
		modelOptions = new HashMap<String, Object>();
		inModelName = "umlOutput";
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", UriUtils.toString(umlTargetModelUri)); //$NON-NLS-1$
		modelOptions.put("newModel", true); //$NON-NLS-1$
		IModel outputInstance = umlFactory.newModel(umlMM, modelOptions);
		launcher.addOutModel(outputInstance, inModelName, umlMetaModelName);
		
		IModel traceInstance = null;
		if (generateTraces) {
//			ModelFactory amwFactory = CoreService.createModelFactory("AMW");
			ModelFactory amwFactory = emfFactory;
			IInjector amwInjector = CoreService.getInjector(amwFactory.getDefaultInjectorName());
			IExtractor amwExtractor = CoreService.getExtractor(amwFactory.getDefaultExtractorName());
			
			URL mmwMMURL = this.getClass().getResource(
					this.mmLocation + "/mmw_traceability.ecore");

			// Chargement du mta modle AMW
			referenceModelOptions = new HashMap<String, Object>();
			referenceModelOptions.put("modelHandlerName", modelHandlers.get(amwMetaModelName)); //$NON-NLS-1$
			referenceModelOptions.put("modelName", amwMetaModelName); //$NON-NLS-1$
			referenceModelOptions.put("path", mmwMMURL.toString()); //$NON-NLS-1$
			IReferenceModel amwMM = amwFactory.newReferenceModel(referenceModelOptions);
			amwInjector.inject(amwMM, mmwMMURL.openStream(), Collections.EMPTY_MAP);
			// Chargement du modle AMW
			modelOptions = new HashMap<String, Object>();
			inModelName = "trace";
			modelOptions.put("modelName", inModelName); //$NON-NLS-1$
			modelOptions.put("path", UriUtils.toString(umlTargetModelUri.trimFileExtension().appendFileExtension("trace.amw"))); //$NON-NLS-1$
			modelOptions.put("newModel", true); //$NON-NLS-1$
			traceInstance = amwFactory.newModel(amwMM, modelOptions);
			launcher.addOutModel(traceInstance, inModelName, amwMetaModelName);
		}
		
		

		final Map<String, Object> options = new HashMap<String, Object>();
		options.put("continueAfterError", "true");
		options.put("printExecutionTime", "true");

		Job transformationThread = new Job("Convert KDM model to UML") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				IStatus result = Status.OK_STATUS;
				try {
					launcher.launch(ILauncher.RUN_MODE, monitor, options,
							transformation.openStream());
				} catch (IOException e) {
					result = Status.CANCEL_STATUS;
					e.printStackTrace();
				}

				return result;
			}
		};
		transformationThread.schedule();
		try {
			transformationThread.join();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		
		/*
		 * Pour les modles UML, l'application des profiles et strotypes se
		 * fait uniquement lors de la sauvegarde du modle (notamment pour
		 * viter de s'emmler les pinceaux avec l'ordre dans lequels ces
		 * oprations doivent tre effectues)
		 */
		umlExtractor.extract(outputInstance, UriUtils.toString(umlTargetModelUri));
		if (generateTraces) {
			emfExtractor.extract(traceInstance, UriUtils.toString(umlTargetModelUri.trimFileExtension().appendFileExtension("trace.amw")));
		}

		Resource output = null;
		if (outputInstance instanceof ASMModelWrapper) {
			ASMModelWrapper wrapper = (ASMModelWrapper) outputInstance;
			ASMModel model = wrapper.getAsmModel();
			if (model instanceof ASMEMFModel) {
				output = ((ASMEMFModel) model).getExtent();
			}
		}
		Resource outputTrace = null;
		if (generateTraces) {
			if (traceInstance instanceof ASMModelWrapper) {
				ASMModelWrapper wrapper = (ASMModelWrapper) traceInstance;
				ASMModel model = wrapper.getAsmModel();
				if (model instanceof ASMEMFModel) {
					outputTrace = ((ASMEMFModel) model).getExtent();
				}
			}
		}

		return new Resource[] { output, outputTrace };
	}

	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.kdm.uml2converter.KdmToUmlConverterInterface#exportKdmToUmlTransformation(org.eclipse.core.runtime.IPath)
	 */
	public void exportKdmToUmlTransformation(final IPath pathParameter) {
		final InputStream transfoFileStream = this.getClass()
				.getResourceAsStream(this.transfoLocation + "/KDMtoUML.atl");
		IPath path = pathParameter;
		if (path.getFileExtension() == null
				|| !path.getFileExtension().equals("atl")) {
			path = path.addFileExtension("atl");
		}
		final IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(
				path);
		final IProject destinationProject = file.getProject();
		Job job = new Job("exporting ATL transformation") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				try {
					/*
					 *  quel que soit la mthode utilise,
					 *   le flux transfoFileStream sera correctement ferm
					 */
					if (file.exists()) {
						file.setContents(transfoFileStream, IResource.FORCE,
								monitor);
					} else {
						file.create(transfoFileStream, IResource.FORCE, monitor);
					}
				} catch (CoreException e) {
					return Status.CANCEL_STATUS;
				}
				return Status.OK_STATUS;
			}
		};
		// when copy job is done, open the file
		job.addJobChangeListener(new JobChangeAdapter() {
			@Override
			public void done(IJobChangeEvent event) {
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						try {
							destinationProject.refreshLocal(
									IResource.DEPTH_INFINITE,
									new NullProgressMonitor());
							IWorkbenchPage page = PlatformUI.getWorkbench()
									.getActiveWorkbenchWindow().getActivePage();
							IDE.openEditor(page, file);
						} catch (Exception e) {
							e.printStackTrace();
						}
					}
				});
			}
		});
		job.schedule();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.kdm.uml2converter.KdmToUmlConverterInterface#instrumentAtlTranformationWithTraceability(java.io.InputStream, org.eclipse.emf.common.util.URI)
	 */
	public IFile instrumentAtlTranformationWithTraceability(
			InputStream sourceTransformation, URI atlFileUri) throws ATLCoreException {
		URI sourceModelUri = atlFileUri.trimFileExtension().appendFragment("-ATL_source").appendFileExtension("ecore");
		URI targetModelUri = atlFileUri.trimFileExtension().appendFragment("-ATL_target").appendFileExtension("ecore");
		
		// passage du fichier ATL au modle ATL
		IModel sourceModel = AtlParser.getDefault().parseToModel(sourceTransformation);
		if (sourceModel instanceof EMFModel) {
			EMFModel ecoreSourceModel = (EMFModel) sourceModel;
			Tools.saveModel(ecoreSourceModel.getResource(), sourceModelUri);
		}
		
		
		IModel targetModel = this.addTraceability(sourceModel, sourceModelUri, targetModelUri);
		
		if (targetModel instanceof EMFModel) {
			EMFModel ecoreTargetModel = (EMFModel) targetModel;
			Tools.saveModel(ecoreTargetModel.getResource(), targetModelUri);
		}
		IFile result = null;
		// passage du modle ATL au fichier ATL
		try {
			result = this.parseFromModel(targetModel, atlFileUri);
		} catch (IOException e) {
			e.printStackTrace();
		}

		return result;
	}

	private final IModel addTraceability(IModel sourceModel, URI sourceModelUri, URI targetModelUri) throws ATLCoreException {
		final URL transformation = this.getClass().getResource(this.transfoLocation + "/ATL2WTracer.asm");

		Map<String, String> modelHandlers = new HashMap<String, String>();
		String atlMetaModelName = "ATL";
		modelHandlers.put(atlMetaModelName, "EMF");
		
		final Map<String, Object> options = new HashMap<String, Object>();
		options.put(ATLLaunchConstants.OPTION_MODEL_HANDLER, modelHandlers);
		options.put(ATLLaunchConstants.IS_REFINING, true);
		
		String launcherName = ATLLaunchConstants.EMF_VM_NAME;
		final ILauncher launcher = CoreService.getLauncher(launcherName);
		launcher.initialize(options);
		
		ModelFactory factory = CoreService.createModelFactory(launcher.getDefaultModelFactoryName());

		IInjector injector = CoreService.getInjector(factory.getDefaultInjectorName());
		IExtractor extractor = CoreService.getExtractor(factory.getDefaultExtractorName());
		
		// chargement du mta modle ATL
		IReferenceModel atlMM = factory.getBuiltInResource("ATL.ecore");
		// chargement du modle source
		Map<String, Object> modelOptions = new HashMap<String, Object>();
		String inModelName = "IN";
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", sourceModelUri.toString()); //$NON-NLS-1$
		modelOptions.put("newModel", false); //$NON-NLS-1$
		IModel input = factory.newModel(atlMM, modelOptions);
		injector.inject(input, sourceModelUri.toString());
		launcher.addInModel(input, inModelName, atlMetaModelName);
		// chargement du modle cible
		modelOptions = new HashMap<String, Object>();
		inModelName = "OUT";
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", targetModelUri.toString()); //$NON-NLS-1$
		modelOptions.put("newModel", true); //$NON-NLS-1$
		IModel outputInstance = factory.newModel(atlMM, modelOptions);
		launcher.addOutModel(outputInstance, inModelName, atlMetaModelName);
		
		/*
		 * encapsulation de la transformation dans un nouveau Thread pour
		 * obtenir une pile d'excution vide.
		 */
		options.put("continueAfterError", "true");
		options.put("printExecutionTime", "true");

		Job transformationThread = new Job("Convert J2SE5 model to KDM") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				IStatus result = Status.OK_STATUS;
				try {
					launcher.launch(ILauncher.RUN_MODE, monitor, options,
							transformation.openStream());
				} catch (IOException e) {
					result = Status.CANCEL_STATUS;
					e.printStackTrace();
				}

				return result;
			}
		};
		transformationThread.schedule();
		try {
			transformationThread.join();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		
		extractor.extract(outputInstance, targetModelUri.toString());
		
		return outputInstance;
	}
	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.kdm.uml2converter.KdmToUmlConverterInterface#parseFromModel(org.eclipse.m2m.atl.engine.vm.nativelib.ASMModel, org.eclipse.emf.common.util.URI)
	 */
	public final IFile parseFromModel(final IModel transformationModel,
			URI atlFileUri) throws IOException {
		IFile file = null;
		OutputStream out = null;
		IPath atlFilePath = null;
		if (atlFileUri.isPlatformResource()) {
			atlFilePath = new Path(atlFileUri.toPlatformString(false));
			file = ResourcesPlugin.getWorkspace().getRoot().getFile(atlFilePath);
			PipedInputStream in = new PipedInputStream();
			out = new PipedOutputStream(in);
			try {
				AtlParser.getDefault().extract(transformationModel, out, null);
			} catch (Exception e) {
				e.printStackTrace();
			} finally {
				try {
					out.close();
				} catch (IOException ioe) {
				}
			}
			try {
				if (file.exists()) {
					file.setContents(in, IResource.FORCE, null);
				} else {
					file.create(in, IResource.FORCE, null);
				}
			} catch (CoreException e) {
				e.printStackTrace();
			}
		} else {
			atlFilePath = new Path(atlFileUri.toFileString());
			out = new FileOutputStream(atlFilePath.toFile());
			try {
				AtlParser.getDefault().extract(transformationModel, out, null);
			} catch (Exception e) {
				e.printStackTrace();
			} finally {
				try {
					out.close();
				} catch (IOException ioe) {
				}
			}
		}

		return file;
	}
}
