/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.adapters;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.provider.IEditingDomainItemProvider;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.emf.edit.provider.IItemPropertySource;
import org.eclipse.emf.edit.provider.IStructuredItemContentProvider;
import org.eclipse.emf.edit.provider.ITreeItemContentProvider;
import org.eclipse.gmt.modisco.common.editor.editors.EditorConfiguration;
import org.eclipse.gmt.modisco.common.editor.util.ImageProvider;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

/**
 * An item provider which is used for splitting big element lists into smaller ones, so as to avoid
 * locking the GUI for too long when the tree is built
 */
public class BigListItemProvider extends TransientItemProvider implements
		IEditingDomainItemProvider, IStructuredItemContentProvider, ITreeItemContentProvider,
		IItemLabelProvider, IItemPropertySource {

	/**
	 * The number of elements inside each range (which is also the number of elements over which the
	 * elements start getting split into ranges)
	 */
	private static int SPLIT_THRESHOLD = 1000;

	/** The number of elements above which a progress dialog is shown when sorting elements */
	private static final int SHOW_PROGRESS_DIALOG_THRESHOLD = 5000;

	/**
	 * The list of all elements, only a portion of which will be displayed by this
	 * BigListItemProvider
	 */
	private List<EObject> elements;
	/** The index of the first element to display (among all the elements) */
	private int startIndex;
	/** The index after the last element to display (among all the elements) */
	private int endIndex;
	/**
	 * The tree parent of the element the BigListItemProvider provides content for
	 */
	private Object parent;
	/** The model element parent of all the elements to display */
	private EObject modelParent;

	/** The sublist of items that are displayed by this {@link BigListItemProvider} */
	private List<EObject> subList = null;

	/**
	 * Create a BigListItemProvider
	 * 
	 * @param elements
	 *            the list of all elements, only a portion of which will be displayed by this
	 *            BigListItemProvider
	 * @param startIndex
	 *            the index of the first element to display (among all the elements)
	 * @param endIndex
	 *            the index after the last element to display (among all the elements)
	 * @param parent
	 *            the parent of the BigListItemProvider in the tree
	 * @param modelParent
	 *            the parent of all the elements
	 * @param adapterFactory
	 *            the adapter factory that is used to create adapters
	 * @param editorConfiguration
	 *            the configuration of the editor used to display the elements
	 */
	public BigListItemProvider(List<EObject> elements, int startIndex, int endIndex, Object parent,
			EObject modelParent, AdapterFactory adapterFactory,
			EditorConfiguration editorConfiguration) {
		super(adapterFactory, modelParent);
		this.parent = parent;
		this.modelParent = modelParent;
		// this.editorConfiguration = editorConfiguration;
		this.elements = elements;
		this.startIndex = startIndex;
		this.endIndex = endIndex;
	}

	@Override
	public Collection<?> getChildren(Object object) {
		if (this.subList == null) {
			int endIndex = this.endIndex;

			if (endIndex > this.elements.size()) {
				endIndex = this.elements.size();
			}

			if (this.startIndex > this.elements.size()) {
				this.subList = Collections.emptyList();
			} else {
				this.subList = this.elements.subList(this.startIndex, endIndex);
			}
		}
		return this.subList;
	}

	@Override
	public Object getParent(Object object) {
		return this.parent;
	}

	public EObject getModelParent() {
		return this.modelParent;
	}

	@Override
	public Collection<? extends EStructuralFeature> getChildrenFeatures(Object object) {
		if (this.parent instanceof LinkItemProvider) {
			// Return the feature corresponding to the link.
			// This is needed for commands to work on elements under BigListItemProvider.
			LinkItemProvider linkItemProvider = (LinkItemProvider) this.parent;
			return linkItemProvider.getChildrenFeatures(object);
		} else
			return super.getChildrenFeatures(object);
	}

	@Override
	public Object getImage(Object object) {
		return ImageProvider.getInstance().getRangeIcon();
	}

	/**
	 * Return the text used to describe the range to the user. The lower and upper bounds are padded
	 * with zeroes so that they get sorted correctly (when the user enables sorting)
	 */
	@Override
	public String getText(Object object) {
		// "[" + startIndex + ".." + (endIndex - 1) + "]";
		int nElements = this.elements.size();
		int maxDigits = (int) Math.ceil(Math.log10(nElements));

		StringBuilder buffer = new StringBuilder();
		buffer.append("[");

		String strStartIndex = "" + this.startIndex;
		String strEndIndex = "" + (this.endIndex - 1);

		zeroes(maxDigits - strStartIndex.length(), buffer);
		buffer.append(strStartIndex);
		buffer.append("..");
		zeroes(maxDigits - strEndIndex.length(), buffer);
		buffer.append(strEndIndex);

		buffer.append("]");

		return buffer.toString();
	}

	/** Append <code>count</code> number of zeroes to <code>buffer</code> */
	private void zeroes(int count, StringBuilder buffer) {
		for (int i = 0; i < count; i++) {
			buffer.append('0');
		}
	}

	@Override
	public Collection<?> getNewChildDescriptors(Object object, EditingDomain editingDomain,
			Object sibling) {
		// disable creating children from BigListItemProviders and siblings from
		// LinkItemProvider
		return Collections.emptyList();
	}

	/**
	 * If the number of elements is not too big (lower than
	 * {@link BigListItemProvider#SPLIT_THRESHOLD 1000}), then return the model elements. If it is
	 * big, then split all the elements into groups of {@link BigListItemProvider#SPLIT_THRESHOLD
	 * 1000} elements, each group being represented by a {@link BigListItemProvider}.
	 * <p>
	 * Also, associate the parent of each element through the {@link ParentAdapter}.
	 * 
	 * @param parent
	 *            the parent of the elements in the tree
	 * @param modelParent
	 *            the model element that is the logical parent of the given elements
	 * @param elements
	 *            the list of all the elements that must be split
	 * @param adapterFactory
	 *            the adapter factory that is used to adapt model elements
	 * @param editorConfiguration
	 *            the configuration of the editor displaying the elements
	 * @param previousElements
	 *            if not <code>null</code>, the list of elements that was returned last time. In
	 *            this case, the same {@link BigListItemProvider}s are reused. This collection
	 *            should be ordered.
	 * @return the new list of children: either the same elements list as was given, or a list of
	 *         {@link BigListItemProvider}s, partitioning the original elements into ranges.
	 */
	public static Collection<?> splitElements(Object parent, EObject modelParent,
			List<EObject> elements, final AdapterFactory adapterFactory,
			EditorConfiguration editorConfiguration, Collection<?> previousElements) {

		// associate a parent with each element through the ParentAdapter
		for (EObject element : elements) {
			ParentAdapter.associateParent(element, parent, editorConfiguration.getAdapterFactory());
		}

		/*
		 * sort the elements before they are split (this is necessary because otherwise the elements
		 * would get sorted inside each range, but not between ranges)
		 */
		if (editorConfiguration.isSortInstances()) {
			sortElements(elements, adapterFactory);
		}

		if (elements.size() <= SPLIT_THRESHOLD)
			return elements;
		else {
			Iterator<?> previousElementsIterator = null;
			boolean reuse = previousElements != null;

			if (reuse)
				previousElementsIterator = previousElements.iterator();

			ArrayList<BigListItemProvider> subLists = new ArrayList<BigListItemProvider>();
			for (int start = 0; start < elements.size(); start += SPLIT_THRESHOLD) {
				// start is inclusive, end is exclusive
				int end = start + SPLIT_THRESHOLD;
				if (end > elements.size()) {
					end = elements.size();
				}

				BigListItemProvider bigListItemProvider = null;

				/*
				 * Try to reuse an existing BigListItemProvider. This is done so that selection
				 * (which is based on the physical identity of the objects) can be restored properly
				 * in the tree viewer.
				 */
				if (reuse && previousElementsIterator.hasNext()) {
					Object obj = previousElementsIterator.next();
					if (obj instanceof BigListItemProvider) {
						bigListItemProvider = (BigListItemProvider) obj;
					} else {
						reuse = false;
					}
				}

				if (bigListItemProvider == null) {
					bigListItemProvider = new BigListItemProvider(elements, start, end, parent,
							modelParent, adapterFactory, editorConfiguration);
				} else {
					bigListItemProvider.setParent(modelParent);
					bigListItemProvider.elements = elements;
					bigListItemProvider.startIndex = start;
					bigListItemProvider.endIndex = end;
					bigListItemProvider.parent = parent;
					bigListItemProvider.modelParent = modelParent;
					bigListItemProvider.adapterFactory = adapterFactory;
					bigListItemProvider.subList = null;
				}

				subLists.add(bigListItemProvider);
			}
			return subLists;
		}
	}

	/**
	 * Sort the given list of {@link EObject}s alphabetically, by their displayed text. Displays a
	 * progress box, which permits cancellation to the user.
	 */
	private static void sortElements(final List<EObject> elements,
			final AdapterFactory adapterFactory) {

		IRunnableWithProgress sortOperation = new IRunnableWithProgress() {
			public void run(final IProgressMonitor monitor) {
				monitor.beginTask("Sorting instances", IProgressMonitor.UNKNOWN);
				Collections.sort(elements, new Comparator<EObject>() {
					int count = 0;

					public int compare(EObject e1, EObject e2) {
						// don't slow down the comparison by too frequent monitor calls
						if (this.count++ % 1000 == 0) {
							if (monitor.isCanceled()) {
								throw new OperationCanceledException();
							}
							// process events waiting on display queue
							// to let user click cancel button
							Display.getDefault().readAndDispatch();
						}

						int metaclassComparison = e1.eClass().getName().compareToIgnoreCase(
								e2.eClass().getName());
						if (metaclassComparison != 0) {
							return metaclassComparison;
						}

						String label1 = null;
						String label2 = null;

						IItemLabelProvider labelProvider1 = (IItemLabelProvider) adapterFactory
								.adapt(e1, IItemLabelProvider.class);
						IItemLabelProvider labelProvider2 = (IItemLabelProvider) adapterFactory
								.adapt(e2, IItemLabelProvider.class);

						if (labelProvider1 instanceof MiaReflectiveItemProvider) {
							MiaReflectiveItemProvider itemProvider = (MiaReflectiveItemProvider) labelProvider1;
							label1 = itemProvider.getName(e1);
						}
						if (labelProvider2 instanceof MiaReflectiveItemProvider) {
							MiaReflectiveItemProvider itemProvider = (MiaReflectiveItemProvider) labelProvider2;
							label2 = itemProvider.getName(e2);
						}

						if (label1 == null) {
							label1 = labelProvider1.getText(e1);
						}

						if (label2 == null) {
							label2 = labelProvider2.getText(e2);
						}

						return label1.compareToIgnoreCase(label2);
					}
				});
			}
		};

		IWorkbench workbench = PlatformUI.getWorkbench();
		IWorkbenchWindow window = workbench.getActiveWorkbenchWindow();
		Shell shell = window != null ? window.getShell() : null;
		ProgressMonitorDialog progressMonitorDialog = new ProgressMonitorDialog(shell);

		try {
			progressMonitorDialog.setOpenOnRun(elements.size() > SHOW_PROGRESS_DIALOG_THRESHOLD);
			progressMonitorDialog.run(false, true, sortOperation);
		} catch (InvocationTargetException e) {
		} catch (InterruptedException e) {
		} catch (OperationCanceledException e) {
			// sorting canceled by user
		}
	}

}
