/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.adapters;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.UnexecutableCommand;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.NotificationWrapper;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.edit.command.AddCommand;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.provider.IEditingDomainItemProvider;
import org.eclipse.emf.edit.provider.IItemColorProvider;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.emf.edit.provider.IItemPropertyDescriptor;
import org.eclipse.emf.edit.provider.IItemPropertySource;
import org.eclipse.emf.edit.provider.IStructuredItemContentProvider;
import org.eclipse.emf.edit.provider.ITreeItemContentProvider;
import org.eclipse.emf.edit.provider.ItemPropertyDescriptorDecorator;
import org.eclipse.emf.edit.provider.ViewerNotification;
import org.eclipse.gmt.modisco.common.editor.editors.EditorConfiguration;
import org.eclipse.gmt.modisco.common.editor.util.ImageProvider;

/**
 * An item provider for displaying in the tree virtual elements representing links between two
 * elements in the model.
 */
public class LinkItemProvider extends TransientItemProvider implements IEditingDomainItemProvider,
		IStructuredItemContentProvider, ITreeItemContentProvider, IItemLabelProvider,
		IItemPropertySource, IItemColorProvider {

	/** The reference corresponding to the link */
	protected EReference reference;
	/** The model element that owns the link */
	protected EObject parent;
	/** The configuration of the editor in which the model is displayed */
	protected final EditorConfiguration editorConfiguration;

	/**
	 * Instantiate a new LinkItemProvider.
	 * 
	 * @param adapterFactory
	 *            the adapter factory used to create adapters
	 * @param parent
	 *            the model element that owns the link
	 * @param reference
	 *            the reference corresponding to the link
	 * @param editorConfiguration
	 *            the configuration of the editor in which the model is displayed
	 */
	public LinkItemProvider(AdapterFactory adapterFactory, EObject parent, EReference reference,
			EditorConfiguration editorConfiguration) {
		super(adapterFactory, parent);
		this.reference = reference;
		this.parent = parent;
		this.editorConfiguration = editorConfiguration;
		parent.eAdapters().add(this);
	}

	/**
	 * @return the reference corresponding to the link
	 */
	public EReference getReference() {
		return this.reference;
	}

	@Override
	public Collection<? extends EStructuralFeature> getChildrenFeatures(Object object) {
		if (this.childrenFeatures == null) {
			super.getChildrenFeatures(object);
			this.childrenFeatures.add(this.reference);
		}
		return this.childrenFeatures;
	}

	@Override
	public void notifyChanged(Notification notification) {
		// fire the notification only if it matches the reference
		if (this.reference == notification.getFeature()) {
			// update the label of the parent (to update link number)
			fireNotifyChanged(new ViewerNotification(notification, this.parent, false, true));

			/*
			 * Huge performance penalty if all the notifications are fired, because the viewer is
			 * refreshed for each command in a batch. So, fire only the notifications that are
			 * mandatory to keep a coherent state.
			 */
			if (notification.getEventType() == Notification.SET) {
				fireNotifyChanged(new NotificationWrapper(this, notification));
			}
		}
	}

	/** @return the number of elements referenced by this link */
	public int getCount() {
		int count;
		Object ref = this.parent.eGet(this.reference);
		if (this.reference.isMany()) {
			count = ((List<?>) ref).size();
		} else {
			count = (ref != null) ? 1 : 0;
		}
		return count;
	}

	@Override
	public String getText(Object object) {

		String multiplicity = "";

		if (this.editorConfiguration.isShowMultiplicity()) {
			multiplicity = getMultiplicity(this.reference);
		}

		String opposite = "";

		if (this.editorConfiguration.isShowOppositeLinks()) {
			EReference oppositeRef = this.reference.getEOpposite();
			if (oppositeRef != null) {
				String oppositeMultiplicity = "";
				if (this.editorConfiguration.isShowMultiplicity()) {
					oppositeMultiplicity = getMultiplicity(oppositeRef);
				}

				opposite = " <-> " + oppositeRef.getName() + oppositeMultiplicity;
			}
		}

		String prefix = (this.reference.isDerived() ? "/" : "");
		return prefix + this.reference.getName() + multiplicity + opposite + " (" + getCount()
				+ ")";
	}

	public static String getMultiplicity(EReference reference) {
		String multLow = (reference.isRequired() ? "1" : "0");
		String multHigh = (reference.isMany() ? "*" : "1");
		return " [" + multLow + ".." + multHigh + "]";
	}

	@Override
	protected void collectNewChildDescriptors(Collection<Object> newChildDescriptors, Object object) {
		super.collectNewChildDescriptors(newChildDescriptors, object);

		// can only create children if it is a composition link
		if (!this.reference.isContainment())
			return;

		Collection<EClass> allClasses = this.editorConfiguration.getAllClasses();

		EFactory factory = this.reference.getEReferenceType().getEPackage().getEFactoryInstance();
		EClass referenceType = this.reference.getEReferenceType();

		// return sub-classes of the reference type
		for (EClass aClass : allClasses) {
			if (!aClass.isAbstract() && referenceType.isSuperTypeOf(aClass)) {
				try {
					Object newInstance = factory.create(aClass);
					newChildDescriptors.add(createChildParameter(this.reference, newInstance));
				} catch (IllegalArgumentException e) {
					// not a legal class for this factory
				}
			}
		}
	}

	// @Override
	// public Command createCommand(Object object, EditingDomain domain,
	// Class<? extends Command> commandClass, CommandParameter commandParameter) {
	// String command = commandClass.toString();
	// System.out.println("LinkItemProvider " + command.substring(command.lastIndexOf('.') + 1));
	//
	// return super.createCommand(object, domain, commandClass, commandParameter);
	// }

	@Override
	protected Command createDragAndDropCommand(EditingDomain domain, Object owner, float location,
			int operations, int operation, Collection<?> collection) {
		// test if the command can be executed
		if (new AddCommand(domain, (EObject) owner, this.reference, collection).canExecute())
			return super.createDragAndDropCommand(domain, owner, location, operations, operation,
					collection);
		else
			return UnexecutableCommand.INSTANCE;
	}

	@Override
	public Object getImage(Object object) {
		return getImageFor(this.reference);
	}

	public static Object getImageFor(EReference reference) {
		EReference opposite = reference.getEOpposite();

		if (reference.isContainment()) {
			if (opposite == null)
				return ImageProvider.getInstance().getUnidirectionalAggregIcon();
			else
				return ImageProvider.getInstance().getAggregIcon();
		} else {
			if (opposite != null && opposite.isContainment())
				return ImageProvider.getInstance().getInvAggregIcon();
		}

		if (opposite == null)
			return ImageProvider.getInstance().getUnidirectionalLinkIcon();
		else
			return ImageProvider.getInstance().getLinkIcon();
	}

	@Override
	public Object getParent(Object object) {
		return this.parent;
	}

	@Override
	public Collection<?> getChildren(Object object) {
		@SuppressWarnings("unchecked")
		Collection<EObject> children = (Collection<EObject>) super.getChildren(this.parent);
		List<EObject> childrenList = new ArrayList<EObject>();
		childrenList.addAll(children);

		// partition the elements if there are too many elements under this link
		return BigListItemProvider.splitElements(this, this.parent, childrenList,
				this.adapterFactory, this.editorConfiguration, null);
	}

	/** Return the list of children, without any partitioning */
	public Collection<EObject> getChildrenElements() {
		@SuppressWarnings("unchecked")
		Collection<EObject> children = (Collection<EObject>) super.getChildren(this.parent);
		return children;
	}

	@Override
	protected String getFeatureText(Object feature) {
		return this.reference.getName();
	}

	/**
	 * Return property descriptors so that the feature corresponding to the selected link appears in
	 * the Properties view.
	 */
	@Override
	public List<IItemPropertyDescriptor> getPropertyDescriptors(Object object) {
		if (this.itemPropertyDescriptors == null) {
			this.itemPropertyDescriptors = new ArrayList<IItemPropertyDescriptor>();

			// get the property descriptors from the parent
			MiaReflectiveItemProvider parentItemProvider = (MiaReflectiveItemProvider) this.adapterFactory
					.adapt(this.parent, IItemPropertySource.class);

			List<IItemPropertyDescriptor> parentDescriptors = parentItemProvider
					.getPropertyDescriptors(this.parent);

			for (IItemPropertyDescriptor descriptor : parentDescriptors) {
				// if the parent's descriptor matches the link's reference, then add it
				if (descriptor.getFeature(this.parent) == this.reference) {
					this.itemPropertyDescriptors.add(new ItemPropertyDescriptorDecorator(
							this.parent, descriptor) {
						@Override
						public String getCategory(Object thisObject) {
							// no category (show at root)
							return null;
						}

						@Override
						public String getId(Object thisObject) {
							return LinkItemProvider.this.reference.getName()
									+ getDisplayName(thisObject);
						}

					});
				}
			}
		}
		return this.itemPropertyDescriptors;
	}

	@Override
	public Collection<?> getNewChildDescriptors(Object object, EditingDomain editingDomain,
			Object sibling) {
		// do not create siblings from BigListItemProvider
		if (sibling instanceof BigListItemProvider)
			return Collections.emptyList();
		else
			return super.getNewChildDescriptors(object, editingDomain, sibling);
	}

	@Override
	public Object getForeground(Object object) {
		if (getCount() == 0)
			return IItemColorProvider.GRAYED_OUT_COLOR;
		return super.getForeground(object);
	}

	@Override
	public Object getCreateChildImage(Object owner, Object feature, Object child,
			Collection<?> selection) {
		/*
		 * Disable children images to plug a leak where images hang to the LinkItemProvider, and
		 * cause the whole model to stay in memory after the editor is closed. This is because an
		 * anonymous class is created, and it acts like a closure by capturing its environment.
		 */
		return null;
	}

}