/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.adapters;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.UnexecutableCommand;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.edit.command.CommandParameter;
import org.eclipse.emf.edit.command.RemoveCommand;
import org.eclipse.emf.edit.command.SetCommand;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.provider.IEditingDomainItemProvider;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.emf.edit.provider.IStructuredItemContentProvider;
import org.eclipse.emf.edit.provider.ItemProviderAdapter;
import org.eclipse.gmt.modisco.common.editor.core.InstancesForMetaclass;
import org.eclipse.gmt.modisco.common.editor.core.InstancesForMetaclasses;
import org.eclipse.gmt.modisco.common.editor.editors.EditorConfiguration;

/**
 * An item provider adapter for displaying in the model editor tree the list of instances of one or
 * more metaclasses selected in the list of metaclasses.
 */
public class MetaclassInstancesItemProvider extends ItemProviderAdapter implements
		IStructuredItemContentProvider, IItemLabelProvider, IEditingDomainItemProvider {

	/**
	 * The configuration of the editor used to display the list of metaclasses and the tree of model
	 * elements
	 */
	private final EditorConfiguration editorConfiguration;
	/** The list of qualified names of the metaclasses whose elements must be displayed */
	private final String[] metaclasses;
	private InstancesForMetaclasses.ModelChangeListener modelChangeListener;

	public MetaclassInstancesItemProvider(AdapterFactory adapterFactory,
			EditorConfiguration editorConfiguration, final String[] metaclasses) {
		super(adapterFactory);
		this.editorConfiguration = editorConfiguration;
		this.metaclasses = metaclasses;

		InstancesForMetaclasses instancesForMetaclasses = this.editorConfiguration
				.getInstancesForMetaclasses();

		this.modelChangeListener = new InstancesForMetaclasses.ModelChangeListener() {
			public void modelChanged() {
				MetaclassInstancesItemProvider.this.markDirty();
			}

			public void removedLastInstanceof(String metaclassQualifiedName) {
			}
		};

		instancesForMetaclasses.addListener(this.modelChangeListener);
	}

	/** Remove the listener when the element is not used anymore */
	@Override
	public void dispose() {
		this.editorConfiguration.getInstancesForMetaclasses().removeListener(
				this.modelChangeListener);
	}

	/**
	 * The elements are cached, so that the same elements are returned each time the getElements
	 * method is called, as long as the list of elements is unchanged. This is needed because the
	 * JFace viewers' selection system relies on the physical identity of the objects.
	 */
	private Collection<?> cachedElements = null;
	/** Whether the elements have to be rebuilt */
	private boolean dirty;

	/** Invalidate the cache */
	private void markDirty() {
		this.dirty = true;
	}

	private boolean oldSortInstances;

	@Override
	public Collection<?> getElements(Object object) {
		// Invalidate cache so that elements are recomputed
		if (this.editorConfiguration.isSortInstances() != this.oldSortInstances) {
			markDirty();
		}
		this.oldSortInstances = this.editorConfiguration.isSortInstances();

		if (this.cachedElements == null || this.dirty) {

			ArrayList<EObject> allElements = new ArrayList<EObject>();

			InstancesForMetaclasses instancesForMetaclasses = this.editorConfiguration
					.getInstancesForMetaclasses();
			for (String metaclassQualifiedName : this.metaclasses) {
				InstancesForMetaclass instancesForMetaclass = instancesForMetaclasses
						.getInstancesForMetaclass(metaclassQualifiedName);
				ArrayList<EObject> elements = instancesForMetaclass.getElements();
				allElements.addAll(elements);

				// also display instances of subclasses
				if (this.editorConfiguration.isDisplayInstancesOfSubclasses()) {
					allElements.addAll(instancesForMetaclass.getDerivedElements());
				}
			}

			this.cachedElements = BigListItemProvider.splitElements(this, null, allElements,
					this.adapterFactory, this.editorConfiguration, this.cachedElements);
			this.dirty = false;
		}
		return this.cachedElements;
	}

	@Override
	public Object getImage(Object object) {
		return null;
	}

	@Override
	public String getText(Object object) {
		// this is never displayed, since this element is the root
		return "Instances";
	}

	/**
	 * This method is overridden so that it is possible to delete elements that are children of a
	 * {@link MetaclassInstancesItemProvider}.
	 */
	@Override
	public Command createCommand(Object object, EditingDomain domain,
			Class<? extends Command> commandClass, CommandParameter commandParameter) {

		// the real owner of the model objects impacted by the command
		EObject owner = null;
		// the feature that contains the model objects impacted by the command
		EStructuralFeature containingFeature = null;

		Collection<?> collection = commandParameter.getCollection();
		for (Object o : collection) {
			if (o instanceof EObject) {
				EObject eObject = (EObject) o;
				// all the objects don't have the same container => cannot create command
				if (owner != null && owner != eObject.eContainer() || containingFeature != null
						&& containingFeature != eObject.eContainingFeature())
					return UnexecutableCommand.INSTANCE;
				owner = eObject.eContainer();
				containingFeature = eObject.eContainingFeature();
			}
		}

		if (owner == null || containingFeature == null)
			return UnexecutableCommand.INSTANCE;

		// remove command only works on multiplicity many features
		if (commandClass == RemoveCommand.class && !containingFeature.isMany()) {
			return new SetCommand(domain, owner, containingFeature, commandParameter.value);
		}

		commandParameter.setOwner(owner);
		commandParameter.feature = containingFeature;
		return super.createCommand(owner, domain, commandClass, commandParameter);
	}

	@Override
	public Collection<?> getNewChildDescriptors(Object object, EditingDomain editingDomain,
			Object sibling) {
		return Collections.emptyList();
	}

	@Override
	public Collection<?> getChildren(Object object) {
		return Collections.emptyList();
	}

	@Override
	public Object getParent(Object object) {
		return null;
	}
}
