/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.adapters;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;

import org.eclipse.core.runtime.Assert;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CommandWrapper;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.edit.command.CommandParameter;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.provider.IEditingDomainItemProvider;
import org.eclipse.emf.edit.provider.IItemColorProvider;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.emf.edit.provider.IItemPropertySource;
import org.eclipse.emf.edit.provider.IStructuredItemContentProvider;
import org.eclipse.emf.edit.provider.ITreeItemContentProvider;
import org.eclipse.emf.edit.provider.ReflectiveItemProvider;
import org.eclipse.emf.edit.ui.provider.ExtendedImageRegistry;
import org.eclipse.gmt.modisco.common.editor.MoDiscoEditorPlugin;
import org.eclipse.gmt.modisco.common.editor.editors.EditorConfiguration;
import org.eclipse.gmt.modisco.common.editor.extensions.icons.IconProvidersRegistry;
import org.eclipse.gmt.modisco.common.editor.extensions.naming.NameProvidersRegistry;
import org.eclipse.gmt.modisco.common.editor.util.EMFUtil;
import org.eclipse.gmt.modisco.common.editor.util.Util;
import org.eclipse.swt.graphics.Image;

/**
 * An item provider adapter for model elements, which uses reflection on the Ecore metamodel
 */
public class MiaReflectiveItemProvider extends ReflectiveItemProvider implements
		IEditingDomainItemProvider, IStructuredItemContentProvider, ITreeItemContentProvider,
		IItemLabelProvider, IItemPropertySource, IItemColorProvider {

	/** The color used for elements that are not part of the first resource */
	private URI EXTERNAL_COLOR = URI.createURI("color://rgb/0/0/255");
	/**
	 * The color used for elements that are not in any resource (actually, these elements should
	 * never appear to the user)
	 */
	private URI NULL_RESOURCE_COLOR = URI.createURI("color://rgb/192/0/0");

	/**
	 * The configuration of the editor used to display the list of metaclasses and the tree of model
	 * elements
	 */
	private final EditorConfiguration editorConfiguration;

	public MiaReflectiveItemProvider(AdapterFactory adapterFactory,
			EditorConfiguration editorConfiguration) {
		super(adapterFactory);
		this.editorConfiguration = editorConfiguration;
	}

	/**
	 * Cache the link item providers so that the same {@link LinkItemProvider} objects are always
	 * used. This allows the viewers to restore the selection (using the physical identity of the
	 * objects)
	 */
	private HashMap<EReference, LinkItemProvider> linkItemProviders = new HashMap<EReference, LinkItemProvider>();

	/**
	 * Cache the attribute item providers so that the same {@link AttributeItemProvider} objects are
	 * always used. This allows the viewers to restore the selection (using the physical identity of
	 * the objects)
	 */
	private HashMap<EAttribute, AttributeItemProvider> attributeItemProviders = new HashMap<EAttribute, AttributeItemProvider>();

	/**
	 * Cache the {@link ContainmentLinkItemProvider} so that the same physical object is always
	 * used.
	 */
	private ContainmentLinkItemProvider containmentLinkItemProvider = null;
	/** The container associated with the cached {@link ContainmentLinkItemProvider} */
	private EObject oldContainer = null;

	@Override
	public Collection<?> getChildren(Object object) {

		Assert.isTrue(object == this.target);

		if (object instanceof EObject) {
			EObject eObject = (EObject) object;

			ArrayList<TransientItemProvider> children = new ArrayList<TransientItemProvider>();

			// show a link for the container
			if (this.editorConfiguration.isShowContainer()) {
				addContainer(eObject, children);
			}

			if (this.editorConfiguration.isShowAttributes()) {
				ArrayList<AttributeItemProvider> attributes = createAttributes(eObject);
				// sort attributes by name
				// TODO: separate preference for attributes?
				if (this.editorConfiguration.isSortLinks()) {
					sortAttributes(attributes);
				}
				children.addAll(attributes);
			}

			ArrayList<LinkItemProvider> links = createLinks(eObject);
			// sort links by name
			if (this.editorConfiguration.isSortLinks()) {
				sortLinks(links);
			}
			if (this.editorConfiguration.isSortLinksByType()) {
				// counting on the fact that sorting preserves the order of equal elements
				sortLinksByType(links);
			}
			children.addAll(links);

			return children;
		}

		return super.getChildren(object);
	}

	/**
	 * Create attribute elements corresponding to model attributes for the given {@link EObject}
	 * 
	 * @return the list of attributes elements created for representing the attributes visually in
	 *         the model editor
	 */
	private ArrayList<AttributeItemProvider> createAttributes(EObject eObject) {
		ArrayList<AttributeItemProvider> attributes = new ArrayList<AttributeItemProvider>();
		/*
		 * For each attribute, create an AttributeItemProvider that will represent the attribute as
		 * an element under the element containing the attribute
		 */
		EList<EAttribute> allAttributes = eObject.eClass().getEAllAttributes();
		for (EAttribute attribute : allAttributes) {

			boolean empty = false;
			if (!this.editorConfiguration.isShowEmptyAttributes()) {
				Object value = ((EObject) this.target).eGet(attribute);
				if (attribute.isMany()) {
					empty = ((List<?>) value).size() == 0;
				} else {
					empty = value == null;
				}
			}

			if (this.editorConfiguration.isShowEmptyAttributes() || !empty) {
				// try to get an already existing AttributeItemProvider from the cache
				AttributeItemProvider attributeItemProvider = this.attributeItemProviders
						.get(attribute);
				if (attributeItemProvider == null) {
					attributeItemProvider = new AttributeItemProvider(this.adapterFactory, eObject,
							attribute, this.editorConfiguration);
					this.attributeItemProviders.put(attribute, attributeItemProvider);
				}

				attributes.add(attributeItemProvider);
			}
		}
		return attributes;
	}

	/**
	 * Create the links corresponding to references for the given {@link EObject}
	 * 
	 * @return the list of links created for representing the references in the model editor
	 */
	private ArrayList<LinkItemProvider> createLinks(EObject eObject) {
		EList<EReference> allReferences = eObject.eClass().getEAllReferences();

		ArrayList<LinkItemProvider> links = new ArrayList<LinkItemProvider>();

		/*
		 * For each reference, create a LinkItemProvider that will represent the reference visually
		 */
		for (EReference reference : allReferences) {

			// filtered out by the user?
			if (reference.isDerived() && !this.editorConfiguration.isShowDerivedLinks()) {
				continue;
			}

			// get the object(s) referenced
			Object ref = eObject.eGet(reference);

			// filtered out by the user?
			if (!this.editorConfiguration.isShowEmptyLinks()) {
				if (ref == null) {
					continue;
				}
				if (ref instanceof EList<?>) {
					EList<?> refList = (EList<?>) ref;
					if (refList.size() == 0) {
						continue;
					}
				}
			}

			// try to get an already existing LinkItemProvider from the cache
			LinkItemProvider linkItemProvider = this.linkItemProviders.get(reference);
			if (linkItemProvider == null) {
				linkItemProvider = new LinkItemProvider(this.adapterFactory, eObject, reference,
						this.editorConfiguration);
				this.linkItemProviders.put(reference, linkItemProvider);
			}

			links.add(linkItemProvider);
		}
		return links;
	}

	/** Adds the container of the given {@link EObject} to the children list */
	private void addContainer(EObject eObject, ArrayList<TransientItemProvider> children) {
		if (eObject.eContainer() != null) {

			if (this.containmentLinkItemProvider == null
					|| this.oldContainer != eObject.eContainer()) {
				this.containmentLinkItemProvider = new ContainmentLinkItemProvider(
						this.adapterFactory, eObject, eObject.eContainer(),
						this.editorConfiguration);
				this.oldContainer = eObject.eContainer();
			}

			// associate a parent with the element under the link
			ParentAdapter.associateParent(eObject.eContainer(), this.containmentLinkItemProvider,
					this.editorConfiguration.getAdapterFactory());
			children.add(this.containmentLinkItemProvider);
		} else if (this.editorConfiguration.isShowEmptyLinks()) {
			// create a ContainmentLinkItemProvider for an empty link
			this.containmentLinkItemProvider = new ContainmentLinkItemProvider(this.adapterFactory,
					eObject, null, this.editorConfiguration);
			this.oldContainer = null;
			children.add(this.containmentLinkItemProvider);
		}
	}

	/** Sort links by name */
	private void sortLinks(List<LinkItemProvider> links) {
		Collections.sort(links, new Comparator<LinkItemProvider>() {
			public int compare(LinkItemProvider o1, LinkItemProvider o2) {
				String name1 = o1.getReference().getName();
				String name2 = o2.getReference().getName();
				return name1.compareTo(name2);
			}
		});
	}

	/** Sort links by type */
	private void sortLinksByType(List<LinkItemProvider> links) {
		Collections.sort(links, new Comparator<LinkItemProvider>() {
			public int compare(LinkItemProvider o1, LinkItemProvider o2) {
				int r1 = rank(o1);
				int r2 = rank(o2);
				return r1 - r2;
			}

			private int rank(LinkItemProvider linkItemProvider) {
				EReference reference = linkItemProvider.getReference();
				EReference opposite = reference.getEOpposite();
				int rank;

				if (reference.isContainment()) {
					if (opposite != null)
						rank = 0;
					else
						rank = 10;
				} else {
					if (opposite != null) {
						if (opposite.isContainment())
							rank = 20;
						else
							rank = 30;
					} else {
						rank = 40;
					}
				}

				if (reference.isDerived())
					rank++;

				return rank;
			}
		});
	}

	/** Sort attributes by name */
	private void sortAttributes(List<AttributeItemProvider> attributes) {
		Collections.sort(attributes, new Comparator<AttributeItemProvider>() {
			public int compare(AttributeItemProvider o1, AttributeItemProvider o2) {
				String name1 = o1.getAttribute().getName();
				String name2 = o2.getAttribute().getName();
				return name1.compareTo(name2);
			}
		});
	}

	@Override
	public String getText(Object object) {
		EObject eObject = (EObject) object;
		EClass eClass = eObject.eClass();

		String label;
		if (this.editorConfiguration.isShowFullQualifiedNames()) {
			label = EMFUtil.getMetaclassQualifiedName(eClass);
		} else {
			label = eClass.getName();
		}

		return "[" + label + "] " + getName(eObject);
	}

	/** Return just the name that will be displayed (without the metaclass) */
	public String getName(EObject eObject) {

		String providedName = getProvidedName(eObject);
		if (providedName != null)
			return providedName;

		String nameFromRegistry = getNameFromRegistry(eObject);
		if (nameFromRegistry != null) {
			return nameFromRegistry;
		}

		return getDefaultName(eObject);
	}

	/**
	 * @return a name for the given {@link EObject} provided by an adapter registered in the
	 *         registry
	 */
	public String getNameFromRegistry(EObject eObject) {
		IItemLabelProvider itemLabelProvider = (IItemLabelProvider) this.editorConfiguration
				.getAdapterFactoryWithRegistry().adapt(eObject, IItemLabelProvider.class);

		if (itemLabelProvider != null && itemLabelProvider != this) {
			return itemLabelProvider.getText(eObject);
		}
		return null;
	}

	/**
	 * @return the name that was provided through a registered name provider for the given
	 *         {@link EObject}
	 */
	public String getProvidedName(EObject eObject) {
		NameProvidersRegistry nameProvidersRegistry = NameProvidersRegistry.getInstance();
		String name = nameProvidersRegistry.getName(eObject);
		if (name != null)
			return Util.truncateBeforeNewline(name);
		return null;
	}

	/** @return a default name based on a string feature of the given {@link EObject} */
	public String getDefaultName(EObject eObject) {
		// find a feature that can be used as a name
		EStructuralFeature feature = getLabelFeature(eObject.eClass());
		if (feature != null) {
			Object value = eObject.eGet(feature);
			if (value != null)
				return Util.truncateBeforeNewline(value.toString());
		}
		return "";
	}

	@Override
	public Object getParent(Object object) {

		/*
		 * All the model elements should have been set an explicit parent through the
		 * "parent adapter".
		 */
		Object adapter = ParentAdapterFactory.getInstance().adapt(object, ParentAdapter.class);
		if ((adapter instanceof ParentAdapter)) {
			ParentAdapter parentAdapter = (ParentAdapter) adapter;
			Object parent = parentAdapter.getParent();
			return parent;
		}

		MoDiscoEditorPlugin.INSTANCE.log("ParentAdapter not found");
		return null;
	}

	/**
	 * Inhibit the creation of children from the parent (must create from links)
	 */
	@Override
	protected void collectNewChildDescriptors(Collection<Object> newChildDescriptors, Object object) {
		// do nothing
	}

	/**
	 * Modify the command so that the parameters are model elements (otherwise, we would get a
	 * {@link ClassCastException} from EMF).
	 */
	@Override
	public Command createCommand(Object object, EditingDomain domain,
			Class<? extends Command> commandClass, CommandParameter commandParameter) {
		CommandParameter localCommandParameter = commandParameter;
		// String command = commandClass.toString();
		// System.out.println("MiaReflectiveItemProvider " +
		// command.substring(command.lastIndexOf('.') + 1));

		Collection<?> oldCollection = localCommandParameter.getCollection();
		Collection<Object> newCollection = new ArrayList<Object>();
		if (oldCollection != null) {
			for (Object o : oldCollection) {
				if (o instanceof LinkItemProvider) {
					LinkItemProvider linkItemProvider = (LinkItemProvider) o;
					newCollection.add(linkItemProvider.getParent(null));
				} else if (o instanceof AttributeItemProvider) {
					AttributeItemProvider attributeItemProvider = (AttributeItemProvider) o;
					newCollection.add(attributeItemProvider.getParent(null));
				} else if (o instanceof ContainmentLinkItemProvider) {
					ContainmentLinkItemProvider containmentLinkItemProvider = (ContainmentLinkItemProvider) o;
					newCollection.add(containmentLinkItemProvider.getParent(null));
				} else if (o instanceof BigListItemProvider) {
					BigListItemProvider bigListItemProvider = (BigListItemProvider) o;
					newCollection.add(bigListItemProvider.getModelParent());
				} else if (o instanceof EObject) {
					newCollection.add(o);
				}
			}

			localCommandParameter = new CommandParameter(localCommandParameter.getOwner(),
					localCommandParameter.getFeature(), localCommandParameter.getValue(),
					newCollection, localCommandParameter.getIndex());
		}

		return super.createCommand(object, domain, commandClass, localCommandParameter);
	}

	/** Wrap the command to redefine the affected objects */
	@Override
	protected Command createRemoveCommand(EditingDomain domain, EObject owner,
			EStructuralFeature feature, Collection<?> collection) {
		return createWrappedCommand(super.createRemoveCommand(domain, owner, feature, collection),
				owner);
	}

	/** Wrap the command to redefine the affected objects */
	@Override
	protected Command createAddCommand(EditingDomain domain, EObject owner,
			EStructuralFeature feature, Collection<?> collection, int index) {
		return createWrappedCommand(super.createAddCommand(domain, owner, feature, collection,
				index), owner);
	}

	/** Wrap the command to redefine the affected objects */
	protected Command createWrappedCommand(Command command, final EObject owner) {

		return new CommandWrapper(command) {
			@Override
			public Collection<?> getAffectedObjects() {
				Collection<?> affected = super.getAffectedObjects();
				if (affected.contains(owner)) {
					// must return an EObject
					affected = Collections.singleton(owner.eContainer());
				}
				return affected;
			}
		};
	}

	@Override
	public Collection<?> getNewChildDescriptors(Object object, EditingDomain editingDomain,
			Object sibling) {
		// do not create siblings from links (confusing for the user)
		if (sibling instanceof LinkItemProvider /* => && sibling != null */)
			return Collections.emptyList();
		else
			return super.getNewChildDescriptors(object, editingDomain, sibling);
	}

	@Override
	public boolean hasChildren(Object object) {
		/*
		 * Override to always return true so that there is no need to compute all children. Side
		 * effect: ghost (+) in the tree which disappears when it is clicked if the element doesn't
		 * have children. The alternative is to compute all children to determine whether there are
		 * any, which is too costly.
		 */
		return true;
	}

	@Override
	public Object getForeground(Object object) {
		/* Show the EObject in a different color if it is not part of the first resource */
		if (object instanceof EObject) {
			EObject eObject = (EObject) object;
			if (eObject.eResource() == null)
				return this.NULL_RESOURCE_COLOR;

			if (!EMFUtil.isInFirstResource(eObject))
				return this.EXTERNAL_COLOR;
		}

		return super.getForeground(object);
	}

	@Override
	public Object getImage(Object object) {

		if (object instanceof EObject) {
			EObject eObject = (EObject) object;

			IconProvidersRegistry iconProvidersRegistry = IconProvidersRegistry.getInstance();
			Image icon = iconProvidersRegistry.getIcon(eObject);
			if (icon != null)
				return icon;
		}

		// See if an image is provided by an adapter factory from the registry
		IItemLabelProvider itemLabelProvider = (IItemLabelProvider) this.editorConfiguration
				.getAdapterFactoryWithRegistry().adapt(object, IItemLabelProvider.class);

		if (itemLabelProvider != null && itemLabelProvider != this) {
			Object image = itemLabelProvider.getImage(object);
			return ExtendedImageRegistry.getInstance().getImage(image);
		}

		return super.getImage(object);
	}

}