/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.adapters;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CommandWrapper;
import org.eclipse.emf.common.command.UnexecutableCommand;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.edit.command.CommandParameter;
import org.eclipse.emf.edit.command.CopyCommand;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.provider.IEditingDomainItemProvider;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.emf.edit.provider.IItemPropertySource;
import org.eclipse.emf.edit.provider.IStructuredItemContentProvider;
import org.eclipse.emf.edit.provider.ITreeItemContentProvider;
import org.eclipse.emf.edit.provider.ItemProviderAdapter;

/**
 * An item provider for "transient" elements. That is, elements which are not model elements, and
 * are here only for display purposes.
 */
public abstract class TransientItemProvider extends ItemProviderAdapter implements
		IEditingDomainItemProvider, IStructuredItemContentProvider, ITreeItemContentProvider,
		IItemLabelProvider, IItemPropertySource {

	private EObject parent;

	/**
	 * Create a transient item provider
	 * 
	 * @param adapterFactory
	 *            the adapter factory used to create adapters
	 * @param parent
	 *            the model element, parent of the transient item
	 */
	public TransientItemProvider(AdapterFactory adapterFactory, EObject parent) {
		super(adapterFactory);
		this.parent = parent;
	}

	@Override
	public Collection<?> getChildren(Object object) {
		return super.getChildren(this.parent);
	}

	@Override
	public Collection<?> getNewChildDescriptors(Object object, EditingDomain editingDomain,
			Object sibling) {
		return super.getNewChildDescriptors(this.parent, editingDomain, sibling);
	}

	public void setParent(EObject parent) {
		this.parent = parent;
	}

	@Override
	public Object getParent(Object object) {
		return this.parent;
	}

	/**
	 * Modify the command so that the parameters are model elements (otherwise, we would get a
	 * {@link ClassCastException} from EMF). Also, set the owner of the command to be the parent
	 * model element.
	 * 
	 */
	@Override
	public Command createCommand(final Object object, final EditingDomain domain,
			Class<? extends Command> commandClass, CommandParameter commandParameter) {
		CommandParameter localCommandParameter = commandParameter;
		// disable copying transient elements
		if (commandClass == CopyCommand.class)
			return UnexecutableCommand.INSTANCE;

		Collection<?> oldCollection = localCommandParameter.getCollection();
		Collection<Object> newCollection = new ArrayList<Object>();
		if (oldCollection != null) {
			for (Object o : oldCollection) {
				if (o instanceof LinkItemProvider) {
					LinkItemProvider linkItemProvider = (LinkItemProvider) o;
					newCollection.add(linkItemProvider.getParent(null));
				} else if (o instanceof BigListItemProvider) {
					BigListItemProvider bigListItemProvider = (BigListItemProvider) o;
					newCollection.add(bigListItemProvider.getModelParent());
				} else if (o instanceof EObject) {
					newCollection.add(o);
				}
			}

			localCommandParameter = new CommandParameter(localCommandParameter.getOwner(),
					localCommandParameter.getFeature(), localCommandParameter.getValue(),
					newCollection, localCommandParameter.getIndex());
		}

		localCommandParameter.setOwner(this.parent);
		return super.createCommand(this.parent, domain, commandClass, localCommandParameter);
	}

	/** Wrap the command to redefine the affected objects */
	@Override
	protected Command createRemoveCommand(EditingDomain domain, EObject owner,
			EStructuralFeature feature, Collection<?> collection) {
		return createWrappedCommand(super.createRemoveCommand(domain, owner, feature, collection),
				owner);
	}

	/** Wrap the command to redefine the affected objects */
	@Override
	protected Command createAddCommand(EditingDomain domain, EObject owner,
			EStructuralFeature feature, Collection<?> collection, int index) {
		return createWrappedCommand(super.createAddCommand(domain, owner, feature, collection,
				index), owner);
	}

	/** Wrap the command to redefine the affected objects */
	protected Command createWrappedCommand(Command command, final EObject owner) {
		return new CommandWrapper(command) {
			@Override
			public Collection<?> getAffectedObjects() {
				Collection<?> affected = super.getAffectedObjects();
				if (affected.contains(owner)) {
					affected = Collections.singleton(TransientItemProvider.this);
				}
				return affected;
			}
		};
	}
}