/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.core;

import java.util.ArrayList;
import java.util.Collection;
import java.util.TreeMap;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.gmt.modisco.common.editor.util.EMFUtil;

/**
 * A list of metaclasses from the model, each metaclass having an associated list of instances of
 * the metaclass.
 */
public class InstancesForMetaclasses implements InstancesForMetaclass.ChangeListener {

	/**
	 * Model elements sorted by EClass (full qualified name). TreeMap (red-black tree),
	 * automatically sorted by String.
	 */
	private TreeMap<String, InstancesForMetaclass> modelElements = new TreeMap<String, InstancesForMetaclass>();

	/**
	 * A list of all metaclasses which do not have a parent. That is, a list of metaclasses from
	 * which all other metaclasses can be found by derivation.
	 */
	private ArrayList<InstancesForMetaclass> rootMetaclasses = new ArrayList<InstancesForMetaclass>();

	public InstancesForMetaclass[] getInstancesForMetaclasses() {
		Collection<InstancesForMetaclass> instancesForMetaclasses = this.modelElements.values();
		return instancesForMetaclasses.toArray(new InstancesForMetaclass[instancesForMetaclasses
				.size()]);
	}

	/**
	 * @param metaclassQualifiedName
	 *            the full qualified name of the metaclass
	 * @return the {@link InstancesForMetaclass} object or <code>null</code> if the metaclass was
	 *         not found
	 */
	public InstancesForMetaclass getInstancesForMetaclass(String metaclassQualifiedName) {
		return this.modelElements.get(metaclassQualifiedName);
	}

	/**
	 * Initialize the list of instances by metaclass from the resource set.
	 * 
	 * @param resourceSet
	 *            the resource set from the model in its initial state
	 */
	public void initModel(ResourceSet resourceSet) {
		EList<Resource> resources = resourceSet.getResources();
		if (resources.size() == 0)
			return;

		// consider only instances from the first resource
		Resource resource = resources.get(0);
		TreeIterator<EObject> allContents = resource.getAllContents();
		while (allContents.hasNext()) {
			EObject eObject = allContents.next();
			addModelElement(eObject, false);
		}
	}

	/**
	 * Add a new element to the MetaclassListItemProvider corresponding to its metaclass.
	 * <p>
	 * Create a new MetaclassListItemProvider if necessary.
	 * 
	 * @param element
	 *            the element to add
	 * @param recursively
	 *            whether to also add elements contained in the given element
	 */
	private void addModelElement(EObject element, boolean recursively) {
		String classQualifiedName = EMFUtil.getMetaclassQualifiedName(element.eClass());

		InstancesForMetaclass instancesForMetaclass = this.modelElements.get(classQualifiedName);

		if (instancesForMetaclass == null) {
			instancesForMetaclass = new InstancesForMetaclass(element.eClass(), this);
			instancesForMetaclass.setListener(this);
			this.modelElements.put(classQualifiedName, instancesForMetaclass);
		}
		instancesForMetaclass.add(element);
		fireModelChanged();

		if (recursively) {
			// also add the contained elements recursively
			EList<EObject> contents = element.eContents();
			for (EObject contained : contents) {
				addModelElement(contained, true);
			}
		}
	}

	/**
	 * Remove an element from the MetaclassListItemProvider corresponding to its class.
	 * <p>
	 * If the last element is removed from the list, then the MetaclassListItemProvider is removed
	 * from the modelElements.
	 * <p>
	 * Also removes elements contained in the given element.
	 * 
	 * @param element
	 *            the element to remove
	 */
	private void removeModelElement(EObject element) {
		String classQualifiedName = EMFUtil.getMetaclassQualifiedName(element.eClass());
		InstancesForMetaclass instancesForMetaclass = this.modelElements.get(classQualifiedName);
		if (instancesForMetaclass != null) {
			instancesForMetaclass.remove(element);
			fireModelChanged();
			if (instancesForMetaclass.isEmpty()) {
				// removing last instance of a metaclass
				fireRemovedLastInstanceof(classQualifiedName);
			}
		}

		// also remove the contained elements recursively
		EList<EObject> contents = element.eContents();
		for (EObject contained : contents) {
			removeModelElement(contained);
		}
	}

	/** A listener to receive notifications about model changes */
	public interface ModelChangeListener {
		/** The model changed (element added or removed) */
		void modelChanged();

		/** The last instance of the given metaclass was removed */
		void removedLastInstanceof(String metaclassQualifiedName);
	}

	/** Listeners for model changes */
	private ArrayList<ModelChangeListener> listeners = new ArrayList<ModelChangeListener>();

	/** Add a listener for model changes */
	public void addListener(ModelChangeListener listener) {
		this.listeners.add(listener);
	}

	/** Remove a previously added listener */
	public void removeListener(ModelChangeListener listener) {
		this.listeners.remove(listener);
	}

	/**
	 * This notification is sent by the {@link InstancesForMetaclass}s when one of their objects
	 * reports a change.
	 * <p>
	 * Respond to model elements creation and removal by updating the corresponding
	 * {@link InstancesForMetaclass}'s list, and firing the "modelChanged" event to a possible
	 * listener.
	 */
	public void notifyChanged(Notification msg) {
		int eventType = msg.getEventType();
		switch (eventType) {
		case Notification.ADD: {
			Object featureObj = msg.getFeature();
			if (featureObj instanceof EReference) {
				EReference feature = (EReference) featureObj;
				if (!feature.isContainment())
					return;
			}
			Object newValue = msg.getNewValue();
			if (newValue instanceof EObject) {
				addModelElement((EObject) newValue, true);
			}
			break;
		}
		case Notification.REMOVE: {
			Object featureObj = msg.getFeature();
			if (featureObj instanceof EReference) {
				EReference feature = (EReference) featureObj;
				if (!feature.isContainment())
					return;
			}
			Object oldValue = msg.getOldValue();
			if (oldValue instanceof EObject) {
				removeModelElement((EObject) oldValue);
			}
			break;
		}
		case Notification.ADD_MANY: {
			Object newValue = msg.getNewValue();
			if (newValue instanceof EList) {
				EList<?> eList = (EList<?>) newValue;
				for (Object object : eList) {
					if (object instanceof EObject) {
						addModelElement((EObject) object, true);
					}
				}
			}
			break;
		}
		case Notification.REMOVE_MANY: {
			Object newValue = msg.getNewValue();
			if (newValue instanceof EList) {
				EList<?> eList = (EList<?>) newValue;
				for (Object object : eList) {
					if (object instanceof EObject) {
						removeModelElement((EObject) object);
					}
				}
			}
			break;
		}
			// features that have a multiplicity of 1
		case Notification.SET: {
			Object newValue = msg.getNewValue();
			Object feature = msg.getFeature();
			if (feature instanceof EReference) {
				EReference reference = (EReference) feature;
				/*
				 * setting a reference should add or remove elements only if the reference is a
				 * composition link
				 */
				if (!reference.isContainment()) {
					break;
				}

				if (newValue == null) {
					Object oldValue = msg.getOldValue();
					if (oldValue instanceof EObject) {
						removeModelElement((EObject) oldValue);
					}
				} else if (newValue instanceof EObject) {
					addModelElement((EObject) newValue, true);
				}
			}
			break;
		}
		case Notification.UNSET: {
			Object oldValue = msg.getOldValue();
			Object feature = msg.getFeature();
			if (feature instanceof EReference) {
				EReference reference = (EReference) feature;
				/*
				 * unsetting a reference should remove the element only if the reference is a
				 * composition link
				 */
				if (!reference.isContainment()) {
					break;
				}
			}
			if (oldValue instanceof EObject) {
				removeModelElement((EObject) oldValue);
			}
			fireModelChanged();
			break;
		}
		}
	}

	private void fireModelChanged() {
		for (ModelChangeListener listener : this.listeners) {
			listener.modelChanged();
		}
	}

	private void fireRemovedLastInstanceof(String metaclassQualifiedName) {
		for (ModelChangeListener listener : this.listeners) {
			listener.removedLastInstanceof(metaclassQualifiedName);
		}
	}

	/** Adds the metaclasses from the given list to the list of metaclasses */
	public void addMetaclasses(Collection<EClass> metaclasses) {
		for (EClass eClass : metaclasses) {
			String metaclassQualifiedName = EMFUtil.getMetaclassQualifiedName(eClass);

			if (this.modelElements.get(metaclassQualifiedName) == null) {
				InstancesForMetaclass instancesForMetaclass = new InstancesForMetaclass(eClass,
						this);
				instancesForMetaclass.setListener(this);
				this.modelElements.put(metaclassQualifiedName, instancesForMetaclass);
			}
		}
	}

	/** Builds the derivation tree of metaclasses */
	public void buildDerivationTree() {
		InstancesForMetaclass[] instancesByMetaclass = getInstancesForMetaclasses();

		// clear previous state
		this.rootMetaclasses.clear();
		for (InstancesForMetaclass instancesForMetaclass : instancesByMetaclass) {
			instancesForMetaclass.clearSubclasses();
		}

		// build the derivation tree
		for (InstancesForMetaclass instancesForMetaclass : instancesByMetaclass) {
			instancesForMetaclass.buildParentsSubclasses();
			if (instancesForMetaclass.getEClass().getESuperTypes().isEmpty()) {
				this.rootMetaclasses.add(instancesForMetaclass);
			}
		}
	}

	public InstancesForMetaclass[] getRootMetaclasses() {
		return this.rootMetaclasses.toArray(new InstancesForMetaclass[this.rootMetaclasses.size()]);
	}
}
