/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.editors;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EventObject;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.command.BasicCommandStack;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CommandStack;
import org.eclipse.emf.common.command.CommandStackListener;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.ui.viewer.IViewerProvider;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EValidator;
import org.eclipse.emf.ecore.plugin.EcorePlugin;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.edit.domain.AdapterFactoryEditingDomain;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.domain.IEditingDomainProvider;
import org.eclipse.emf.edit.provider.AdapterFactoryItemDelegator;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory.Descriptor.Registry;
import org.eclipse.emf.edit.provider.resource.ResourceItemProviderAdapterFactory;
import org.eclipse.emf.edit.ui.action.EditingDomainActionBarContributor;
import org.eclipse.emf.edit.ui.celleditor.AdapterFactoryTreeEditor;
import org.eclipse.emf.edit.ui.dnd.EditingDomainViewerDropAdapter;
import org.eclipse.emf.edit.ui.dnd.LocalTransfer;
import org.eclipse.emf.edit.ui.dnd.ViewerDragAdapter;
import org.eclipse.emf.edit.ui.provider.AdapterFactoryContentProvider;
import org.eclipse.emf.edit.ui.provider.AdapterFactoryLabelProvider;
import org.eclipse.emf.edit.ui.provider.UnwrappingSelectionProvider;
import org.eclipse.emf.edit.ui.util.EditUIUtil;
import org.eclipse.emf.edit.ui.view.ExtendedPropertySheetPage;
import org.eclipse.gmt.modisco.common.editor.MoDiscoEditorPlugin;
import org.eclipse.gmt.modisco.common.editor.adapters.AttributeItemProvider;
import org.eclipse.gmt.modisco.common.editor.adapters.BigListItemProvider;
import org.eclipse.gmt.modisco.common.editor.adapters.LinkItemProvider;
import org.eclipse.gmt.modisco.common.editor.adapters.MetaclassInstancesItemProvider;
import org.eclipse.gmt.modisco.common.editor.adapters.MiaReflectiveItemProvider;
import org.eclipse.gmt.modisco.common.editor.adapters.MiaReflectiveItemProviderAdapterFactory;
import org.eclipse.gmt.modisco.common.editor.adapters.ParentAdapterFactory;
import org.eclipse.gmt.modisco.common.editor.adapters.SearchResultsItemProvider;
import org.eclipse.gmt.modisco.common.editor.core.InstancesForMetaclass;
import org.eclipse.gmt.modisco.common.editor.core.InstancesForMetaclasses;
import org.eclipse.gmt.modisco.common.editor.util.EMFUtil;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.resource.FontDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.INavigationLocation;
import org.eclipse.ui.INavigationLocationProvider;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.contexts.IContextActivation;
import org.eclipse.ui.contexts.IContextService;
import org.eclipse.ui.dialogs.SaveAsDialog;
import org.eclipse.ui.ide.IGotoMarker;
import org.eclipse.ui.part.EditorPart;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.views.properties.IPropertySheetPage;
import org.eclipse.ui.views.properties.PropertySheetPage;

/**
 * MoDisco model editor: a two pane editor. The left pane shows the list of metaclasses and the
 * right pane shows the list of instances of the metaclass selected in the left pane.
 */
public class EcoreEditor extends EditorPart implements IEditingDomainProvider, ISelectionProvider,
		IMenuListener, IViewerProvider, IGotoMarker, INavigationLocationProvider {

	public static final String EDITOR_ID = "org.eclipse.gmt.modisco.common.editor.editorID";
	private static final String PROPERTY_VIEW_ID = "org.eclipse.ui.views.PropertySheet";
	public static final String ECORE_FILE_EXTENSION = "ecore";
	public static final String EMOF_FILE_EXTENSION = "emof";

	/** This keeps track of the editing domain that is used to track all changes to the model. */
	protected AdapterFactoryEditingDomain editingDomain;

	/** The adapter factory used for providing views of the model. */
	protected ComposedAdapterFactory adapterFactory;

	/**
	 * Another adapter factory that uses adapter factories from the registry in addition to those
	 * defined in {@link AdapterFactory#adapterFactory}.
	 */
	protected ComposedAdapterFactory adapterFactoryWithRegistry;

	/** The property sheet page. */
	protected PropertySheetPage propertySheetPage;

	/** The tree part of the model editor. */
	protected TreeViewer treeViewer;

	/** The tree component (instances pane) */
	private Tree tree;

	/** The list of metaclasses that appears in the left pane of the editor */
	protected MetaclassViewer metaclassViewer;

	/** This listens to the tree viewer's selection. */
	protected ISelectionChangedListener selectionChangedListener;

	/**
	 * This keeps track of all the {@link org.eclipse.jface.viewers.ISelectionChangedListener}s that
	 * are listening to this editor.
	 */
	protected Collection<ISelectionChangedListener> selectionChangedListeners = new ArrayList<ISelectionChangedListener>();

	/** This keeps track of the selection of the editor as a whole. */
	protected ISelection editorSelection = StructuredSelection.EMPTY;

	/** This listens for workspace changes. */
	protected WorkspaceChangeListener resourceChangeListener;

	/**
	 * The configuration of this editor, which is passed to the adapters, so that they can use it to
	 * change their behavior accordingly
	 */
	private EditorConfiguration editorConfiguration;

	/** The composite in which the editor is displayed */
	private Composite parentComposite;

	/** The search box */
	private Text searchBox;

	/** The search job, to allow cancellation in case of a big model */
	private Job searchJob = null;

	/** The input of the tree viewer, saved before filtering by search, and restored afterwards */
	private Object savedInput;

	/** A job to allow optimal refreshing of the viewers */
	private Job refreshJob = null;

	/** A listener on changes to fonts settings */
	private IPropertyChangeListener fontChangeListener;

	/**
	 * Whether the resource being edited comes from the package registry, in which case it must not
	 * be unloaded
	 */
	private boolean editingRegistryResource;
	private SashForm topSashForm;
	private CustomTreePainter orderedReferencePainter;

	/** Create the Mia Ecore model editor. */
	public EcoreEditor() {
		super();
		this.editorConfiguration = new EditorConfiguration(this);

		// load user settings
		this.editorConfiguration.load();

		this.resourceChangeListener = new WorkspaceChangeListener(this);
	}

	/** Set up the editing domain for the model editor. */
	protected void initializeEditingDomain() {
		// Create an adapter factory that yields item providers.
		this.adapterFactory = new ComposedAdapterFactory();

		this.adapterFactory.addAdapterFactory(new ResourceItemProviderAdapterFactory());
		this.adapterFactory.addAdapterFactory(ParentAdapterFactory.getInstance());
		this.adapterFactory.addAdapterFactory(new MiaReflectiveItemProviderAdapterFactory(
				this.editorConfiguration));

		this.editorConfiguration.setAdapterFactory(this.adapterFactory);

		// Create a second adapter factory that delegates to the first one
		// and also looks in the registry
		Registry registry = ComposedAdapterFactory.Descriptor.Registry.INSTANCE;
		this.adapterFactoryWithRegistry = new ComposedAdapterFactory(registry);
		this.adapterFactoryWithRegistry.addAdapterFactory(this.adapterFactory);

		this.editorConfiguration.setAdapterFactoryWithRegistry(this.adapterFactoryWithRegistry);

		// Create the command stack that will notify this editor as commands are executed.
		BasicCommandStack commandStack = new BasicCommandStack();

		/*
		 * Add a listener to set the most recent command's affected objects to be the selection of
		 * the tree viewer.
		 */
		commandStack.addCommandStackListener(new CommandStackListener() {
			public void commandStackChanged(final EventObject event) {
				EcoreEditor.this.parentComposite.getDisplay().asyncExec(new Runnable() {
					public void run() {
						firePropertyChange(IEditorPart.PROP_DIRTY);

						// Try to select the affected objects.
						Command mostRecentCommand = ((CommandStack) event.getSource())
								.getMostRecentCommand();
						if (mostRecentCommand != null) {
							setSelectionToViewer(mostRecentCommand.getAffectedObjects());
						}
						if (EcoreEditor.this.propertySheetPage != null
								&& !EcoreEditor.this.propertySheetPage.getControl().isDisposed()) {
							EcoreEditor.this.propertySheetPage.refresh();
						}
					}
				});
			}
		});

		// Create the editing domain with a special command stack.
		this.editingDomain = new AdapterFactoryEditingDomain(this.adapterFactory, commandStack,
				new HashMap<Resource, Boolean>());
	}

	/** Set the selection into the tree viewer */
	public void setSelectionToViewer(final Collection<?> selection) {
		// Try to select the items in the tree viewer
		if (selection != null && !selection.isEmpty() && this.treeViewer != null) {
			this.treeViewer.setSelection(new StructuredSelection(selection.toArray()), true);
		}
	}

	/**
	 * This returns the editing domain as required by the {@link IEditingDomainProvider} interface.
	 * This is important for implementing the static methods of {@link AdapterFactoryEditingDomain}
	 * and for supporting {@link org.eclipse.emf.edit.ui.action.CommandAction}.
	 */
	public EditingDomain getEditingDomain() {
		return this.editingDomain;
	}

	/** @return the viewer as required by the {@link IViewerProvider} interface. */
	public Viewer getViewer() {
		return this.treeViewer;
	}

	public MetaclassViewer getMetaclassViewer() {
		return this.metaclassViewer;
	}

	/** Create a context menu for the viewer, add a listener, and register the menu for extension. */
	protected void createContextMenuFor(StructuredViewer viewer) {
		MenuManager contextMenu = new MenuManager("#PopUp");
		contextMenu.add(new Separator("additions"));
		contextMenu.setRemoveAllWhenShown(true);
		contextMenu.addMenuListener(this);
		Menu menu = contextMenu.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
		getSite().registerContextMenu(contextMenu, new UnwrappingSelectionProvider(viewer));
	}

	/** Create drag & drop support on the given tree viewer */
	private void createDragAndDropSupport(TreeViewer viewer) {
		int dndOperations = DND.DROP_COPY | DND.DROP_MOVE | DND.DROP_LINK;
		Transfer[] transfers = new Transfer[] { LocalTransfer.getInstance() };
		viewer.addDragSupport(dndOperations, transfers, new ViewerDragAdapter(viewer));
		viewer.addDropSupport(dndOperations, transfers, new EditingDomainViewerDropAdapter(
				this.editingDomain, viewer));
	}

	/** Load a resource into the editing domain's resource set based on the editor's input. */
	public void createModel() {
		this.editingDomain.getResourceSet().getURIConverter().getURIMap().putAll(
				EcorePlugin.computePlatformURIMap());

		this.editingRegistryResource = false;
		URI resourceURI = EditUIUtil.getURI(getEditorInput());
		try {
			IEditorInput editorInput = getEditorInput();
			if (editorInput instanceof FileEditorInput) {
				// Load the resource through the editing domain.
				this.editingDomain.getResourceSet().getResource(resourceURI, true);
			} else {
				String resolveURI = resourceURI.toString();
				EPackage ePackage = this.editingDomain.getResourceSet().getPackageRegistry()
						.getEPackage(resolveURI);
				Resource resource = ePackage.eResource();
				// TODO ? add referenced elements? for now no particular needs.
				if (resource != null) {
					this.editingDomain.getResourceSet().getResources().add(resource);
					this.editingRegistryResource = true;
				}
			}
		} catch (Exception e) {
			MoDiscoEditorPlugin.INSTANCE.log(e);
			MessageDialog.openError(this.parentComposite.getShell(), "Error Loading Model", e
					.toString());
			this.editingDomain.getResourceSet().getResource(resourceURI, false);
		}
	}

	/** Create the editor composite. */
	@Override
	public void createPartControl(Composite parent) {

		this.parentComposite = parent;
		parent.setLayout(new FillLayout());

		initializeEditingDomain();
		createModel();
		this.editorConfiguration.setResourceSet(this.editingDomain.getResourceSet());

		// register close listener to save user settings when editor is closed
		getSite().getPage().addPartListener(new SaveSettingsOnCloseListener());

		InstancesForMetaclasses instancesForMetaclasses = new InstancesForMetaclasses();
		this.editorConfiguration.setInstancesForMetaclasses(instancesForMetaclasses);

		// add all known metaclasses to the list (so that empty metaclasses can be displayed)
		instancesForMetaclasses.addMetaclasses(this.editorConfiguration.getAllClasses());

		instancesForMetaclasses.initModel(this.editingDomain.getResourceSet());
		instancesForMetaclasses.buildDerivationTree();

		this.topSashForm = new SashForm(parent, SWT.HORIZONTAL);

		createLeftPane(this.topSashForm);
		createTreePane(this.topSashForm);
		setupInitialTreeFont();

		this.topSashForm.setWeights(new int[] { 30, 70 });

		// keep track of the selection, to be able to restore it when using the "back" button
		addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				markNavigationLocation();
			}
		});

		/*
		 * Select the metaclass corresponding to the 'root' element (that is, the 'model' element).
		 * This in turn sets the input of the treeview.
		 */
		this.metaclassViewer.selectRootElement();

		instancesForMetaclasses.addListener(new InstancesForMetaclasses.ModelChangeListener() {

			// clear selection when the selected metaclass is removed from the list
			public void removedLastInstanceof(String metaclassQualifiedName) {
				String[] selectedMetaclasses = EcoreEditor.this.metaclassViewer
						.getSelectedMetaclassesQualifiedNames();
				if (selectedMetaclasses.length == 1) {
					if (metaclassQualifiedName.equals(selectedMetaclasses[0])) {
						EcoreEditor.this.metaclassViewer.clearSelection();
					}
				}
			}

			// refresh the viewers when a model element is added or removed
			public void modelChanged() {
				EcoreEditor.this.modelChanged();
			}
		});

		createContextMenuFor(this.treeViewer);
		createDragAndDropSupport(this.treeViewer);
	}

	/**
	 * Delayed viewer refresh, to avoid refreshing the views 100 times when 100 items are added or
	 * deleted in a single operation (or in quick succession).
	 */
	private void modelChanged() {
		if (this.refreshJob == null) {
			this.refreshJob = new Job("Refresh model viewers") {
				@Override
				protected IStatus run(IProgressMonitor monitor) {
					Display.getDefault().syncExec(new Runnable() {
						public void run() {
							EcoreEditor.this.metaclassViewer.refresh();

							if (!EcoreEditor.this.treeViewer.getControl().isDisposed())
								EcoreEditor.this.treeViewer.refresh();
						}
					});
					return Status.OK_STATUS;
				}
			};
		} else {
			/*
			 * if modelChanged is called again before the previous operation could finish, then
			 * cancel it and schedule a new refresh job.
			 */
			this.refreshJob.cancel();
		}
		this.refreshJob.setPriority(Job.DECORATE);
		this.refreshJob.schedule(100);
	}

	/**
	 * Create the left pane, containing the metaclass viewer.
	 * 
	 * @param parent
	 *            the parent composite
	 */
	protected void createLeftPane(Composite parent) {
		Composite listPaneComposite = new Composite(parent, SWT.NONE);

		LeftPane leftPane = new LeftPane(listPaneComposite, this.editorConfiguration);

		this.metaclassViewer = leftPane.getMetaclassViewer();
		this.metaclassViewer
				.addSelectionChangedListener(new MetaclassListViewerSelectionChangedListener());
		fixClipboardCommands(leftPane.getMetaclassViewer().getViewer().getControl());
	}

	/**
	 * Create the tree and its viewer
	 * 
	 * @param parent
	 *            the parent composite of the tree
	 */
	protected void createTreePane(Composite parent) {

		Composite treePane = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout(1, false);
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 2;
		treePane.setLayout(gridLayout);

		createTreeToolBar(treePane);

		this.tree = new Tree(treePane, SWT.MULTI | SWT.BORDER);
		GridData treeGridData = new GridData();
		treeGridData.grabExcessHorizontalSpace = true;
		treeGridData.grabExcessVerticalSpace = true;
		treeGridData.horizontalAlignment = SWT.FILL;
		treeGridData.verticalAlignment = SWT.FILL;
		this.tree.setLayoutData(treeGridData);

		this.tree.addKeyListener(new KeyAdapter() {
			@Override
			public void keyPressed(KeyEvent e) {
				// Escape on the tree: restore view (after search)
				if (e.keyCode == SWT.ESC) {
					EcoreEditor.this.searchBox.setText("");
					search();

				}
				// Enter: browse selected item
				else if (e.keyCode == SWT.CR) {
					handleEnter();
				}
			}
		});

		this.orderedReferencePainter = new CustomTreePainter(this.tree, this.editorConfiguration);

		this.treeViewer = new TreeViewer(this.tree);
		// speeds up all operations on a big model
		this.treeViewer.setUseHashlookup(true);

		hookViewerToSelection(this.treeViewer);

		// the content is provided through item providers created by the adapter factory
		this.treeViewer.setContentProvider(new AdapterFactoryContentProvider(this.adapterFactory));
		// set a label provider that handles fonts and colors too
		this.treeViewer.setLabelProvider(new AdapterFactoryLabelProvider.FontAndColorProvider(
				this.adapterFactory, this.treeViewer));

		new AdapterFactoryTreeEditor(this.treeViewer.getTree(), this.adapterFactory);

		registerShowPropertyViewOnDblClick(this.tree);
	}

	/**
	 * The user pressed the "Enter" key : if selected element is a model element, then browse to it.
	 * If it is an attribute, then open properties view.
	 */
	protected void handleEnter() {
		ISelection selection = getSelection();
		if (selection instanceof ITreeSelection) {
			ITreeSelection treeSelection = (ITreeSelection) selection;
			final Object selectedElement = treeSelection.getFirstElement();
			if (selectedElement instanceof EObject) {
				browse();
			} else if (selectedElement instanceof AttributeItemProvider
					|| selectedElement instanceof LinkItemProvider) {
				openPropertiesView();
			}
		}
	}

	/**
	 * Set the initial font using saved user preferences, and add a listener to be notified of
	 * changes in the preferences
	 */
	private void setupInitialTreeFont() {
		setupPreferredFont();

		this.fontChangeListener = new IPropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent event) {
				if (event.getProperty().equals(JFaceResources.DIALOG_FONT)) {
					setupPreferredFont();
				}
			}
		};
		JFaceResources.getFontRegistry().addListener(this.fontChangeListener);
	}

	/** Set the user preferred font on the editor */
	private void setupPreferredFont() {
		FontDescriptor descriptor = FontDescriptor.createFrom(JFaceResources.getDialogFont());
		descriptor = descriptor.increaseHeight(this.editorConfiguration.getFontSizeDelta());
		Font customFont = descriptor.createFont(Display.getDefault());
		descriptor = descriptor.setStyle(SWT.ITALIC);
		Font customItalicFont = descriptor.createFont(Display.getDefault());
		this.editorConfiguration.setCustomFont(customFont);
		this.editorConfiguration.setCustomItalicFont(customItalicFont);
		this.tree.setFont(customFont);
		this.treeViewer.refresh();
		this.metaclassViewer.setFont(customFont);
		this.metaclassViewer.refresh();
	}

	private void createTreeToolBar(Composite parent) {

		Composite treeToolBarComposite = new Composite(parent, SWT.BORDER);
		GridData treeToolBarGridData = new GridData();
		treeToolBarGridData.grabExcessHorizontalSpace = true;
		treeToolBarGridData.horizontalAlignment = SWT.FILL;
		treeToolBarComposite.setLayoutData(treeToolBarGridData);

		GridLayout gridLayout = new GridLayout(2, false);
		gridLayout.horizontalSpacing = 20;
		gridLayout.marginHeight = 2;
		treeToolBarComposite.setLayout(gridLayout);

		new TreeToolBar(treeToolBarComposite, this);

		createSearchPanel(treeToolBarComposite);
	}

	/** Register a listener to show the property view when the tree is double-clicked */
	private void registerShowPropertyViewOnDblClick(Tree tree) {
		tree.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseDoubleClick(MouseEvent event) {
				if (event.button == 1) {
					openPropertiesView();
				}
			}
		});
	}

	/** A listener to save user settings when the editor is closed */
	private class SaveSettingsOnCloseListener implements IPartListener {
		public void partClosed(IWorkbenchPart part) {
			if (part == EcoreEditor.this) {
				EcoreEditor.this.editorConfiguration.save();
				getSite().getPage().removePartListener(this);
			}
		}

		public void partOpened(IWorkbenchPart part) {
		}

		public void partDeactivated(IWorkbenchPart part) {
		}

		public void partBroughtToTop(IWorkbenchPart part) {
		}

		public void partActivated(IWorkbenchPart part) {
		}
	}

	/**
	 * Create the search panel, containing a label and search field.
	 * 
	 * @param parent
	 *            the parent composite, which must have a GridLayout
	 */
	protected void createSearchPanel(Composite parent) {
		Composite searchPanel = new Composite(parent, SWT.NONE);
		searchPanel.setLayout(new GridLayout(2, false));

		searchPanel.setLayout(new FillLayout());
		GridData searchPanelGridData = new GridData();
		searchPanelGridData.grabExcessHorizontalSpace = true;
		searchPanelGridData.horizontalAlignment = SWT.FILL;
		searchPanel.setLayoutData(searchPanelGridData);

		// Label searchLabel = new Label(searchPanel, SWT.NONE);
		// searchLabel.setText("&Search:");

		this.searchBox = new Text(searchPanel, SWT.SEARCH);
		this.searchBox.addKeyListener(new KeyAdapter() {
			@Override
			public void keyPressed(KeyEvent e) {
				// ENTER: start a search
				if (e.keyCode == SWT.CR || e.keyCode == SWT.KEYPAD_CR) {
					search();
				}
				// ESCAPE: restore the view after a search
				else if (e.keyCode == SWT.ESC) {
					EcoreEditor.this.searchBox.setText("");
					search();
				}
			}
		});

		// fix a bug where Ctrl+X, Ctrl+C and Ctrl+V modified the model
		// instead of the search box contents
		fixClipboardCommands(this.searchBox);

		this.searchBox.setMessage("Search : Enter a string and press <Enter>");
	}

	/**
	 * Replace the default context by a dummy one when the given widget is focused. The dummy
	 * context is used to redefine the "Ctrl+X", "Ctrl+C" and "Ctrl+V" key bindings to a command
	 * which does nothing. This is useful to prevent these commands associated to the existing
	 * context from being executed when this widget is focused.
	 */
	private void fixClipboardCommands(final Control control) {
		final IWorkbench workbench = PlatformUI.getWorkbench();
		final IContextService contextService = (IContextService) workbench
				.getService(IContextService.class);

		final IContextActivation[] contextActivationRef = new IContextActivation[1];

		control.addFocusListener(new FocusListener() {
			public void focusGained(final FocusEvent event) {
				IContextActivation contextActivation = contextService
						.activateContext("org.eclipse.gmt.modisco.common.editor.dummyContext");
				contextActivationRef[0] = contextActivation;
			}

			public void focusLost(final FocusEvent event) {
				if (contextActivationRef[0] != null) {
					contextService.deactivateContext(contextActivationRef[0]);
				}
			}
		});
	}

	/**
	 * Called when the user initiates a search. Find the list of all the instances of the selected
	 * metaclass that have a name containing the search string (entered by the user in the search
	 * box), in a case insensitive way. Set this list as input of the tree viewer.
	 * <p>
	 * Do the search in a Job that can be canceled by the user.
	 * <p>
	 * If the search box is empty, then restore the previous tree viewer input.
	 */
	protected void search() {
		final String searchText = this.searchBox.getText().toLowerCase();

		if (searchText.trim().length() == 0) {
			if (this.savedInput != null && this.savedInput != this.treeViewer.getInput()) {
				this.treeViewer.setInput(this.savedInput);
			}
			return;
		}

		final List<EClass> selectedMetaclasses = findSelectedEClasses();
		if (selectedMetaclasses.isEmpty())
			return;

		this.searchJob = new Job("Search in Model Editor") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {

				ResourceSet resourceSet = EcoreEditor.this.editingDomain.getResourceSet();
				EList<Resource> resources = resourceSet.getResources();
				if (resources.size() < 1)
					return Status.OK_STATUS;

				// the item provider that will become the input of the tree viewer
				final SearchResultsItemProvider searchResultsItemProvider = new SearchResultsItemProvider(
						EcoreEditor.this.adapterFactory, EcoreEditor.this.editorConfiguration);

				Resource resource = resources.get(0);
				TreeIterator<EObject> allContents = resource.getAllContents();
				while (allContents.hasNext()) {
					EObject eObject = allContents.next();
					if (selectedMetaclasses.contains(eObject.eClass())) {
						Adapter adapter = EcoreEditor.this.adapterFactory.adapt(eObject,
								IItemLabelProvider.class);
						if (adapter != null) {
							// use this provider because we need the undecorated name
							MiaReflectiveItemProvider provider = (MiaReflectiveItemProvider) adapter;
							String objectText = provider.getName(eObject).toLowerCase();
							if (objectText.contains(searchText)) {
								searchResultsItemProvider.add(eObject);
							}
						}
					}

					// if the user canceled the search
					if (monitor.isCanceled())
						return Status.CANCEL_STATUS;
				}

				// do only the minimum amount of work in the UI thread
				Display.getDefault().syncExec(new Runnable() {
					public void run() {
						// do not try to restore selection
						EcoreEditor.this.treeViewer.setSelection(null);
						EcoreEditor.this.treeViewer.setInput(searchResultsItemProvider);
					}
				});
				return Status.OK_STATUS;
			}
		};
		this.searchJob.setUser(true);
		this.searchJob.setPriority(Job.LONG);
		this.searchJob.schedule();
	}

	/**
	 * @return the {@link EClass}es corresponding to the metaclasses selected in the metaclass list
	 *         (empty list if none)
	 */
	private List<EClass> findSelectedEClasses() {

		ArrayList<EClass> selectedMetaclasses = new ArrayList<EClass>();

		ISelection selection = this.metaclassViewer.getSelection();
		if (selection instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection) selection;

			Iterator<?> iterator = structuredSelection.iterator();
			while (iterator.hasNext()) {
				Object element = iterator.next();
				if (element instanceof InstancesForMetaclass) {
					InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) element;
					addMetaclassToSelection(instancesForMetaclass, selectedMetaclasses,
							this.editorConfiguration.isDisplayInstancesOfSubclasses());
				}
			}
		}
		return selectedMetaclasses;
	}

	/**
	 * Adds the given metaclass to the given selection.
	 * 
	 * @param addSubclasses
	 *            also adds subclasses
	 */
	private void addMetaclassToSelection(InstancesForMetaclass instancesForMetaclass,
			ArrayList<EClass> selectedMetaclasses, boolean addSubclasses) {
		selectedMetaclasses.add(instancesForMetaclass.getEClass());
		// add subclasses recursively
		if (addSubclasses) {
			InstancesForMetaclass[] subclasses = instancesForMetaclass.getSubclasses();
			for (InstancesForMetaclass subclass : subclasses) {
				addMetaclassToSelection(subclass, selectedMetaclasses, addSubclasses);
			}
		}
	}

	/**
	 * Listen to selection changes in the metaclass list viewer, and update the tree viewer
	 * accordingly.
	 */
	private final class MetaclassListViewerSelectionChangedListener implements
			ISelectionChangedListener {

		/** Memorize the tree selection for each metaclass */
		private HashMap<String, ITreeSelection> selections = new HashMap<String, ITreeSelection>();
		/** The metaclass that was selected before the selection changed */
		private String lastSelectedMetaclass = null;

		/**
		 * Get the selected entries corresponding to the given selection in the metaclass viewer
		 * (may be empty)
		 */
		private InstancesForMetaclass[] getEntriesForSelection(ISelection selection) {
			ArrayList<InstancesForMetaclass> instancesForMetaclasses = new ArrayList<InstancesForMetaclass>();

			if (selection instanceof IStructuredSelection) {
				IStructuredSelection structuredSelection = (IStructuredSelection) selection;

				for (Iterator<?> iterator = structuredSelection.iterator(); iterator.hasNext();) {
					Object element = iterator.next();
					if (element instanceof InstancesForMetaclass) {
						InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) element;
						instancesForMetaclasses.add(instancesForMetaclass);
					}
				}
			}

			return instancesForMetaclasses
					.toArray(new InstancesForMetaclass[instancesForMetaclasses.size()]);
		}

		/**
		 * Updates the tree viewer when the selection changes in the metaclass viewer. Keeps track
		 * of the tree selection for each metaclass, and restores it when possible.
		 */
		public void selectionChanged(SelectionChangedEvent event) {

			// save the selection
			if (this.lastSelectedMetaclass != null) {
				this.selections.put(this.lastSelectedMetaclass,
						(ITreeSelection) EcoreEditor.this.treeViewer.getSelection());
			}

			ISelection metaclassSelection = event.getSelection();
			InstancesForMetaclass[] entries = getEntriesForSelection(metaclassSelection);

			ITreeSelection savedSelection = null;
			Object input = null;
			if (entries.length > 0) {
				savedSelection = this.selections.get(entries[0].getClassQualifiedName());

				String[] selectedMetaclassesNames = new String[entries.length];
				int i = 0;
				for (InstancesForMetaclass entry : entries) {
					selectedMetaclassesNames[i++] = entry.getClassQualifiedName();
				}

				// clean up the previous item provider
				if (EcoreEditor.this.savedInput instanceof MetaclassInstancesItemProvider) {
					MetaclassInstancesItemProvider metaclassInstancesItemProvider = (MetaclassInstancesItemProvider) EcoreEditor.this.savedInput;
					metaclassInstancesItemProvider.dispose();
				}

				input = new MetaclassInstancesItemProvider(EcoreEditor.this.adapterFactory,
						EcoreEditor.this.editorConfiguration, selectedMetaclassesNames);
				this.lastSelectedMetaclass = entries[0].getClassQualifiedName();
			} else {
				this.lastSelectedMetaclass = null;
			}

			try {
				// speedup and avoid flickering
				EcoreEditor.this.treeViewer.getTree().setRedraw(false);

				EcoreEditor.this.searchBox.setText("");

				// do not restore selection twice
				EcoreEditor.this.treeViewer.setSelection(null);

				// set the list of model elements as the input of the treeview
				EcoreEditor.this.treeViewer.setInput(input);
				EcoreEditor.this.savedInput = input;
				EcoreEditor.this.treeViewer.setSelection(savedSelection, true);
			} finally {
				EcoreEditor.this.treeViewer.getTree().setRedraw(true);
			}

			markNavigationLocation();
		}
	}

	public void setShowEmptyLinks(boolean value) {
		this.editorConfiguration.setShowEmptyLinks(value);
		refreshTree();
	}

	public boolean isShowEmptyLinks() {
		return this.editorConfiguration.isShowEmptyLinks();
	}

	public void setSortInstances(boolean value) {
		this.editorConfiguration.setSortInstances(value);
		refreshTree();
	}

	public boolean isSortInstances() {
		return this.editorConfiguration.isSortInstances();
	}

	public void setShowDerivedLinks(boolean value) {
		this.editorConfiguration.setShowDerivedLinks(value);
		refreshTree();
	}

	public boolean isShowDerivedLinks() {
		return this.editorConfiguration.isShowDerivedLinks();
	}

	public void setShowFullQualifiedNames(boolean value) {
		this.editorConfiguration.setShowFullQualifiedNames(value);
		refreshTree();
		this.metaclassViewer.refresh();
	}

	public boolean isShowFullQualifiedNames() {
		return this.editorConfiguration.isShowFullQualifiedNames();
	}

	public void setShowMultiplicity(boolean value) {
		this.editorConfiguration.setShowMultiplicity(value);
		refreshTree();
	}

	public boolean isShowMultiplicity() {
		return this.editorConfiguration.isShowMultiplicity();
	}

	public void setShowOppositeLinks(boolean value) {
		this.editorConfiguration.setShowOppositeLinks(value);
		refreshTree();
	}

	public boolean isShowOppositeLinks() {
		return this.editorConfiguration.isShowOppositeLinks();
	}

	public void setShowContainer(boolean value) {
		this.editorConfiguration.setShowContainer(value);
		refreshTree();
	}

	public boolean isShowContainer() {
		return this.editorConfiguration.isShowContainer();
	}

	public void setSortLinks(boolean value) {
		this.editorConfiguration.setSortLinks(value);
		refreshTree();
	}

	public boolean isSortLinks() {
		return this.editorConfiguration.isSortLinks();
	}

	public void setSortLinksByType(boolean value) {
		this.editorConfiguration.setSortLinksByType(value);
		refreshTree();
	}

	public boolean isSortLinksByType() {
		return this.editorConfiguration.isSortLinksByType();
	}

	public void setShowAttributes(boolean value) {
		this.editorConfiguration.setShowAttributes(value);
		refreshTree();
	}

	public boolean isShowAttributes() {
		return this.editorConfiguration.isShowAttributes();
	}

	public void setShowEmptyAttributes(boolean value) {
		this.editorConfiguration.setShowEmptyAttributes(value);
		refreshTree();
	}

	public boolean isShowEmptyAttributes() {
		return this.editorConfiguration.isShowEmptyAttributes();
	}

	public void setFontSizeDelta(int value) {
		this.editorConfiguration.setFontSizeDelta(value);
		setupPreferredFont();
	}

	public int getFontSizeDelta() {
		return this.editorConfiguration.getFontSizeDelta();
	}

	public void setShowOrdering(boolean value) {
		this.editorConfiguration.setShowOrdering(value);
		refreshTree();
	}

	public boolean isShowOrdering() {
		return this.editorConfiguration.isShowOrdering();
	}

	/** This is how the framework determines which interfaces we implement. */
	@SuppressWarnings("unchecked")
	@Override
	public Object getAdapter(Class key) {
		if (key.equals(IPropertySheetPage.class))
			return getPropertySheetPage();
		else if (key.equals(IGotoMarker.class))
			return this;
		else
			return super.getAdapter(key);
	}

	/** This accesses a cached version of the property sheet page. */
	public IPropertySheetPage getPropertySheetPage() {
		if (this.propertySheetPage == null) {
			this.propertySheetPage = new ExtendedPropertySheetPage(this.editingDomain) {
				@Override
				public void setSelectionToViewer(List<?> selection) {
					EcoreEditor.this.setSelectionToViewer(selection);
					EcoreEditor.this.setFocus();
				}

				@Override
				public void setActionBars(IActionBars actionBars) {
					super.setActionBars(actionBars);
					getActionBarContributor().shareGlobalActions(this, actionBars);
				}

				@Override
				public void selectionChanged(IWorkbenchPart part, ISelection selection) {
					// avoid the slow view redrawing
					getControl().setRedraw(false);
					try {
						super.selectionChanged(part, selection);
					} finally {
						getControl().setRedraw(true);
					}
				}
			};
			this.propertySheetPage.setPropertySourceProvider(new AdapterFactoryContentProvider(
					this.adapterFactory));
		}

		return this.propertySheetPage;
	}

	/** Handles activation of the editor or its associated views. */
	protected void handleActivate() {
		// Recompute the read only state.
		if (this.editingDomain.getResourceToReadOnlyMap() != null) {
			this.editingDomain.getResourceToReadOnlyMap().clear();

			// Refresh any actions that may become enabled or disabled.
			setSelection(getSelection());
		}
	}

	/**
	 * Handles what to do with changed resources on activation.
	 * 
	 * @param changedResources
	 *            the resources that have changed
	 */
	protected void handleChangedResources(Collection<Resource> changedResources) {

		if (isDirty()) {
			changedResources.addAll(this.editingDomain.getResourceSet().getResources());
		}
		this.editingDomain.getCommandStack().flush();

		for (Resource resource : changedResources) {
			if (resource.isLoaded()) {
				resource.unload();
				try {
					resource.load(Collections.EMPTY_MAP);
				} catch (IOException exception) {
					MoDiscoEditorPlugin.INSTANCE.log(exception);
					MessageDialog.openError(this.parentComposite.getShell(),
							"Error loading resource", exception.toString());
				}
			}
		}

		if (AdapterFactoryEditingDomain.isStale(this.editorSelection)) {
			setSelection(StructuredSelection.EMPTY);
		}
	}

	/** This is for implementing {@link IEditorPart} : simply tests the command stack. */
	@Override
	public boolean isDirty() {
		return ((BasicCommandStack) this.editingDomain.getCommandStack()).isSaveNeeded();
	}

	/** This is for implementing {@link IEditorPart} : saves the model file. */
	@Override
	public void doSave(IProgressMonitor progressMonitor) {
		// Save only resources that have actually changed.
		final Map<Object, Object> saveOptions = new HashMap<Object, Object>();
		saveOptions.put(Resource.OPTION_SAVE_ONLY_IF_CHANGED,
				Resource.OPTION_SAVE_ONLY_IF_CHANGED_MEMORY_BUFFER);

		// Do the work within an operation because this is a long running
		// activity that modifies the workbench.
		WorkspaceModifyOperation operation = new WorkspaceModifyOperation() {
			// This is the method that gets invoked when the operation runs.
			@Override
			public void execute(IProgressMonitor monitor) {
				// Save the resources to the file system.
				boolean first = true;
				for (Resource resource : EcoreEditor.this.editingDomain.getResourceSet()
						.getResources()) {
					if ((first || !resource.getContents().isEmpty() || isPersisted(resource))
							&& !EcoreEditor.this.editingDomain.isReadOnly(resource)) {
						try {
							long timeStamp = resource.getTimeStamp();
							resource.save(saveOptions);
							if (resource.getTimeStamp() != timeStamp) {
								EcoreEditor.this.resourceChangeListener.addSavedResource(resource);
							}
						} catch (Exception exception) {
							MessageDialog.openError(EcoreEditor.this.parentComposite.getShell(),
									"Error saving resource", exception.toString());
							MoDiscoEditorPlugin.INSTANCE.log(exception);
						}
						first = false;
					}
				}
			}
		};

		try {
			// This runs the operation, and shows progress.
			new ProgressMonitorDialog(getSite().getShell()).run(true, false, operation);

			// Refresh the necessary state.
			((BasicCommandStack) this.editingDomain.getCommandStack()).saveIsDone();
			firePropertyChange(IEditorPart.PROP_DIRTY);

			// the model can change when saving
			this.treeViewer.refresh();
		} catch (Exception exception) {
			// Something went wrong that shouldn't.
			MoDiscoEditorPlugin.INSTANCE.log(exception);
		}
	}

	/**
	 * This returns whether something has been persisted to the URI of the specified resource. The
	 * implementation uses the URI converter from the editor's resource set to try to open an input
	 * stream.
	 */
	protected boolean isPersisted(Resource resource) {
		boolean result = false;
		try {
			InputStream stream = this.editingDomain.getResourceSet().getURIConverter()
					.createInputStream(resource.getURI());
			if (stream != null) {
				result = true;
				stream.close();
			}
		} catch (IOException e) {
			// Ignore
		}
		return result;
	}

	/** This always returns true */
	@Override
	public boolean isSaveAsAllowed() {
		return true;
	}

	/**
	 * Open a "Save As" dialog and save the first resource to the file selected by the user. This
	 * also changes the editor's input to the new file.
	 */
	@Override
	public void doSaveAs() {
		SaveAsDialog saveAsDialog = new SaveAsDialog(getSite().getShell());
		saveAsDialog.create();
		saveAsDialog.setMessage(MoDiscoEditorPlugin.INSTANCE.getString("_UI_SaveAs_message"));
		saveAsDialog.open();
		IPath path = saveAsDialog.getResult();
		if (path != null) {
			IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(path);
			if (file != null) {
				ResourceSet resourceSet = this.editingDomain.getResourceSet();
				Resource currentResource = resourceSet.getResources().get(0);
				String currentExtension = currentResource.getURI().fileExtension();

				URI newURI = URI.createPlatformResourceURI(file.getFullPath().toString(), true);
				String newExtension = newURI.fileExtension();

				if (currentExtension.equals(ECORE_FILE_EXTENSION)
						&& newExtension.equals(EMOF_FILE_EXTENSION)
						|| currentExtension.equals(EMOF_FILE_EXTENSION)
						&& newExtension.equals(ECORE_FILE_EXTENSION)) {
					Resource newResource = resourceSet.createResource(newURI);
					newResource.getContents().addAll(currentResource.getContents());
					resourceSet.getResources().remove(0);
					resourceSet.getResources().move(0, newResource);
				} else {
					currentResource.setURI(newURI);
				}

				IFileEditorInput modelFile = new FileEditorInput(file);
				setInputWithNotify(modelFile);
				setPartName(file.getName());
				doSave(getActionBars().getStatusLineManager().getProgressMonitor());
			}
		}
	}

	/** Implements {@link IGotoMarker}. Select the EObject corresponding to the given marker */
	public void gotoMarker(IMarker marker) {
		String uriAttribute = marker.getAttribute(EValidator.URI_ATTRIBUTE, null);
		if (uriAttribute != null) {
			URI uri = URI.createURI(uriAttribute);
			EObject eObject = this.editingDomain.getResourceSet().getEObject(uri, true);
			if (eObject != null) {
				markNavigationLocation();
				String classQualifiedName = EMFUtil.getMetaclassQualifiedName(eObject.eClass());
				// Select the metaclass in the metaclass list viewer
				this.metaclassViewer.selectMetaclass(classQualifiedName);
				selectElement(eObject);
				markNavigationLocation();
			}
		}
	}

	/** This is called during startup to initialize the editor with its site and input. */
	@Override
	public void init(IEditorSite site, IEditorInput editorInput) {
		try {
			setSite(site);
			setInputWithNotify(editorInput);
			setPartName(editorInput.getName());
			site.setSelectionProvider(this);
			ResourcesPlugin.getWorkspace().addResourceChangeListener(this.resourceChangeListener,
					IResourceChangeEvent.POST_CHANGE);
		} catch (Exception e) {
			e.printStackTrace();
			MoDiscoEditorPlugin.INSTANCE.log(e);
		}
	}

	/** Focus the tree viewer when the editor is focused */
	@Override
	public void setFocus() {
		if (this.treeViewer != null) {
			this.treeViewer.getTree().setFocus();
		}
	}

	/** This implements {@link org.eclipse.jface.viewers.ISelectionProvider}. */
	public void addSelectionChangedListener(ISelectionChangedListener listener) {
		this.selectionChangedListeners.add(listener);
	}

	/** This implements {@link org.eclipse.jface.viewers.ISelectionProvider}. */
	public void removeSelectionChangedListener(ISelectionChangedListener listener) {
		this.selectionChangedListeners.remove(listener);
	}

	/**
	 * This implements {@link org.eclipse.jface.viewers.ISelectionProvider} to return this editor's
	 * overall selection.
	 */
	public ISelection getSelection() {
		return this.editorSelection;
	}

	/**
	 * This implements {@link org.eclipse.jface.viewers.ISelectionProvider} to set this editor's
	 * overall selection. Calling this result will notify the listeners.
	 */
	public void setSelection(ISelection selection) {
		this.editorSelection = selection;

		for (ISelectionChangedListener listener : this.selectionChangedListeners) {
			listener.selectionChanged(new SelectionChangedEvent(this, selection));
		}
		setStatusLineMessage(selection);
	}

	/** Connect the given viewer to the editor's selection */
	private void hookViewerToSelection(Viewer viewer) {
		if (viewer == null)
			return;

		if (this.selectionChangedListener == null) {
			// Create the listener on demand
			this.selectionChangedListener = new ISelectionChangedListener() {
				// This just notifies those things that are affected by the selection
				public void selectionChanged(SelectionChangedEvent selectionChangedEvent) {
					setSelection(selectionChangedEvent.getSelection());
				}
			};
		}

		viewer.addSelectionChangedListener(this.selectionChangedListener);

		// Set the editor's selection based on the viewer's selection
		setSelection(viewer.getSelection());
	}

	/** Set the status line message depending on the given selection */
	public void setStatusLineMessage(ISelection selection) {
		IStatusLineManager statusLineManager = getActionBars().getStatusLineManager();

		if (statusLineManager != null) {
			if (selection instanceof IStructuredSelection) {
				Collection<?> collection = ((IStructuredSelection) selection).toList();
				switch (collection.size()) {
				case 0: {
					statusLineManager.setMessage(getString("_UI_NoObjectSelected"));
					break;
				}
				case 1: {
					String text = new AdapterFactoryItemDelegator(this.adapterFactory)
							.getText(collection.iterator().next());
					statusLineManager.setMessage(getString("_UI_SingleObjectSelected", text));
					break;
				}
				default: {
					statusLineManager.setMessage(getString("_UI_MultiObjectSelected", Integer
							.toString(collection.size())));
					break;
				}
				}
			} else {
				statusLineManager.setMessage("");
			}
		}
	}

	/** This looks up a string in the plugin's plugin.properties file. */
	private static String getString(String key) {
		return MoDiscoEditorPlugin.INSTANCE.getString(key);
	}

	/** This looks up a string in plugin.properties, making a substitution. */
	private static String getString(String key, Object s1) {
		return MoDiscoEditorPlugin.INSTANCE.getString(key, new Object[] { s1 });
	}

	/**
	 * This implements {@link org.eclipse.jface.action.IMenuListener} to help fill the context menus
	 * with contributions from the Edit menu.
	 */
	public void menuAboutToShow(IMenuManager menuManager) {
		((IMenuListener) getEditorSite().getActionBarContributor()).menuAboutToShow(menuManager);
	}

	public EditingDomainActionBarContributor getActionBarContributor() {
		return (EditingDomainActionBarContributor) getEditorSite().getActionBarContributor();
	}

	public IActionBars getActionBars() {
		return getActionBarContributor().getActionBars();
	}

	public AdapterFactory getAdapterFactory() {
		return this.adapterFactory;
	}

	/** Clean up */
	@Override
	public void dispose() {
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this.resourceChangeListener);

		this.adapterFactory.dispose();
		this.adapterFactoryWithRegistry.dispose();
		this.topSashForm.dispose();

		if (!this.editingRegistryResource) {
			// unload all resources
			final EList<Resource> resources = this.editingDomain.getResourceSet().getResources();
			Job cleanupJob = new Job("Unload resources") {

				@Override
				protected IStatus run(IProgressMonitor monitor) {
					for (Resource resource : resources) {
						// System.out.println("unloading resource " + resource.getURI().toString());
						// long before = System.currentTimeMillis();
						resource.unload();
						// System.out.println("unloaded resource " + resource.getURI().toString());
						// System.out.println((System.currentTimeMillis() - before) + " ms");

					}
					return Status.OK_STATUS;
				}
			};
			cleanupJob.setPriority(Job.DECORATE);
			cleanupJob.setSystem(true);
			cleanupJob.schedule();
		}

		if (getActionBarContributor().getActiveEditor() == this) {
			getActionBarContributor().setActiveEditor(null);
		}

		if (this.propertySheetPage != null) {
			this.propertySheetPage.dispose();
		}

		if (this.treeViewer != null && this.selectionChangedListener != null) {
			this.treeViewer.removeSelectionChangedListener(this.selectionChangedListener);
			this.selectionChangedListener = null;
		}

		// dispose fonts
		this.editorConfiguration.setCustomFont(null);
		this.editorConfiguration.setCustomItalicFont(null);

		// avoid leaking editor
		this.editorConfiguration.clearEditor();

		JFaceResources.getFontRegistry().removeListener(this.fontChangeListener);

		this.orderedReferencePainter.dispose();

		super.dispose();
	}

	/** Open the selected element in the list of elements of the same metaclass */
	public void browse() {
		ISelection selection = getSelection();
		if (selection instanceof ITreeSelection) {
			ITreeSelection treeSelection = (ITreeSelection) selection;
			final Object selectedElement = treeSelection.getFirstElement();
			if (selectedElement instanceof EObject) {
				browseTo((EObject) selectedElement);
			}
		}
	}

	/** Open the given element in the list of elements of the same metaclass */
	public void browseTo(EObject element) {
		if (!(EMFUtil.isInFirstResource(element)))
			return;
		markNavigationLocation();
		String classQualifiedName = EMFUtil.getMetaclassQualifiedName(element.eClass());
		// Select the metaclass in the metaclass list viewer
		this.metaclassViewer.selectMetaclass(classQualifiedName);
		selectElement(element);
		markNavigationLocation();
	}

	/**
	 * Select the given element in the tree viewer. The element is looked for at the root and under
	 * BigListItemProviders in the case the elements are split.
	 * <p>
	 * This method relies on the elements staying the same for different calls to the getElements
	 * and getChildren methods on the item providers (this means these methods must not return new
	 * elements each time)
	 * */
	private void selectElement(Object selectedElement) {
		// deselect any previously selected elements
		this.treeViewer.setSelection(null);

		Object input = this.treeViewer.getInput();
		if (input instanceof MetaclassInstancesItemProvider) {
			MetaclassInstancesItemProvider metaclassListItemProvider = (MetaclassInstancesItemProvider) input;
			Collection<?> elements = metaclassListItemProvider.getElements(null);
			for (Object element : elements) {
				// if the elements are split
				if (element instanceof BigListItemProvider) {
					BigListItemProvider bigListItemProvider = (BigListItemProvider) element;
					Collection<?> subElements = bigListItemProvider.getChildren(null);
					for (Object subElement : subElements) {
						if (subElement == selectedElement) {
							TreePath treePath = new TreePath(new Object[] { element, subElement });
							this.treeViewer.setSelection(new TreeSelection(treePath), true);
							return;
						}
					}

				}

				if (element == selectedElement) {
					TreePath treePath = new TreePath(new Object[] { element });
					this.treeViewer.setSelection(new TreeSelection(treePath), true);
					return;
				}
			}
		}
	}

	/** Implements {@link INavigationLocationProvider} */
	public INavigationLocation createEmptyNavigationLocation() {
		return new EcoreEditorNavigationLocation(this);
	}

	/**
	 * Implements {@link INavigationLocationProvider}. Create a location that can be navigated back
	 * to, using the "back" button
	 */
	public INavigationLocation createNavigationLocation() {
		// navigation to the first element of the selection
		String firstSelectedMetaclass = this.metaclassViewer
				.getFirstSelectedMetaclassQualifiedName();
		return new EcoreEditorNavigationLocation(this, firstSelectedMetaclass,
				(ITreeSelection) this.treeViewer.getSelection());
	}

	/** Mark a location that can be navigated back to, using the "back" button */
	protected void markNavigationLocation() {
		getSite().getPage().getNavigationHistory().markLocation(this);
	}

	/**
	 * Restore a location (used for implementing history).
	 * 
	 * @param metaclassQualifiedName
	 *            the metaclass viewer's selection
	 * @param the
	 *            tree viewer's selection
	 */
	public void restoreLocation(String metaclassQualifiedName, ITreeSelection selection) {
		this.metaclassViewer.selectMetaclass(metaclassQualifiedName);
		this.treeViewer.setSelection(selection);
	}

	/** Refresh the tree viewer part of the editor */
	public void refreshTree() {
		if (!this.treeViewer.getTree().isDisposed()) {
			try {
				this.treeViewer.getTree().setRedraw(false);
				this.treeViewer.refresh();
			} finally {
				this.treeViewer.getTree().setRedraw(true);
			}
		}
	}

	public EditorConfiguration getEditorConfiguration() {
		return this.editorConfiguration;
	}

	private void openPropertiesView() {
		try {
			getEditorSite().getPage().showView(PROPERTY_VIEW_ID);
		} catch (PartInitException exception) {
			MoDiscoEditorPlugin.INSTANCE.log(exception);
		}
	}
}
