/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.editors;

import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.INavigationLocation;
import org.eclipse.ui.NavigationLocation;

/** A location in the Ecore editor, used for history navigation (back-forward buttons) */
public class EcoreEditorNavigationLocation extends NavigationLocation {

	/** The title of the editor */
	private String title;
	/** The metaclass selected in the metaclass viewer */
	private String selectedMetaclass;
	/** The selection in the tree viewer */
	private ITreeSelection treeSelection;

	/** Create an empty navigation location */
	public EcoreEditorNavigationLocation(IEditorPart editorPart) {
		super(editorPart);
		this.title = editorPart.getTitle();
		this.selectedMetaclass = null;
		this.treeSelection = null;
	}

	/**
	 * Create a navigation location with the selected metaclass and selected element(s).
	 * 
	 * @param editorPart
	 *            the editor
	 * @param selectedMetaclass
	 *            the metaclass selected in the metaclass viewer
	 * @param treeSelection
	 *            the selection in the tree viewer
	 */
	public EcoreEditorNavigationLocation(IEditorPart editorPart, String selectedMetaclass,
			ITreeSelection treeSelection) {
		super(editorPart);
		this.title = editorPart.getTitle();
		this.selectedMetaclass = selectedMetaclass;
		this.treeSelection = treeSelection;
	}

	@Override
	public String getText() {
		if (this.selectedMetaclass != null)
			return this.title + " - " + this.selectedMetaclass;
		else
			return this.title;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.INavigationLocation#mergeInto(org.eclipse.ui. INavigationLocation)
	 */
	public boolean mergeInto(INavigationLocation currentLocation) {
		if (currentLocation == null)
			return false;

		if (currentLocation instanceof EcoreEditorNavigationLocation) {
			EcoreEditorNavigationLocation location = (EcoreEditorNavigationLocation) currentLocation;

			if (!this.title.equals(location.title))
				return false;

			if (this.selectedMetaclass == location.selectedMetaclass) {
				location.treeSelection = this.treeSelection;
				return true;
			}
			// merge with empty location
			if (location.selectedMetaclass == null || location.selectedMetaclass == "") {
				location.selectedMetaclass = this.selectedMetaclass;
				location.treeSelection = this.treeSelection;
				return true;
			}
		}

		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.INavigationLocation#restoreLocation()
	 */
	public void restoreLocation() {
		// empty location
		if (this.selectedMetaclass == null)
			return;

		IEditorPart editorPart = getEditorPart();
		if (editorPart instanceof EcoreEditor) {
			EcoreEditor editor = (EcoreEditor) editorPart;
			editor.restoreLocation(this.selectedMetaclass, this.treeSelection);
		}
	}

	public void restoreState(IMemento memento) {
	}

	public void saveState(IMemento memento) {
	}

	public void update() {
	}

	@Override
	public void releaseState() {
		super.releaseState();
		// do not hang on to selection: permit GC to free model
		this.treeSelection = null;
	}
}
