/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.editors;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.gmt.modisco.common.editor.MoDiscoEditorPlugin;
import org.eclipse.gmt.modisco.common.editor.core.InstancesForMetaclasses;
import org.eclipse.gmt.modisco.common.editor.util.EMFUtil;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.swt.graphics.Font;

/** The configuration of a model editor, with access to the underlying data */
public class EditorConfiguration {

	/** Property name for the fontSizeDelta property */
	public static final String FONT_SIZE_DELTA_PROP = "fontSizeDelta";

	public enum MetaclassesSortMode {
		ByName, ByCount
	};

	private static final String SHOW_EMPTY_LINKS_SETTING = "showEmptyLinks";
	private static final String SHOW_DERIVED_LINKS_SETTING = "showDerivedLinks";
	private static final String SORT_INSTANCES_SETTING = "showInstances";
	private static final String SORT_LINKS_SETTING = "sortLinks";
	private static final String SORT_LINKS_BY_TYPE_SETTING = "sortLinksByType";
	private static final String SHOW_FULL_QUALIFIED_NAMES_SETTING = "showFullQualifiedNames";
	private static final String SHOW_MULTIPLICITY_SETTING = "showMultiplicity";
	private static final String SHOW_OPPOSITE_LINKS_SETTING = "showOppositeLinks";
	private static final String SHOW_CONTAINER_SETTING = "showContainer";
	private static final String METACLASSES_SORT_MODE = "metaclassesSortMode";
	private static final String SHOW_METACLASSES_FULL_QUALIFIED_NAMES_SETTING = "showMetaclassesFullQualifiedNames";
	private static final String SHOW_EMPTY_METACLASSES = "showEmptyMetaclasses";
	private static final String GROUP_BY_PACKAGE = "groupByPackage";
	private static final String DISPLAY_INSTANCES_OF_SUBCLASSES = "displayInstancesOfSubclasses";
	private static final String SHOW_DERIVATION_TREE = "showDerivationTree";
	private static final String SHOW_ATTRIBUTES = "showAttributes";
	private static final String SHOW_EMPTY_ATTRIBUTES = "showEmptyAttributes";
	private static final String SHOW_ORDERING = "showOrdering";
	private static final String FONT_SIZE_DELTA = "fontSizeDelta";

	/** Show empty links? */
	private boolean showEmptyLinks = false;
	/** Sort model elements by name in the tree? */
	private boolean sortInstances = false;
	/** Sort links by name in the tree? */
	private boolean sortLinks = false;
	/** Sort links by type in the tree? */
	private boolean sortLinksByType = false;
	/** Show derived links (links computed from other links) */
	private boolean showDerivedLinks = false;
	/** Show full qualified names or short names? */
	private boolean showFullQualifiedNames = false;
	/** Show full qualified names or short names of metaclasses? */
	private boolean showMetaclassesFullQualifiedNames = false;
	/** Show multiplicity? */
	private boolean showMultiplicity = false;
	/** Show opposite links? */
	private boolean showOppositeLinks = false;
	/** Show EMF eContainer as a virtual link? */
	private boolean showContainer = false;
	/** Show metaclasses which don't have any instance in the model? */
	private boolean showEmptyMetaclasses = false;
	/** Whether to display metaclasses grouped by package */
	private boolean groupByPackage = false;
	/**
	 * Whether to display instances on a given metaclass when the element has the type of a subclass
	 */
	private boolean displayInstancesOfSubclasses;
	/** Whether to show metaclasses as a derivation tree */
	private boolean showDerivationTree;
	/** How to sort metaclasses (in the metaclass pane) */
	private MetaclassesSortMode metaclassesSortMode = MetaclassesSortMode.ByName;
	/** Whether to show attributes as links */
	private boolean showAttributes;
	/** Whether to show empty attributes (null or empty list) */
	private boolean showEmptyAttributes;
	/** Whether to show ordering information */
	private boolean showOrdering;
	/** The size difference in points between the default font size and the user's preferred size */
	private int fontSizeDelta = 0;

	/** A font derived from the JFace default font, with a custom size defined by the user */
	private Font customFont = null;
	/** An italic font derived from the JFace default font, with a custom size defined by the user */
	private Font customItalicFont = null;

	/** All the classes in the resource set used by the editor */
	private Collection<EClass> allClasses;

	private ResourceSet resourceSet;
	private ComposedAdapterFactory adapterFactory;
	/** All the model elements by metaclass */
	private InstancesForMetaclasses instancesForMetaclasses;
	private ComposedAdapterFactory adapterFactoryWithRegistry;
	private EcoreEditor editor;

	private List<PropertyChangeListener> listeners = new ArrayList<PropertyChangeListener>();

	public EditorConfiguration(EcoreEditor editor) {
		this.editor = editor;
	}

	public EcoreEditor getEditor() {
		return this.editor;
	}

	public boolean isShowEmptyLinks() {
		return this.showEmptyLinks;
	}

	public void setShowEmptyLinks(boolean showEmptyLinks) {
		this.showEmptyLinks = showEmptyLinks;
	}

	public boolean isSortInstances() {
		return this.sortInstances;
	}

	public void setSortInstances(boolean sortInstances) {
		this.sortInstances = sortInstances;
	}

	public boolean isShowDerivedLinks() {
		return this.showDerivedLinks;
	}

	public void setShowDerivedLinks(boolean showDerivedLinks) {
		this.showDerivedLinks = showDerivedLinks;
	}

	public boolean isShowFullQualifiedNames() {
		return this.showFullQualifiedNames;
	}

	public void setShowFullQualifiedNames(boolean showFullQualifiedNames) {
		this.showFullQualifiedNames = showFullQualifiedNames;
	}

	public boolean isShowMultiplicity() {
		return this.showMultiplicity;
	}

	public void setShowMultiplicity(boolean showMultiplicity) {
		this.showMultiplicity = showMultiplicity;
	}

	public boolean isShowOppositeLinks() {
		return this.showOppositeLinks;
	}

	public void setShowOppositeLinks(boolean showOppositeLinks) {
		this.showOppositeLinks = showOppositeLinks;
	}

	public boolean isShowContainer() {
		return this.showContainer;
	}

	public void setShowContainer(boolean showContainer) {
		this.showContainer = showContainer;
	}

	public MetaclassesSortMode getMetaclassesSortMode() {
		return this.metaclassesSortMode;
	}

	public void setMetaclassesSortMode(MetaclassesSortMode mode) {
		this.metaclassesSortMode = mode;
	}

	public boolean isShowMetaclassesFullQualifiedNames() {
		return this.showMetaclassesFullQualifiedNames;
	}

	public void setShowMetaclassesFullQualifiedNames(boolean value) {
		this.showMetaclassesFullQualifiedNames = value;
	}

	public void setSortLinks(boolean value) {
		this.sortLinks = value;
	}

	public boolean isSortLinks() {
		return this.sortLinks;
	}

	public void setSortLinksByType(boolean value) {
		this.sortLinksByType = value;
	}

	public boolean isSortLinksByType() {
		return this.sortLinksByType;
	}

	public void setShowEmptyMetaclasses(boolean value) {
		this.showEmptyMetaclasses = value;
	}

	public boolean isShowEmptyMetaclasses() {
		return this.showEmptyMetaclasses;
	}

	public void setGroupByPackage(boolean value) {
		this.groupByPackage = value;
	}

	public boolean isGroupByPackage() {
		return this.groupByPackage;
	}

	public void setDisplayInstancesOfSubclasses(boolean value) {
		this.displayInstancesOfSubclasses = value;
	}

	public boolean isDisplayInstancesOfSubclasses() {
		return this.displayInstancesOfSubclasses;
	}

	public void setShowDerivationTree(boolean value) {
		this.showDerivationTree = value;
	}

	public boolean isShowDerivationTree() {
		return this.showDerivationTree;
	}

	public void setShowAttributes(boolean value) {
		this.showAttributes = value;
	}

	public boolean isShowAttributes() {
		return this.showAttributes;
	}

	public void setShowEmptyAttributes(boolean value) {
		this.showEmptyAttributes = value;
	}

	public boolean isShowEmptyAttributes() {
		return this.showEmptyAttributes;
	}

	public void setShowOrdering(boolean value) {
		this.showOrdering = value;
	}

	public boolean isShowOrdering() {
		return this.showOrdering;
	}

	public void setFontSizeDelta(int value) {
		firePropertyChanged(FONT_SIZE_DELTA_PROP, this.fontSizeDelta, value);
		this.fontSizeDelta = value;
	}

	public int getFontSizeDelta() {
		return this.fontSizeDelta;
	}

	public void setCustomFont(Font customFont) {
		if (this.customFont != null && this.customFont != JFaceResources.getDialogFont()
				&& this.customFont != customFont) {
			this.customFont.dispose();
		}
		this.customFont = customFont;
	}

	public Font getCustomFont() {
		return this.customFont;
	}

	public void setCustomItalicFont(Font customItalicFont) {
		if (this.customItalicFont != null
				&& this.customItalicFont != JFaceResources.getDialogFont()
				&& this.customItalicFont != customItalicFont) {
			this.customItalicFont.dispose();
		}
		this.customItalicFont = customItalicFont;
	}

	public Font getCustomItalicFont() {
		return this.customItalicFont;
	}

	/**
	 * Set the resource set underlying the editor, so that the list of metaclasses can be computed
	 * once and for all.
	 */
	public void setResourceSet(ResourceSet resourceSet) {
		this.resourceSet = resourceSet;
		this.allClasses = EMFUtil.findAllClasses(resourceSet);
	}

	public ResourceSet getResourceSet() {
		return this.resourceSet;
	}

	/** Returns all the classes in the resource set used by the editor */
	public Collection<EClass> getAllClasses() {
		return this.allClasses;
	}

	public void setAdapterFactory(ComposedAdapterFactory adapterFactory) {
		this.adapterFactory = adapterFactory;
	}

	public void setInstancesForMetaclasses(InstancesForMetaclasses instancesForMetaclasses) {
		this.instancesForMetaclasses = instancesForMetaclasses;
	}

	public InstancesForMetaclasses getInstancesForMetaclasses() {
		return this.instancesForMetaclasses;
	}

	/**
	 * Return the composed adapter factory that is used to adapt all the elements to show them in
	 * the editor's viewers
	 */
	public ComposedAdapterFactory getAdapterFactory() {
		return this.adapterFactory;
	}

	public void setAdapterFactoryWithRegistry(ComposedAdapterFactory adapterFactoryWithRegistry) {
		this.adapterFactoryWithRegistry = adapterFactoryWithRegistry;
	}

	public ComposedAdapterFactory getAdapterFactoryWithRegistry() {
		return this.adapterFactoryWithRegistry;
	}

	/** Load user settings */
	public void load() {
		IDialogSettings dialogSettings = MoDiscoEditorPlugin.getPlugin().getDialogSettings();

		this.showEmptyLinks = getPreference(SHOW_EMPTY_LINKS_SETTING, true, dialogSettings);
		this.showDerivedLinks = getPreference(SHOW_DERIVED_LINKS_SETTING, true, dialogSettings);
		this.sortInstances = getPreference(SORT_INSTANCES_SETTING, false, dialogSettings);
		this.sortLinks = getPreference(SORT_LINKS_SETTING, false, dialogSettings);
		this.sortLinksByType = getPreference(SORT_LINKS_BY_TYPE_SETTING, false, dialogSettings);
		this.showFullQualifiedNames = getPreference(SHOW_FULL_QUALIFIED_NAMES_SETTING, false,
				dialogSettings);
		this.showMultiplicity = getPreference(SHOW_MULTIPLICITY_SETTING, false, dialogSettings);
		this.showOppositeLinks = getPreference(SHOW_OPPOSITE_LINKS_SETTING, false, dialogSettings);
		this.showContainer = getPreference(SHOW_CONTAINER_SETTING, true, dialogSettings);
		this.showMetaclassesFullQualifiedNames = getPreference(
				SHOW_METACLASSES_FULL_QUALIFIED_NAMES_SETTING, false, dialogSettings);
		this.showEmptyMetaclasses = getPreference(SHOW_EMPTY_METACLASSES, false, dialogSettings);
		this.groupByPackage = getPreference(GROUP_BY_PACKAGE, false, dialogSettings);
		this.displayInstancesOfSubclasses = getPreference(DISPLAY_INSTANCES_OF_SUBCLASSES, false,
				dialogSettings);
		this.showDerivationTree = getPreference(SHOW_DERIVATION_TREE, false, dialogSettings);
		this.showAttributes = getPreference(SHOW_ATTRIBUTES, true, dialogSettings);
		this.showEmptyAttributes = getPreference(SHOW_EMPTY_ATTRIBUTES, true, dialogSettings);
		this.showOrdering = getPreference(SHOW_ORDERING, false, dialogSettings);
		this.fontSizeDelta = getPreference(FONT_SIZE_DELTA, 0, dialogSettings);

		String metaclassesSortMode = dialogSettings.get(METACLASSES_SORT_MODE);
		if (metaclassesSortMode != null) {
			this.metaclassesSortMode = MetaclassesSortMode.valueOf(metaclassesSortMode);
		}
	}

	private boolean getPreference(String settingKey, boolean defaultValue,
			IDialogSettings dialogSettings) {
		String value = dialogSettings.get(settingKey);
		if (value == null)
			return defaultValue;
		return Boolean.valueOf(value);
	}

	private int getPreference(String settingKey, int defaultValue, IDialogSettings dialogSettings) {
		String value = dialogSettings.get(settingKey);
		if (value == null)
			return defaultValue;
		try {
			return Integer.parseInt(value);
		} catch (NumberFormatException e) {
			return defaultValue;
		}
	}

	/** Save user settings */
	public void save() {
		IDialogSettings dialogSettings = MoDiscoEditorPlugin.getPlugin().getDialogSettings();
		dialogSettings.put(SHOW_EMPTY_LINKS_SETTING, this.showEmptyLinks);
		dialogSettings.put(SHOW_DERIVED_LINKS_SETTING, this.showDerivedLinks);
		dialogSettings.put(SORT_INSTANCES_SETTING, this.sortInstances);
		dialogSettings.put(SORT_LINKS_SETTING, this.sortLinks);
		dialogSettings.put(SORT_LINKS_BY_TYPE_SETTING, this.sortLinksByType);
		dialogSettings.put(SHOW_FULL_QUALIFIED_NAMES_SETTING, this.showFullQualifiedNames);
		dialogSettings.put(SHOW_MULTIPLICITY_SETTING, this.showMultiplicity);
		dialogSettings.put(SHOW_OPPOSITE_LINKS_SETTING, this.showOppositeLinks);
		dialogSettings.put(SHOW_CONTAINER_SETTING, this.showContainer);
		dialogSettings.put(METACLASSES_SORT_MODE, this.metaclassesSortMode.name());
		dialogSettings.put(SHOW_METACLASSES_FULL_QUALIFIED_NAMES_SETTING,
				this.showMetaclassesFullQualifiedNames);
		dialogSettings.put(SHOW_EMPTY_METACLASSES, this.showEmptyMetaclasses);
		dialogSettings.put(GROUP_BY_PACKAGE, this.groupByPackage);
		dialogSettings.put(DISPLAY_INSTANCES_OF_SUBCLASSES, this.displayInstancesOfSubclasses);
		dialogSettings.put(SHOW_DERIVATION_TREE, this.showDerivationTree);
		dialogSettings.put(SHOW_ATTRIBUTES, this.showAttributes);
		dialogSettings.put(SHOW_EMPTY_ATTRIBUTES, this.showEmptyAttributes);
		dialogSettings.put(SHOW_ORDERING, this.showOrdering);
		dialogSettings.put(FONT_SIZE_DELTA, this.fontSizeDelta);
	}

	/** Clear the editor field */
	public void clearEditor() {
		this.editor = null;
	}

	/** @return the metamodel name or <code>null</code> if not found */
	public String getMetamodelName() {
		String nsURI = getMetamodelURI();
		if (nsURI != null) {
			String[] parts = nsURI.split("/");
			if (parts.length > 0) {
				return parts[parts.length - 1];
			}
		}
		return null;
	}

	/**
	 * @return the NsURI of the metamodel corresponding to an element in the first resource or
	 *         <code>null</code> if not found
	 */
	public String getMetamodelURI() {
		EList<Resource> resources = this.resourceSet.getResources();
		if (resources.size() > 0) {
			Resource firstResource = resources.get(0);
			EList<EObject> contents = firstResource.getContents();
			if (contents.size() > 0) {
				EObject firstElement = contents.get(0);
				EClass eClass = firstElement.eClass();
				if (eClass != null) {
					EPackage ePackage = eClass.getEPackage();
					if (ePackage != null) {
						return ePackage.getNsURI();
					}
				}
			}
		}

		return null;
	}

	public void addPropertyChangeListener(PropertyChangeListener listener) {
		this.listeners.add(listener);
	}

	public void removePropertyChangeListener(PropertyChangeListener listener) {
		this.listeners.remove(listener);
	}

	private void firePropertyChanged(String propertyName, int oldValue, int newValue) {
		for (PropertyChangeListener listener : this.listeners) {
			PropertyChangeEvent event = new PropertyChangeEvent(this, propertyName, oldValue,
					newValue);
			listener.propertyChange(event);
		}
	}
}
