/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.editors;

import java.util.List;
import java.util.Set;
import java.util.TreeMap;
import java.util.Map.Entry;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.gmt.modisco.common.editor.adapters.MetaclassListItemProvider;
import org.eclipse.gmt.modisco.common.editor.adapters.MetaclassListItemProvider.ChangeListener;
import org.eclipse.gmt.modisco.common.editor.util.EMFUtil;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.ListViewer;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;

/** The list viewer for displaying the list of metaclasses that appears in the model editor. */
public class MetaclassListViewer extends ListViewer implements ChangeListener {

	/** Model elements sorted by EClass (full qualified name) */
	private TreeMap<String, MetaclassListItemProvider> modelElements;
	/** The resource set containing all the resources of the model */
	private final ResourceSet resourceSet;
	/** the adapter factory used to create adapters */
	private final AdapterFactory adapterFactory;
	/** the configuration of the model editor */
	private final EditorConfiguration editorConfiguration;

	/**
	 * @param parent
	 *            the composite in which this viewer must be created
	 * @param style
	 *            the SWT style bits
	 * @param resourceSet
	 *            the resource set containing all the model elements
	 * @param adapterFactory
	 *            the adapter factory, used to create adapters
	 * @param editorConfiguration
	 *            the configuration of the model editor, which is passed on to the
	 *            {@link MetaclassListViewer}s.
	 */
	public MetaclassListViewer(Composite parent, int style, ResourceSet resourceSet,
			AdapterFactory adapterFactory, EditorConfiguration editorConfiguration) {
		super(parent, style);
		this.resourceSet = resourceSet;
		this.adapterFactory = adapterFactory;
		this.editorConfiguration = editorConfiguration;
		initViewer();
		initModel();
	}

	/**
	 * Initialize the <code>modelElements</code> list with model elements from the resource set.
	 */
	private void initModel() {
		/* TreeMap (red-black tree), automatically sorted by String */
		this.modelElements = new TreeMap<String, MetaclassListItemProvider>();

		EList<Resource> resources = this.resourceSet.getResources();
		if (resources.size() == 0)
			return;

		// show only the first resource instances
		Resource resource = resources.get(0);
		TreeIterator<EObject> allContents = resource.getAllContents();
		while (allContents.hasNext()) {
			EObject eObject = allContents.next();
			addModelElement(eObject, false);
		}

		this.setInput(this.modelElements);
	}

	/** Select the class corresponding to the root element (the 'model' element) */
	public void selectRootElement() {
		this.setSelection(null);
		EList<Resource> resources = this.resourceSet.getResources();
		if (resources.size() > 0) {
			EList<EObject> contents = resources.get(0).getContents();
			if (contents.size() > 0) {
				EObject object = contents.get(0);
				String classFullQualifiedName = EMFUtil.getMetaclassQualifiedName(object.eClass());
				// the list viewer's model elements are Entries
				Set<Entry<String, MetaclassListItemProvider>> entrySet = this.modelElements.entrySet();
				for (Entry<String, MetaclassListItemProvider> entry : entrySet) {
					if (entry.getKey().equals(classFullQualifiedName)) {
						this.setSelection(new StructuredSelection(entry), true);
						break;
					}
				}
			}
		}
	}

	/** Select the metaclass with the given full qualified name */
	public void selectMetaclassName(String classFullQualifiedName) {
		// the list viewer's model elements are Entries
		Set<Entry<String, MetaclassListItemProvider>> entrySet = this.modelElements.entrySet();
		for (Entry<String, MetaclassListItemProvider> entry : entrySet) {
			if (entry.getKey().equals(classFullQualifiedName)) {
				this.setSelection(new StructuredSelection(entry), true);
				break;
			}
		}
	}

	/** Return the currently selected metaclass qualified name, or an empty String if none */
	public String getSelectedMetaclassQualifiedName() {
		ISelection selection = this.getSelection();
		if (selection instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			// single select
			Object selectedElement = structuredSelection.getFirstElement();
			if (selectedElement instanceof Entry) {
				// type erasure on generics
				@SuppressWarnings("unchecked")
				Entry entry = (Entry) selectedElement;
				return (String) entry.getKey();
			}
		}
		return "";
	}

	/** Initialize the viewer with a content and label provider */
	private void initViewer() {

		this.setContentProvider(new IStructuredContentProvider() {
			public Object[] getElements(Object inputElement) {
				// unchecked cast, due to type erasure on generics
				@SuppressWarnings("unchecked")
				TreeMap<String, List<EObject>> modelElements = (TreeMap<String, List<EObject>>) inputElement;
				return modelElements.entrySet().toArray();
			}

			public void dispose() {
			}

			public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			}
		});

		this.setLabelProvider(new LabelProvider() {
			@Override
			public Image getImage(Object element) {
				return null;
			}

			@Override
			public String getText(Object element) {
				// unchecked cast, due to type erasure on generics
				@SuppressWarnings("unchecked")
				Entry<String, MetaclassListItemProvider> entry = (Entry<String, MetaclassListItemProvider>) element;

				String name = entry.getKey();
				// only show the short name of the class (instead of the full qualified name)
				if (!MetaclassListViewer.this.editorConfiguration.isShowFullQualifiedNames()) {
					int lastDot = name.lastIndexOf('.');
					if (lastDot != -1) {
						name = name.substring(lastDot + 1);
					}
				}

				return name + " (" + entry.getValue().size() + ")";
			}
		});
	}

	/** A listener for model changes : instance creation or removal */
	public interface ModelChangeListener {
		void modelChanged();
	}

	/** A listener for model changes */
	private ModelChangeListener listener = null;

	/** Set the listener for model changes (instance creation or removal) */
	public void setListener(ModelChangeListener listener) {
		this.listener = listener;
	}

	/**
	 * This notification is sent by the {@link MetaclassListItemProvider}s when one of their objects
	 * reports a change.
	 * <p>
	 * Respond to model elements creation and removal by updating the corresponding
	 * {@link MetaclassListItemProvider}'s list, and firing the "modelChanged" event to a possible
	 * listener.
	 */
	public void notifyChanged(Notification msg) {
		int eventType = msg.getEventType();
		switch (eventType) {
		case Notification.ADD: {
			Object featureObj = msg.getFeature();
			if (featureObj instanceof EReference) {
				EReference feature = (EReference) featureObj;
				if (!feature.isContainment())
					return;
			}
			Object newValue = msg.getNewValue();
			if (newValue instanceof EObject) {
				addModelElement((EObject) newValue, true);
			}
			modelChanged();
			break;
		}
		case Notification.REMOVE: {
			Object featureObj = msg.getFeature();
			if (featureObj instanceof EReference) {
				EReference feature = (EReference) featureObj;
				if (!feature.isContainment())
					return;
			}
			Object oldValue = msg.getOldValue();
			if (oldValue instanceof EObject) {
				removeModelElement((EObject) oldValue);
			}
			modelChanged();
			break;
		}
		case Notification.ADD_MANY: {
			Object newValue = msg.getNewValue();
			if (newValue instanceof EList) {
				EList<?> eList = (EList<?>) newValue;
				for (Object object : eList) {
					if (object instanceof EObject) {
						addModelElement((EObject) object, true);
					}
				}
			}
			modelChanged();
			break;
		}
		case Notification.REMOVE_MANY: {
			Object newValue = msg.getNewValue();
			if (newValue instanceof EList) {
				EList<?> eList = (EList<?>) newValue;
				for (Object object : eList) {
					if (object instanceof EObject) {
						removeModelElement((EObject) object);
					}
				}
			}
			modelChanged();
			break;
		}
			// features that have a multiplicity of 1
		case Notification.SET: {
			Object newValue = msg.getNewValue();
			Object feature = msg.getFeature();
			if (feature instanceof EReference) {
				EReference reference = (EReference) feature;
				/*
				 * setting a reference should add or remove elements only if the reference is a
				 * composition link
				 */
				if (!reference.isContainment()) {
					break;
				}

				if (newValue == null) {
					Object oldValue = msg.getOldValue();
					if (oldValue instanceof EObject) {
						removeModelElement((EObject) oldValue);
						modelChanged();
					}
				} else if (newValue instanceof EObject) {
					addModelElement((EObject) newValue, true);
					modelChanged();
				}
			}
			break;
		}
		case Notification.UNSET: {
			Object oldValue = msg.getOldValue();
			Object feature = msg.getFeature();
			if (feature instanceof EReference) {
				EReference reference = (EReference) feature;
				/*
				 * unsetting a reference should remove the element only if the reference is a
				 * composition link
				 */
				if (!reference.isContainment()) {
					break;
				}
			}
			if (oldValue instanceof EObject) {
				removeModelElement((EObject) oldValue);
			}
			modelChanged();
			break;
		}
		}
	}

	/**
	 * Delayed viewer refresh and notification, to avoid refreshing the views 100 times when 100
	 * items are added or deleted in a single operation (or in quick succession).
	 */
	private void modelChanged() {
		if (this.refreshJob == null) {
			this.refreshJob = new Job("Refresh model viewers") {
				@Override
				protected IStatus run(IProgressMonitor monitor) {
					Display.getDefault().syncExec(new Runnable() {
						public void run() {
							MetaclassListViewer.this.refresh();
							fireModelChanged();
						}
					});
					return Status.OK_STATUS;
				}
			};
		}

		/*
		 * if modelChanged is called again before the previous operation could finish, then cancel
		 * it and schedule a new refresh job.
		 */
		this.refreshJob.cancel();
		this.refreshJob.schedule(100);
	}

	/** A job to allow optimal refreshing of the viewers */
	private Job refreshJob = null;

	/**
	 * Add a new element to the MetaclassListItemProvider corresponding to its metaclass.
	 * <p>
	 * Create a new MetaclassListItemProvider if necessary.
	 * 
	 * @param element
	 *            the element to add
	 * @param recursively
	 *            whether to also add elements contained in the given element
	 */
	private void addModelElement(EObject element, boolean recursively) {
		String classQualifiedName = EMFUtil.getMetaclassQualifiedName(element.eClass());

		MetaclassListItemProvider listItemProvider = this.modelElements.get(classQualifiedName);

		if (listItemProvider == null) {
			listItemProvider = new MetaclassListItemProvider(this.adapterFactory, this.editorConfiguration,
					element.eClass());
			listItemProvider.setListener(this);
			this.modelElements.put(classQualifiedName, listItemProvider);
		}
		listItemProvider.add(element);

		if (recursively) {
			// also add the contained elements recursively
			EList<EObject> contents = element.eContents();
			for (EObject contained : contents) {
				addModelElement(contained, true);
			}
		}
	}

	/**
	 * Remove an element from the MetaclassListItemProvider corresponding to its class.
	 * <p>
	 * If the last element is removed from the list, then the MetaclassListItemProvider is removed
	 * from the modelElements.
	 * <p>
	 * Also removes elements contained in the given element.
	 * 
	 * @param element
	 *            the element to remove
	 */
	private void removeModelElement(EObject element) {
		String classQualifiedName = EMFUtil.getMetaclassQualifiedName(element.eClass());
		MetaclassListItemProvider list = this.modelElements.get(classQualifiedName);
		if (list != null) {
			list.remove(element);
			if (list.isEmpty()) {
				// removing selected element
				if (getSelectedMetaclassQualifiedName().equals(classQualifiedName)) {
					this.setSelection(null);
				}
				this.modelElements.remove(classQualifiedName);
			}
		}

		// also remove the contained elements recursively
		EList<EObject> contents = element.eContents();
		for (EObject contained : contents) {
			removeModelElement(contained);
		}
	}

	/** Fire a "model changed" event to the potential listener */
	private void fireModelChanged() {
		if (this.listener != null) {
			this.listener.modelChanged();
		}
	}
}