/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.editors;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.gmt.modisco.common.editor.MoDiscoEditorPlugin;
import org.eclipse.jface.dialogs.MessageDialog;

/** Listens to workspace changes to detect changed resources */
public class WorkspaceChangeListener implements IResourceChangeListener {

	/** Resources that have been removed since last activation. */
	protected Collection<Resource> removedResources = new ArrayList<Resource>();

	/** Resources that have been changed since last activation. */
	protected Collection<Resource> changedResources = new ArrayList<Resource>();

	/** Resources that have been saved. */
	protected Collection<Resource> savedResources = new ArrayList<Resource>();

	private final EcoreEditor editor;

	public WorkspaceChangeListener(EcoreEditor editor) {
		this.editor = editor;
	}

	public void resourceChanged(IResourceChangeEvent event) {
		IResourceDelta delta = event.getDelta();
		try {
			class ResourceDeltaVisitor implements IResourceDeltaVisitor {
				protected ResourceSet resourceSet = WorkspaceChangeListener.this.editor
						.getEditingDomain().getResourceSet();
				protected Collection<Resource> innerChangedResources = new ArrayList<Resource>();
				protected Collection<Resource> innerRemovedResources = new ArrayList<Resource>();

				public boolean visit(IResourceDelta delta) {
					if (delta.getResource().getType() == IResource.FILE) {
						if (delta.getKind() == IResourceDelta.REMOVED
								|| delta.getKind() == IResourceDelta.CHANGED
								&& delta.getFlags() != IResourceDelta.MARKERS) {
							Resource resource = this.resourceSet.getResource(URI.createURI(delta
									.getFullPath().toString()), false);
							if (resource != null) {
								if (delta.getKind() == IResourceDelta.REMOVED) {
									this.innerRemovedResources.add(resource);
								} else if (!WorkspaceChangeListener.this.savedResources
										.remove(resource)) {
									this.innerChangedResources.add(resource);
								}
							}
						}
					}

					return true;
				}

				public Collection<Resource> getChangedResources() {
					return this.innerChangedResources;
				}

				public Collection<Resource> getRemovedResources() {
					return this.innerRemovedResources;
				}
			}

			ResourceDeltaVisitor visitor = new ResourceDeltaVisitor();
			delta.accept(visitor);

			if (!visitor.getRemovedResources().isEmpty()) {
				this.removedResources.addAll(visitor.getRemovedResources());
				if (!this.editor.isDirty()) {
					this.editor.getSite().getShell().getDisplay().asyncExec(new Runnable() {
						public void run() {
							WorkspaceChangeListener.this.editor.getSite().getPage().closeEditor(
									WorkspaceChangeListener.this.editor, false);
						}
					});
				}
			}

			if (!visitor.getChangedResources().isEmpty()) {
				this.changedResources.addAll(visitor.getChangedResources());
				if (this.editor.getSite().getPage().getActiveEditor() == this.editor) {
					this.editor.getSite().getShell().getDisplay().asyncExec(new Runnable() {
						public void run() {
							handleActivate();
						}
					});
				}
			}
		} catch (CoreException exception) {
			MoDiscoEditorPlugin.INSTANCE.log(exception);
		}
	}

	/** Handles activation of the editor or its associated views. */
	protected void handleActivate() {
		this.editor.handleActivate();

		if (!this.removedResources.isEmpty()) {
			if (handleDirtyConflict()) {
				this.editor.getSite().getPage().closeEditor(this.editor, false);
			} else {
				this.removedResources.clear();
				this.changedResources.clear();
				this.savedResources.clear();
			}
		} else if (!this.changedResources.isEmpty()) {
			this.changedResources.removeAll(this.savedResources);
			handleChangedResources();
			this.changedResources.clear();
			this.savedResources.clear();
		}
	}

	/** Handles what to do with changed resources on activation. */
	protected void handleChangedResources() {
		if (!this.changedResources.isEmpty() && (!this.editor.isDirty() || handleDirtyConflict())) {
			this.editor.handleChangedResources(this.changedResources);
		}
	}

	/** Shows a dialog that asks if conflicting changes should be discarded. */
	protected boolean handleDirtyConflict() {
		return MessageDialog.openQuestion(this.editor.getSite().getShell(),
				getString("_UI_FileConflict_label"), getString("_WARN_FileConflict"));
	}

	/** This looks up a string in the plugin's plugin.properties file. */
	private static String getString(String key) {
		return MoDiscoEditorPlugin.INSTANCE.getString(key);
	}

	public void addSavedResource(Resource resource) {
		this.savedResources.add(resource);
	}
}
