/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.editors.table;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.gmt.modisco.common.editor.MoDiscoEditorPlugin;
import org.eclipse.gmt.modisco.common.editor.adapters.LinkItemProvider;
import org.eclipse.gmt.modisco.common.editor.editors.EcoreEditor;
import org.eclipse.gmt.modisco.common.editor.editors.EditorConfiguration;
import org.eclipse.gmt.modisco.common.editor.editors.table.ColumnDescription.Type;
import org.eclipse.gmt.modisco.common.editor.util.ImageProvider;
import org.eclipse.gmt.modisco.common.editor.util.Util;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.MenuDetectEvent;
import org.eclipse.swt.events.MenuDetectListener;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.EditorPart;

public class TableEditor extends EditorPart {

	private static final int MAX_INITIAL_COLUMN_WIDTH = 300;
	public static final String EDITOR_ID = "org.eclipse.gmt.modisco.common.editor.table.editorID";
	/**
	 * The number of items above which a virtual table is used. The virtual style is not always used
	 * because it prevents correct packing of columns (since the elements' labels are not computed
	 * in advance).
	 */
	private static final int USE_VIRTUAL_THRESHOLD = 1000;

	/** The data key for columns that are hidden because they are empty */
	private static final String KEY_EMPTY_HIDDEN = "emptyHidden";

	/**
	 * The data key for columns that are hidden because the feature (attribute or reference) they
	 * represent is not common to all the elements
	 */
	private static final String KEY_NOT_COMMON_HIDDEN = "notCommonHidden";

	protected TableViewer tableViewer;
	private TableEditorInput tableEditorInput;
	private TableEditorLabelProvider tableEditorLabelProvider;
	private List<ColumnDescription> columnsConfiguration;
	private TableEditorToolBar toolBar;
	private TableEditorContentProvider tableEditorContentProvider;

	@Override
	public void init(IEditorSite site, IEditorInput input) throws PartInitException {
		if (input instanceof TableEditorInput) {
			this.tableEditorInput = (TableEditorInput) input;
			setSite(site);
			setInput(this.tableEditorInput);
			setPartName("Table Viewer");
		} else {
			throw new PartInitException("Input should be of type TableEditorInput");
		}
	}

	@Override
	public void createPartControl(Composite parent) {

		Composite editorComposite = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout(1, false);
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 2;
		editorComposite.setLayout(gridLayout);

		createToolBar(editorComposite);
		this.toolBar.setLabelText(this.tableEditorInput.getDescription());

		GridData tableGridData = new GridData();
		tableGridData.grabExcessHorizontalSpace = true;
		tableGridData.grabExcessVerticalSpace = true;
		tableGridData.horizontalAlignment = SWT.FILL;
		tableGridData.verticalAlignment = SWT.FILL;

		int virtualFlag = 0;
		if (this.tableEditorInput.getElements().size() > USE_VIRTUAL_THRESHOLD) {
			virtualFlag = SWT.VIRTUAL;
		}

		this.tableViewer = new TableViewer(editorComposite, SWT.SINGLE | SWT.H_SCROLL
				| SWT.V_SCROLL | SWT.FULL_SELECTION | virtualFlag);
		this.tableViewer.getTable().setLayoutData(tableGridData);

		// use the same font size as in the main editor
		this.tableViewer.getTable().setFont(
				this.tableEditorInput.getEditorConfiguration().getCustomFont());
		this.tableViewer.getTable().setLinesVisible(true);
		this.tableViewer.getTable().setHeaderVisible(true);

		this.tableEditorContentProvider = new TableEditorContentProvider();
		this.columnsConfiguration = createColumnConfiguration(false);

		this.tableEditorLabelProvider = new TableEditorLabelProvider(this.columnsConfiguration,
				this.tableEditorInput.getEditorConfiguration());

		createColumns(this.columnsConfiguration);

		this.tableViewer.setContentProvider(this.tableEditorContentProvider);
		this.tableViewer.setLabelProvider(this.tableEditorLabelProvider);
		this.tableViewer.setInput(this.tableEditorInput);

		createContextMenu();
		registerOpenListeners();
		registerCloseWhenMainEditorClosed();

		packColumns();
	}

	private void createToolBar(Composite parent) {
		Composite toolBarComposite = new Composite(parent, SWT.NONE);
		this.toolBar = new TableEditorToolBar(toolBarComposite, this);
	}

	private void registerCloseWhenMainEditorClosed() {
		EcoreEditor editor = this.tableEditorInput.getEditorConfiguration().getEditor();
		if (editor == null)
			return;

		editor.getEditorSite().getPage().addPartListener(new IPartListener() {

			public void partClosed(IWorkbenchPart part) {
				EcoreEditor editor = TableEditor.this.tableEditorInput.getEditorConfiguration()
						.getEditor();
				if (editor != null && part == editor) {
					TableEditor.this.getSite().getPage().closeEditor(TableEditor.this, false);
				}
			}

			public void partOpened(IWorkbenchPart part) {
			}

			public void partDeactivated(IWorkbenchPart part) {
			}

			public void partBroughtToTop(IWorkbenchPart part) {
			}

			public void partActivated(IWorkbenchPart part) {
			}
		});
	}

	private void createContextMenu() {
		final Menu menu = new Menu(this.tableViewer.getTable());

		this.tableViewer.getTable().addMenuDetectListener(new MenuDetectListener() {
			public void menuDetected(MenuDetectEvent e) {
				Point clickedPoint = new Point(e.x, e.y);

				Point widgetRelativeClickedPoint = Display.getDefault().map(null,
						TableEditor.this.tableViewer.getTable(), clickedPoint);

				ViewerCell cell = TableEditor.this.tableViewer.getCell(widgetRelativeClickedPoint);
				final int column = (cell == null) ? 0 : cell.getColumnIndex();

				// clear dynamic items
				MenuItem[] menuItems = menu.getItems();
				for (MenuItem menuItem : menuItems) {
					menuItem.dispose();
				}

				ColumnDescription columnDescription = TableEditor.this.columnsConfiguration
						.get(column);
				EObject element = null;
				final EObject mainElement = getSelectedElement();
				if (mainElement == null)
					return;

				if (columnDescription.getType() == Type.REFERENCE) {
					boolean oneElement = true;
					if (columnDescription.getReference().isMany()) {
						// test whether the many-valued list has only 1 element
						@SuppressWarnings("unchecked")
						EList<EObject> list = (EList<EObject>) mainElement.eGet(columnDescription
								.getReference());
						oneElement = list.size() == 1;
					}

					if (oneElement) {
						element = findElementToOpenInMainEditor(column);
						if (element != null) {
							createBrowseMenuItemFor(element);
						}
					} else {
						createViewInTableMenuItemFor(mainElement, columnDescription.getReference());
					}
				} else if (columnDescription.getType() == Type.ECONTAINER) {
					element = mainElement.eContainer();
					if (element != null) {
						createBrowseMenuItemFor(element);
					}
				}

				if (element != mainElement) {
					createBrowseMenuItemFor(mainElement);
				}

				menu.setLocation(clickedPoint);
				menu.setVisible(true);
			}

			private void createBrowseMenuItemFor(final EObject element) {
				if (element == null)
					return;

				String text = TableEditor.this.tableEditorLabelProvider.getTextFor(element);
				// Image image = tableEditorLabelProvider.getImageFor(element);

				MenuItem browseMenuItem = new MenuItem(menu, SWT.PUSH);
				browseMenuItem.setText("Browse " + Util.truncateBeforeNewline(text));
				// browseMenuItem.setImage(image);

				browseMenuItem.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(SelectionEvent e) {
						openElementInMainEditor(element);
					}
				});
			}

			private void createViewInTableMenuItemFor(final EObject parentElement,
					final EReference reference) {
				@SuppressWarnings("unchecked")
				final EList<EObject> elements = (EList<EObject>) parentElement.eGet(reference);

				MenuItem menuItem = new MenuItem(menu, SWT.PUSH);
				menuItem.setText(MoDiscoEditorPlugin.INSTANCE
						.getString("_UI_OpenTableEditorOnSelection_menu_item"));

				if (elements.size() == 0)
					menuItem.setEnabled(false);

				menuItem.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(SelectionEvent e) {
						EditorConfiguration editorConfiguration = TableEditor.this.tableEditorInput
								.getEditorConfiguration();
						String description = getEditorDescriptionFor(parentElement, reference,
								editorConfiguration);
						openOn(TableEditor.this.tableEditorInput.getEditorConfiguration(),
								elements, description);
					}
				});
			}
		});
	}

	/**
	 * Registers listeners used to open an element in the main editor by double-click or
	 * &lt;Enter&gt;
	 */
	private void registerOpenListeners() {
		// double-click
		this.tableViewer.getTable().addMouseListener(new MouseAdapter() {
			@Override
			public void mouseDoubleClick(MouseEvent e) {
				ViewerCell cell = TableEditor.this.tableViewer.getCell(new Point(e.x, e.y));
				int column;
				if (cell == null)
					column = 0;
				else
					column = cell.getColumnIndex();

				openSelectionInMainEditor(column);
			}
		});

		// enter
		this.tableViewer.getTable().addKeyListener(new KeyAdapter() {
			@Override
			public void keyPressed(KeyEvent e) {
				if (e.keyCode == SWT.CR) {
					openSelectionInMainEditor(0);
				}
			}
		});
	}

	/** pack the table's columns with a limit on max width */
	private void packColumns() {
		TableColumn[] tableColumns = this.tableViewer.getTable().getColumns();
		for (TableColumn tableColumn : tableColumns) {
			tableColumn.pack();
			if (tableColumn.getWidth() > MAX_INITIAL_COLUMN_WIDTH)
				tableColumn.setWidth(MAX_INITIAL_COLUMN_WIDTH);
		}
	}

	/**
	 * Create a configuration about what to display in columns
	 * 
	 * @param hideEmptyColumns
	 *            whether to hide empty columns
	 */
	private List<ColumnDescription> createColumnConfiguration(boolean hideEmptyColumns) {
		// TODO: hideEmptyColumns unused
		ArrayList<ColumnDescription> columns = new ArrayList<ColumnDescription>();
		columns.add(new ColumnDescription(ColumnDescription.Type.DEFAULT_LABEL));
		columns.add(new ColumnDescription(ColumnDescription.Type.METACLASS_NAME));
		columns.add(new ColumnDescription(ColumnDescription.Type.ECONTAINER));

		for (EAttribute attribute : this.tableEditorInput.getAttributes()) {
			if (!hideEmptyColumns || !isEmpty(attribute)) {
				columns.add(new ColumnDescription(attribute));
			}
		}

		// first, non-derived references
		for (EReference reference : this.tableEditorInput.getReferences()) {
			if (!reference.isDerived()) {
				if (!hideEmptyColumns || !isEmpty(reference)) {
					columns.add(new ColumnDescription(reference));
				}
			}
		}

		// then derived references
		for (EReference reference : this.tableEditorInput.getReferences()) {
			if (reference.isDerived()) {
				if (!hideEmptyColumns || !isEmpty(reference)) {
					columns.add(new ColumnDescription(reference));
				}
			}
		}

		return columns;
	}

	/** Whether no element has this reference set */
	private boolean isEmpty(EReference reference) {
		Object[] elements = this.tableEditorContentProvider.getElements(this.tableEditorInput);
		for (Object element : elements) {
			if (element instanceof EObject) {
				EObject eObject = (EObject) element;
				if (eObject.eClass().getEAllReferences().contains(reference)) {
					Object value = eObject.eGet(reference);
					if (reference.isMany()) {
						@SuppressWarnings("unchecked")
						EList<EObject> list = (EList<EObject>) value;
						if (!list.isEmpty())
							return false;
					} else {
						if (value != null)
							return false;
					}
				}
			}
		}
		return true;
	}

	/** Whether no element has this attribute set */
	private boolean isEmpty(EAttribute attribute) {
		Object[] elements = this.tableEditorContentProvider.getElements(this.tableEditorInput);
		for (Object element : elements) {
			if (element instanceof EObject) {
				EObject eObject = (EObject) element;
				if (eObject.eClass().getEAllAttributes().contains(attribute)) {
					Object value = eObject.eGet(attribute);
					if (attribute.isMany()) {
						@SuppressWarnings("unchecked")
						EList<EObject> list = (EList<EObject>) value;
						if (!list.isEmpty())
							return false;
					} else {
						if (value != null)
							return false;
					}
				}
			}
		}
		return true;
	}

	private void createColumns(List<ColumnDescription> columns) {
		int columnIndex = 0;
		for (ColumnDescription columnDescription : columns) {
			TableViewerColumn column = new TableViewerColumn(this.tableViewer, SWT.NONE);
			TableColumn tableColumn = column.getColumn();
			tableColumn.setWidth(100);
			tableColumn.setResizable(true);
			tableColumn.setMoveable(true);
			column.getColumn().setData(columnDescription);

			switch (columnDescription.getType()) {
			// shows the same thing as in model browser tree
			case DEFAULT_LABEL:
				tableColumn.setText("[Label]");
				tableColumn.setWidth(125);
				break;
			case METACLASS_NAME:
				tableColumn.setText("[Metaclass]");
				break;
			case ECONTAINER:
				tableColumn.setText("/eContainer");
				break;
			case ATTRIBUTE:
				EAttribute attribute = columnDescription.getAttribute();
				if (attribute.isDerived()) {
					tableColumn.setText("/" + attribute.getName());
				} else {
					tableColumn.setText(attribute.getName());
				}
				tableColumn.setImage(ImageProvider.getInstance().getAttributeIcon());
				break;
			case REFERENCE:
				EReference reference = columnDescription.getReference();
				String multiplicity = LinkItemProvider.getMultiplicity(reference);
				if (reference.isDerived()) {
					tableColumn.setText("/" + reference.getName() + multiplicity);
				} else {
					tableColumn.setText(reference.getName() + multiplicity);
				}

				tableColumn.setImage((Image) LinkItemProvider.getImageFor(reference));
				// tableColumn.setImage(ImageProvider.getInstance().getLinkIcon());
				break;
			}

			final int fColumnIndex = columnIndex;
			new TableSorter(this.tableViewer, column) {
				@Override
				protected int doCompare(Viewer viewer, Object e1, Object e2) {
					// reuse the label provider to get the name to compare
					String label1 = TableEditor.this.tableEditorLabelProvider.getColumnText(e1,
							fColumnIndex);
					String label2 = TableEditor.this.tableEditorLabelProvider.getColumnText(e2,
							fColumnIndex);
					return label1.compareToIgnoreCase(label2);
				}
			};

			columnIndex++;
		}
	}

	@Override
	public void setFocus() {
		if (this.tableViewer != null) {
			this.tableViewer.getTable().setFocus();
		}
	}

	/**
	 * Whether the table editor can be opened on the given selection. <code>true</code> if the
	 * selection is a link or a collection of model objects.
	 */
	public static boolean canBeOpenedOnSelection(ISelection selection) {
		if (selection instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection) selection;

			if (structuredSelection.size() == 1
					&& structuredSelection.getFirstElement() instanceof LinkItemProvider) {
				LinkItemProvider linkItemProvider = (LinkItemProvider) structuredSelection
						.getFirstElement();
				return linkItemProvider.getChildren(null).size() > 0;
			} else {
				boolean empty = true;
				Iterator<?> iterator = structuredSelection.iterator();
				while (iterator.hasNext()) {
					Object selectedElement = iterator.next();
					if (selectedElement instanceof EObject) {
						empty = false;
					} else {
						return false;
					}
				}
				return !empty;
			}
		}
		return false;
	}

	/**
	 * Open a new table view editor on the given list of elements.
	 * 
	 * @param editorConfiguration
	 *            the main editor configuration
	 * @param elements
	 *            the elements to show as rows in the table
	 * @param description
	 *            a descriptive String of what will be displayed in the table view editor
	 */
	public static void openOn(EditorConfiguration editorConfiguration, List<EObject> elements,
			String description) {
		TableEditorInput input = new TableEditorInput(editorConfiguration, elements, description);

		IWorkbenchPage activePage = PlatformUI.getWorkbench().getActiveWorkbenchWindow()
				.getActivePage();
		try {
			IDE.openEditor(activePage, input, TableEditor.EDITOR_ID, true);
		} catch (PartInitException e) {
			MoDiscoEditorPlugin.INSTANCE.log(e);
		}
	}

	private void openSelectionInMainEditor(int column) {
		EObject element = findElementToOpenInMainEditor(column);
		if (element != null) {
			openElementInMainEditor(element);
		}
	}

	/** @return the first selected element or <code>null</code> if none */
	private EObject getSelectedElement() {
		ISelection selection = this.tableViewer.getSelection();
		if (selection instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			Object firstElement = structuredSelection.getFirstElement();
			if (firstElement instanceof EObject) {
				return (EObject) firstElement;
			}
		}
		return null;
	}

	/** Find and return the element that should be opened when the given column is clicked */
	private EObject findElementToOpenInMainEditor(int column) {
		EObject mainElement = getSelectedElement();
		if (mainElement == null)
			return null;

		EObject targetObject = null;
		if (column > 0) {

			ColumnDescription columnDescription = this.columnsConfiguration.get(column);
			if (columnDescription.getType() == Type.ATTRIBUTE) {
				Object value = mainElement.eGet(columnDescription.getAttribute());
				if (value instanceof EObject) {
					targetObject = (EObject) value;
				}
			} else if (columnDescription.getType() == Type.REFERENCE) {
				Object value = mainElement.eGet(columnDescription.getReference());
				if (columnDescription.getReference().isMany()) {
					@SuppressWarnings("unchecked")
					EList<EObject> list = (EList<EObject>) value;
					if (list.size() == 1) {
						targetObject = list.get(0);
					}
				} else {
					targetObject = (EObject) value;
				}
			} else if (columnDescription.getType() == Type.ECONTAINER) {
				Object value = mainElement.eContainer();
				if (value instanceof EObject) {
					targetObject = (EObject) value;
				}
			}
		}

		if (targetObject != null)
			return targetObject;
		else
			return mainElement;
	}

	private void openElementInMainEditor(EObject element) {
		EcoreEditor editor = TableEditor.this.tableEditorInput.getEditorConfiguration().getEditor();
		if (editor != null) {
			editor.getSite().getPage().activate(editor);
			editor.browseTo(element);
		}
	}

	@Override
	public boolean isDirty() {
		return false;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	@Override
	public void doSave(IProgressMonitor monitor) {
	}

	@Override
	public void doSaveAs() {
	}

	/** Hide columns which don't contain any elements */
	public void setHideEmptyColumns(boolean hideEmptyColumns) {
		try {
			this.tableViewer.getControl().setRedraw(false);
			for (ColumnDescription columnDescription : this.columnsConfiguration) {
				boolean bEmpty = false;
				if (columnDescription.getType() == Type.ATTRIBUTE) {
					bEmpty = isEmpty(columnDescription.getAttribute());
				} else if (columnDescription.getType() == Type.REFERENCE) {
					bEmpty = isEmpty(columnDescription.getReference());
				}

				TableColumn column = findColumn(columnDescription);
				if (hideEmptyColumns && bEmpty) {
					if (column != null) {
						// indicates that the column is hidden because it is empty
						column.setData(KEY_EMPTY_HIDDEN, Boolean.TRUE);
						hideColumn(column);
					}
				} else if (!hideEmptyColumns && Boolean.TRUE == column.getData(KEY_EMPTY_HIDDEN)) {
					column.setData(KEY_EMPTY_HIDDEN, Boolean.FALSE);
					showColumn(column);
				}
			}
		} finally {
			this.tableViewer.getControl().setRedraw(true);
		}
	}

	/**
	 * Whether to show columns which are specific to a few elements (false), or only columns which
	 * are common to all the elements (true)
	 */
	public void setOnlyShowCommonColumns(boolean onlyShowCommonColumns) {
		try {
			this.tableViewer.getControl().setRedraw(false);
			EClass[] metaclasses = this.tableEditorInput.getMetaclasses();

			for (ColumnDescription columnDescription : this.columnsConfiguration) {
				boolean common = true;
				if (columnDescription.getType() == Type.ATTRIBUTE) {
					common = isCommon(columnDescription.getAttribute(), metaclasses);
				} else if (columnDescription.getType() == Type.REFERENCE) {
					common = isCommon(columnDescription.getReference(), metaclasses);
				}

				TableColumn column = findColumn(columnDescription);
				if (onlyShowCommonColumns && !common) {
					if (column != null) {
						// indicates that the column is hidden because its feature
						// is not common to all the elements
						column.setData(KEY_NOT_COMMON_HIDDEN, Boolean.TRUE);
						hideColumn(column);
					}
				} else if (!onlyShowCommonColumns
						&& Boolean.TRUE == column.getData(KEY_NOT_COMMON_HIDDEN)) {
					column.setData(KEY_NOT_COMMON_HIDDEN, Boolean.FALSE);
					showColumn(column);
				}
			}
		} finally {
			this.tableViewer.getControl().setRedraw(true);
		}
	}

	/** Hide the given table column */
	private void hideColumn(TableColumn column) {
		column.setWidth(0);
		column.setResizable(false);
	}

	/** Show the given column after checking that it shouldn't be hidden anymore */
	private void showColumn(TableColumn column) {
		if (Boolean.TRUE != column.getData(KEY_EMPTY_HIDDEN)
				&& Boolean.TRUE != column.getData(KEY_NOT_COMMON_HIDDEN)) {
			column.pack();
			column.setResizable(true);
		}
	}

	/** @return whether the given feature is common to all the given metaclasses */
	private boolean isCommon(EStructuralFeature feature, EClass[] metaclasses) {
		for (EClass metaclass : metaclasses) {
			if (!metaclass.getEAllStructuralFeatures().contains(feature))
				return false;
		}
		return true;
	}

	/**
	 * Find the column with the given {@link ColumnDescription}
	 * 
	 * @return
	 */
	private TableColumn findColumn(ColumnDescription columnDescription) {
		TableColumn[] columns = this.tableViewer.getTable().getColumns();
		for (TableColumn tableColumn : columns) {
			if (tableColumn.getData() == columnDescription)
				return tableColumn;
		}
		return null;
	}

	/**
	 * Get a descriptive label that will be shown when a table editor is opened on the given
	 * reference
	 */
	public static String getEditorDescriptionFor(EObject parent, EReference reference,
			EditorConfiguration editorConfiguration) {
		AdapterFactory adapterFactory = editorConfiguration.getAdapterFactory();

		IItemLabelProvider labelProvider = (IItemLabelProvider) adapterFactory.adapt(parent,
				IItemLabelProvider.class);
		String parentLabel = labelProvider.getText(parent);

		return "\"" + parentLabel + "\" \u2014> " + reference.getName() + " contents";
	}
}
