/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.extensions.icons;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.gmt.modisco.common.editor.MoDiscoEditorPlugin;
import org.eclipse.gmt.modisco.common.editor.extensions.AbstractRegistry;
import org.eclipse.gmt.modisco.common.editor.util.EMFUtil;
import org.eclipse.swt.graphics.Image;

/**
 * The singleton registry of name providers, which initializes the registry by reading extensions
 * when first accessed. It can provide names for model elements, using {@link IconProvider}s
 * provided through the naming extension point.
 */
public class IconProvidersRegistry extends AbstractRegistry {

	private static final String METACLASS_ELEMENT = "metaclass";
	private static final String EXTENSION_POINT_NAMESPACE = "org.eclipse.gmt.modisco.common.editor";
	private static final String EXTENSION_POINT_NAME = "icons";
	private static final String METAMODEL_ID_ATTRIBUTE = "id";
	private static final String METAMODEL_ELEMENT = "metamodel";
	private static final String ICON_PROVIDER_ELEMENT = "iconProvider";
	private static final String ICON_PROVIDER_CLASS = "class";
	private static final String FILTER_ELEMENT = "filter";
	private static final String METACLASS_NAME_ATTRIBUTE = "name";

	private static IconProvidersRegistry instance = null;

	/**
	 * A map of metamodels to lists of FilteredIconProvider that provide icons for the metamodel's
	 * instances
	 */
	private Map<String, List<FilteredIconProvider>> iconProviders = new HashMap<String, List<FilteredIconProvider>>();

	public static IconProvidersRegistry getInstance() {
		if (instance == null) {
			instance = new IconProvidersRegistry();
		}
		return instance;
	}

	/**
	 * Query the icon providers registry for an icon for the given {@link EObject}.
	 * 
	 * @param eObject
	 *            the model instance whose icon is queried
	 * 
	 * @return the icon or <code>null</code> if no icon was provided
	 */
	public Image getIcon(EObject eObject) {
		EClass eClass = eObject.eClass();
		if (eClass == null)
			return null;

		String metaclassQualifiedName = EMFUtil.getMetaclassQualifiedName(eClass);

		// find the nsURI of the package containing the metamodel definition
		EPackage ePackage = eClass.getEPackage();
		if (ePackage == null)
			return null;
		String nsURI = ePackage.getNsURI();

		List<FilteredIconProvider> filteredIconProviders = this.iconProviders.get(nsURI);
		if (filteredIconProviders == null) {
			// try to retrieve a generic icon provider
			filteredIconProviders = this.iconProviders.get("*");
			if (filteredIconProviders == null) {
				return null;
			}
		}
		for (FilteredIconProvider filteredIconProvider : filteredIconProviders) {
			if (filteredIconProvider.filter(metaclassQualifiedName)) {
				Image icon = filteredIconProvider.getIcon(eObject);
				if (icon != null)
					return icon;
			}
		}
		return null;
	}

	/** Initialize the registry by reading the extension point to discover extensions. */
	public IconProvidersRegistry() {

		IExtensionRegistry extensionRegistry = Platform.getExtensionRegistry();
		IExtensionPoint extensionPoint = extensionRegistry.getExtensionPoint(
				EXTENSION_POINT_NAMESPACE, EXTENSION_POINT_NAME);

		IExtension[] extensions = extensionPoint.getExtensions();
		for (IExtension extension : extensions) {
			IConfigurationElement[] configurationElements = extension.getConfigurationElements();
			for (IConfigurationElement configurationElement : configurationElements) {
				String name = configurationElement.getName();
				if (name.equalsIgnoreCase(METAMODEL_ELEMENT)) {
					readMetamodelElement(configurationElement);
				} else {
					logUnknownElement(configurationElement);
				}
			}
		}
	}

	/** Read a 'metamodel' element */
	private void readMetamodelElement(IConfigurationElement configurationElement) {
		String metamodelId = configurationElement.getAttribute(METAMODEL_ID_ATTRIBUTE);
		if (metamodelId == null) {
			logMissingAttribute(configurationElement, METAMODEL_ID_ATTRIBUTE);
			return;
		}

		IConfigurationElement[] children = configurationElement.getChildren();
		for (IConfigurationElement child : children) {
			String name = child.getName();
			if (name.equalsIgnoreCase(ICON_PROVIDER_ELEMENT)) {
				readIconProviderElement(child, metamodelId);
			} else {
				logUnknownElement(configurationElement);
			}
		}

	}

	/**
	 * Read an 'iconProvider' element, and adds the icon providers to the registry.
	 * 
	 * @param configurationElement
	 *            the 'iconProvider' element
	 * @param metamodelId
	 *            the nsURI of the package containing the metamodel's metaclasses
	 */
	private void readIconProviderElement(IConfigurationElement configurationElement,
			String metamodelId) {

		// if not null, filters on the metaclass name
		HashSet<String> filteredMetaclasses = null;

		IConfigurationElement[] filterElements = configurationElement.getChildren(FILTER_ELEMENT);
		if (filterElements.length > 0) {
			// optional element : 0 or 1
			IConfigurationElement[] metaclassElements = filterElements[0]
					.getChildren(METACLASS_ELEMENT);
			for (IConfigurationElement metaclassElement : metaclassElements) {
				String metaclassName = metaclassElement.getAttribute(METACLASS_NAME_ATTRIBUTE);
				if (metaclassName != null) {
					if (filteredMetaclasses == null) {
						filteredMetaclasses = new HashSet<String>();
					}
					filteredMetaclasses.add(metaclassName);
				}
			}
		}

		final HashSet<String> fFilteredMetaclasses = filteredMetaclasses;

		Object iconProviderObject;
		try {
			iconProviderObject = configurationElement
					.createExecutableExtension(ICON_PROVIDER_CLASS);
		} catch (CoreException e) {
			MoDiscoEditorPlugin.INSTANCE.log(e);
			return;
		}
		if (iconProviderObject == null) {
			logMissingAttribute(configurationElement, ICON_PROVIDER_CLASS);
			return;
		}

		List<FilteredIconProvider> filteredIconProviders = this.iconProviders.get(metamodelId);
		if (filteredIconProviders == null) {
			filteredIconProviders = new ArrayList<FilteredIconProvider>();
		}

		/*
		 * The user can either provide a FilteredIconProvider or an IconProvider. In either case, if
		 * a filter is set in the extension definition, then it will get applied before the user
		 * code is called.
		 */
		if (iconProviderObject instanceof FilteredIconProvider) {
			final FilteredIconProvider filteredIconProvider = (FilteredIconProvider) iconProviderObject;
			filteredIconProviders.add(new FilteredIconProvider() {

				public Image getIcon(EObject object) {
					return filteredIconProvider.getIcon(object);
				}

				public boolean filter(String metaclass) {
					if (fFilteredMetaclasses != null && !fFilteredMetaclasses.contains(metaclass))
						return false;
					return filteredIconProvider.filter(metaclass);
				}
			});
		} else if (iconProviderObject instanceof IconProvider) {
			final IconProvider iconProvider = (IconProvider) iconProviderObject;

			FilteredIconProvider filteredNameProvider = new FilteredIconProvider() {

				public Image getIcon(EObject object) {
					return iconProvider.getIcon(object);
				}

				public boolean filter(String metaclass) {
					if (fFilteredMetaclasses == null)
						return true;
					return fFilteredMetaclasses.contains(metaclass);
				}
			};
			filteredIconProviders.add(filteredNameProvider);
		} else {
			logError(configurationElement, "Given class is not an IconProvider");
		}
		this.iconProviders.put(metamodelId, filteredIconProviders);
	}
}
