/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.extensions.naming;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.gmt.modisco.common.editor.MoDiscoEditorPlugin;
import org.eclipse.gmt.modisco.common.editor.extensions.AbstractRegistry;
import org.eclipse.gmt.modisco.common.editor.util.EMFUtil;

/**
 * The singleton registry of name providers, which initializes the registry by reading extensions
 * when first accessed. It can provide names for model elements, using {@link NameProvider}s
 * provided through the naming extension point.
 */
public class NameProvidersRegistry extends AbstractRegistry {

	private static final String METACLASS_ELEMENT = "metaclass";
	private static final String EXTENSION_POINT_NAMESPACE = "org.eclipse.gmt.modisco.common.editor";
	private static final String EXTENSION_POINT_NAME = "naming";
	private static final String METAMODEL_ID_ATTRIBUTE = "id";
	private static final String METAMODEL_ELEMENT = "metamodel";
	private static final String NAME_PROVIDER_ELEMENT = "nameProvider";
	private static final String NAME_PROVIDER_CLASS = "class";
	private static final String FILTER_ELEMENT = "filter";
	private static final String METACLASS_NAME_ATTRIBUTE = "name";

	private static NameProvidersRegistry instance = null;

	/**
	 * A map of metamodels to lists of FilteredNameProvider that provide names for the metamodel's
	 * instances
	 */
	private Map<String, List<FilteredNameProvider>> nameProviders = new HashMap<String, List<FilteredNameProvider>>();

	public static NameProvidersRegistry getInstance() {
		if (instance == null) {
			instance = new NameProvidersRegistry();
		}
		return instance;
	}

	/**
	 * Query the name providers registry for a name for the given {@link EObject}.
	 * 
	 * @param eObject
	 *            the model instance whose name is queried
	 * 
	 * @return the name or <code>null</code> if no name was provided
	 */
	public String getName(EObject eObject) {
		EClass eClass = eObject.eClass();
		if (eClass == null)
			return null;

		String metaclassQualifiedName = EMFUtil.getMetaclassQualifiedName(eClass);

		// the nsURI of the package containing the metamodel definition
		EPackage ePackage = eClass.getEPackage();
		if (ePackage == null)
			return null;

		String nsURI = ePackage.getNsURI();

		List<FilteredNameProvider> filteredNameProviders = this.nameProviders.get(nsURI);
		if (filteredNameProviders == null) {
			// try to retrieve a generic name provider
			filteredNameProviders = this.nameProviders.get("*");
			if (filteredNameProviders == null) {
				return null;
			}
		}

		for (FilteredNameProvider filteredNameProvider : filteredNameProviders) {
			if (filteredNameProvider.filter(metaclassQualifiedName)) {
				String name = filteredNameProvider.getName(eObject);
				if (name != null)
					return name;
			}
		}
		return null;
	}

	/** Initialize the registry by reading the extension point to discover extensions. */
	public NameProvidersRegistry() {

		IExtensionRegistry extensionRegistry = Platform.getExtensionRegistry();
		IExtensionPoint extensionPoint = extensionRegistry.getExtensionPoint(
				EXTENSION_POINT_NAMESPACE, EXTENSION_POINT_NAME);

		IExtension[] extensions = extensionPoint.getExtensions();
		for (IExtension extension : extensions) {
			IConfigurationElement[] configurationElements = extension.getConfigurationElements();
			for (IConfigurationElement configurationElement : configurationElements) {
				String name = configurationElement.getName();
				if (name.equalsIgnoreCase(METAMODEL_ELEMENT)) {
					readMetamodelElement(configurationElement);
				} else {
					logUnknownElement(configurationElement);
				}
			}
		}
	}

	/** Read a 'metamodel' element */
	private void readMetamodelElement(IConfigurationElement configurationElement) {
		String metamodelId = configurationElement.getAttribute(METAMODEL_ID_ATTRIBUTE);
		if (metamodelId == null) {
			logMissingAttribute(configurationElement, METAMODEL_ID_ATTRIBUTE);
			return;
		}

		IConfigurationElement[] children = configurationElement.getChildren();
		for (IConfigurationElement child : children) {
			String name = child.getName();
			if (name.equalsIgnoreCase(NAME_PROVIDER_ELEMENT)) {
				readNameProviderElement(child, metamodelId);
			} else {
				logUnknownElement(configurationElement);
			}
		}

	}

	/**
	 * Read a 'nameProvider' element, and adds the name providers to the registry.
	 * 
	 * @param configurationElement
	 *            the 'nameProvider' element
	 * @param metamodelId
	 *            the nsURI of the package containing the metamodel's metaclasses
	 */
	private void readNameProviderElement(IConfigurationElement configurationElement,
			String metamodelId) {

		// if not null, filters on the metaclass name
		HashSet<String> filteredMetaclasses = null;

		IConfigurationElement[] filterElements = configurationElement.getChildren(FILTER_ELEMENT);
		if (filterElements.length > 0) {
			// optional element : 0 or 1
			IConfigurationElement[] metaclassElements = filterElements[0]
					.getChildren(METACLASS_ELEMENT);
			for (IConfigurationElement metaclassElement : metaclassElements) {
				String metaclassName = metaclassElement.getAttribute(METACLASS_NAME_ATTRIBUTE);
				if (metaclassName != null) {
					if (filteredMetaclasses == null) {
						filteredMetaclasses = new HashSet<String>();
					}
					filteredMetaclasses.add(metaclassName);
				}
			}
		}

		final HashSet<String> fFilteredMetaclasses = filteredMetaclasses;

		Object nameProviderObject;
		try {
			nameProviderObject = configurationElement
					.createExecutableExtension(NAME_PROVIDER_CLASS);
		} catch (CoreException e) {
			MoDiscoEditorPlugin.INSTANCE.log(e);
			return;
		}
		if (nameProviderObject == null) {
			logMissingAttribute(configurationElement, NAME_PROVIDER_CLASS);
			return;
		}

		List<FilteredNameProvider> filteredNameProviders = this.nameProviders.get(metamodelId);
		if (filteredNameProviders == null) {
			filteredNameProviders = new ArrayList<FilteredNameProvider>();
		}

		/*
		 * The user can either provide a FilteredNameProvider or a NameProvider. In either case, if
		 * a filter is set in the extension definition, then it will get applied before the user
		 * code is called.
		 */
		if (nameProviderObject instanceof FilteredNameProvider) {
			final FilteredNameProvider filteredNameProvider = (FilteredNameProvider) nameProviderObject;
			filteredNameProviders.add(new FilteredNameProvider() {

				public String getName(EObject eObject) {
					return filteredNameProvider.getName(eObject);
				}

				public boolean filter(String metaclass) {
					if (fFilteredMetaclasses != null && !fFilteredMetaclasses.contains(metaclass))
						return false;
					return filteredNameProvider.filter(metaclass);
				}
			});
		} else if (nameProviderObject instanceof NameProvider) {
			final NameProvider nameProvider = (NameProvider) nameProviderObject;

			FilteredNameProvider filteredNameProvider = new FilteredNameProvider() {

				public String getName(EObject eObject) {
					return nameProvider.getName(eObject);
				}

				public boolean filter(String metaclass) {
					if (fFilteredMetaclasses == null)
						return true;
					return fFilteredMetaclasses.contains(metaclass);
				}
			};
			filteredNameProviders.add(filteredNameProvider);
		} else {
			logError(configurationElement, "Given class is not a NameProvider");
		}
		this.nameProviders.put(metamodelId, filteredNameProviders);
	}
}
