/*******************************************************************************
 * Copyright (c) 2008, 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.core.utils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FilenameFilter;
import java.io.IOException;

/**
 * @author Gabriel Barbier
 *
 */
public class FolderUtilsImpl 
    implements FolderUtils
{
    private static boolean debug = false;

    private FolderUtilsImpl.ConfigurationManagementFilter configurationManagementFilter;

    private static class ConfigurationManagementFilter 
        implements FilenameFilter
    {
        private String filterName = "CVS";


        /**
          * Tests if a specified file should be included in a file list.
          * 
          * 
          * @param dir the directory in which the file was found.
          * @param name the name of the file.
          * 
          * @return  true if and only if the name should be included in the file list; false otherwise.
          */
        public boolean accept(
            final File dir,
            final String name)
        {
			boolean result = false;
			if (name.equals(this.filterName) == false) {
				result = true;
			}
			return result;
        }
    }


    FolderUtilsImpl()
    {
    	this.configurationManagementFilter = new ConfigurationManagementFilter();
    }

    public final void clearFolder(
        final File dirtyFolder)
    {
        assert dirtyFolder != null;
		assert dirtyFolder.exists();
		assert dirtyFolder.isDirectory();
		/*
		 * pour supprimer le contenu du rpertoire, il faut faire une rcursion
		 * sur tous les objets (File) de ce rpertoire.
		 * Dans le cas d'un rpertoire, il faut d'abord supprimer son propre contenu
		 * avant de pouvoir le supprimer  son tour.
		 * Dans le cas d'un fichier simple, il suffit de le supprimer.
		 */
		File[] files = dirtyFolder.listFiles();
		for (File file : files) {
			if (file.isDirectory()) {
				// c'est un rpertoire
				this.clearFolder(file);	
			}
			file.delete();
		}
    }

    /**
      * this method compare two folders, in fact content of these two folders.
      * It is recursive into each folder
      */
    public final boolean compareFolders(
        final File folderSource,
        final File folderTarget)
    {
    	return compareFolders(folderSource, folderTarget, this.configurationManagementFilter);
    }
 
    public final boolean compareFolders(
        	final File folderSource,
        	final File folderTarget,
        	final FilenameFilter filter) {
        	assert ((folderSource != null) && (folderTarget != null));
     		assert ((folderSource.isDirectory()) && (folderTarget.isDirectory()));
     		boolean result = false;
     		if (folderSource.equals(folderTarget)) {
     			result = true;
     		} else {
     			result = this.recursiveCompareFolders(folderSource, folderTarget, filter);
     			if ((FolderUtilsImpl.debug == true) && (result == false)) {
     				System.err.println("folders " + folderSource.getName() + " and " + folderTarget.getName() + " are not equals.");
     			}
     		}
            return result;
    }

    public final boolean compareFiles(
        final File source,
        final File target)
    {
        assert ((source.isFile()) && (target.isFile()));
		boolean result = true;
		if (source.getName().equals(target.getName()) == false) {
			result = false;
		} else {
			/*
			 * Comparaison du contenu des deux fichiers
			 * Chaque ligne d'un fichier va tre compare
			 *  la ligne correspondante de l'autre fichier
			 */
			try {
				BufferedReader sourceReader = new BufferedReader(new FileReader(source));
				BufferedReader targetReader = new BufferedReader(new FileReader(target));
				String sourceLine = sourceReader.readLine();
				String targetLine = targetReader.readLine();
				while ((sourceLine != null) && (targetLine != null)) {
					result = result && sourceLine.equals(targetLine);
					sourceLine = sourceReader.readLine();
					targetLine = targetReader.readLine();
				}
				if ((sourceLine != null) || (targetLine != null)) {
					result = false;
					System.err.println("These files have not the same number of lines.");
				}
			} catch (FileNotFoundException e) {
				result = false;
			} catch (IOException e) {
				result = false;
			}
		}
		if ((FolderUtilsImpl.debug == true) && (result == true)) {
			System.out.println("files " + source.getName() + " and " + target.getName() + " are equals.");
		}
		return result;
    }

    public final String getFileContent(
        final File source)
    {
		StringBuilder result = new StringBuilder();
		try {
			BufferedReader sourceReader = new BufferedReader(new FileReader(source));
			String sourceLine = sourceReader.readLine();
			while (sourceLine != null) {
				result.append(sourceLine);
				result.append("\n");
				sourceLine = sourceReader.readLine();
			}
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return result.toString();
    }

    /**
      * this method compare two folders, in fact content of these two folders.
      * It is recursive into each folder
      */
    private final boolean recursiveCompareFolders(
        final File folderSource,
        final File folderTarget,
        final FilenameFilter filter)
    {
		if (FolderUtilsImpl.debug == true) {
			System.err.println("comparison of " + folderSource.getName() + " and " + folderTarget.getName());
		}
		boolean result = true;

		File[] sourceContents = folderSource.listFiles(filter);
		File[] targetContents = folderTarget.listFiles(filter);
		if (sourceContents.length != targetContents.length) {
			result = false;
			if (FolderUtilsImpl.debug == true) {
				System.err.println("folders " + folderSource.getName() + " and " + folderTarget.getName() + " have not the same number of childrens (" + sourceContents.length + ", " + targetContents.length +")");
			}
		} else {
			for (File sourceContent : sourceContents) {
				/*
				 * pour chaque objet File du rpertoire courant,
				 * il faut tester si il existe son quivalent dans 
				 * le contenu du rpertoire cible.
				 * La recherche est base sur le nom (path name),
				 * il peut donc y avoir des problmes si 
				 * un rpertoire et un fichier porte le mme nom.
				 */
				File targetContent = this.getCorrespondingTargetContent(sourceContent, targetContents);
				if (targetContent == null) {
					result = false;
					if (FolderUtilsImpl.debug == true) {
						System.err.println("There is no corresponding element in target folder for " + sourceContent.getName());
					}
				} else {
					/*
					 * il y a maintenant deux cas  distinguer :
					 * le cas d'un fichier simple et le cas d'un rpertoire.
					 * Dans le cas d'un rpertoire, il suffit de faire la rcursion.
					 * Dans le cas d'un fichier, il faut comparer le contenu des fichiers.
					 */
					if (sourceContent.isDirectory()) {
						result = result && this.recursiveCompareFolders(sourceContent, targetContent, filter);
						if ((FolderUtilsImpl.debug == true) && (result == false)) {
							System.err.println("folders " + sourceContent.getName() + " and " + targetContent.getName() + " are not equals.");
						}
					} else {
						result = result && this.compareFiles(sourceContent, targetContent);
						if ((FolderUtilsImpl.debug == true) && (result == false)) {
							System.err.println("files " + sourceContent.getName() + " and " + targetContent.getName() + " are not equals.");
						}
					}
				}
			}
		}
		return result;
    }

    private final File getCorrespondingTargetContent(
        final File sourceContent,
        final File[] targetContents)
    {
		File targetContent = null;
		for (File temp : targetContents) {
			if (sourceContent.getName().equals(temp.getName())) {
				targetContent = temp;
			}
		}
		return targetContent;
    }
    
	public final void copyDirectory(File srcDir, File destDir)
			throws IOException {

		if (destDir.exists() == false) {
			destDir.mkdirs();
		}
		File[] filesList = srcDir.listFiles();
		File dest;
		// Copies each file and directory, one by one
		for (File src : filesList) {
			dest = new File(destDir.getPath() + File.separator
					+ src.getName());
			if (src.isDirectory()) {
				this.copyDirectory(src, dest);
			} else {
				this.copyFile(src, dest);
			}
		}
		
	}
	
	public final boolean copyFile(File source, File destination) {
		boolean resultat = false;
		FileInputStream sourceFile = null;
		FileOutputStream destinationFile = null;
		try {
			// File creation
			destination.createNewFile();
			sourceFile = new FileInputStream(source);
			destinationFile = new FileOutputStream(destination);
			// 0.5 Mo buffer reading
			byte buffer[] = new byte[512 * 1024];
			int nbLecture;
			while ((nbLecture = sourceFile.read(buffer)) != -1) {
				destinationFile.write(buffer, 0, nbLecture);
			}

			// Copy
			resultat = true;
		} catch (java.io.FileNotFoundException f) {
		} catch (java.io.IOException e) {
		} finally {
			try {
				if (sourceFile != null) {
					sourceFile.close();
				}
				if (destinationFile != null) {
					destinationFile.close();
				}
			} catch (Exception e) {
			}
		}
		return resultat;
	}
	
	public final void deleteDirectory(File directory) throws IOException {
		if (!directory.exists()) {
			return;
		}

		clearFolder(directory);
		if (directory.list().length == 0) {
			// delete directory 
			if (!directory.delete()) {
				String message = "Unable to delete directory " + directory + ".";
				throw new IOException(message);
			}
		}
	}

}
