/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Grgoire Dup (Mia-Software)
 *******************************************************************************/
package org.eclipse.gmt.modisco.common.core.utils;

import java.util.Iterator;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.Diagnostic;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.Diagnostician;
import org.eclipse.gmt.modisco.common.core.CommonModiscoActivator;

/**
 * @author Grgoire Dup (Mia-Software)
 * 
 */
public final class ValidationUtils {

	public static final String EMF_PROBLEM_MARKER = "org.eclipse.emf.ecore.diagnostic";

	private ValidationUtils() {
		// This is an utility class
	}

	public static synchronized boolean validate(final Resource resource,
			final IResource file) throws CoreException {
		if (file != null) {
			file.deleteMarkers(null, true, IResource.DEPTH_INFINITE);
		}
		boolean result = true;
		Iterator<EObject> contents = resource.getContents().iterator();
		while (contents.hasNext()) {
			EObject eObject = contents.next();
			Diagnostic diagnostic = Diagnostician.INSTANCE.validate(eObject);
			if (diagnostic.getSeverity() != Diagnostic.OK) {
				result = (diagnostic.getSeverity() < Diagnostic.ERROR);
				if (file != null) {
					convertDiagnosticToMarker(file, diagnostic);
				} else {
					convertDiagnosticToLog(resource, diagnostic);
				}
			}
		}
		return result;

	}

	/**
	 * @param file
	 * @param diagnostic
	 */
	private static void convertDiagnosticToLog(final Resource resource,
			final Diagnostic diagnostic) {
		if (diagnostic.getChildren().size() > 0) {
			for (Diagnostic subDiagnostic : diagnostic.getChildren()) {
				convertDiagnosticToLog(resource, subDiagnostic);
			}
		} else {
			IStatus status = null;
			String pluginId = resource.getURI().segment(1);
			if (diagnostic.getSeverity() == Diagnostic.ERROR) {
				status = new Status(IStatus.ERROR, pluginId, diagnostic
						.getMessage());
			} else if (diagnostic.getSeverity() == Diagnostic.WARNING) {
				status = new Status(IStatus.WARNING, pluginId, diagnostic
						.getMessage());
			} else if (diagnostic.getSeverity() == Diagnostic.INFO) {
				status = new Status(IStatus.INFO, pluginId, diagnostic
						.getMessage());
			}
			CommonModiscoActivator.getDefault().getLog().log(status);
		}

	}

	/**
	 * @param file
	 * @param diagnostic
	 * @throws CoreException
	 */
	private static void convertDiagnosticToMarker(final IResource file,
			final Diagnostic diagnostic) throws CoreException {
		if (diagnostic.getChildren().size() > 0) {
			for (Diagnostic subDiagnostic : diagnostic.getChildren()) {
				convertDiagnosticToMarker(file, subDiagnostic);
			}
		} else {
			IMarker marker = file
					.createMarker(ValidationUtils.EMF_PROBLEM_MARKER);
			marker.setAttribute(IMarker.MESSAGE, diagnostic.getMessage());

			if (diagnostic.getSeverity() == Diagnostic.ERROR) {
				marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
			} else if (diagnostic.getSeverity() == Diagnostic.WARNING) {
				marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_WARNING);
			} else if (diagnostic.getSeverity() == Diagnostic.INFO) {
				marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_INFO);
			}
			marker.setAttribute(IMarker.LOCATION, diagnostic.getSource());
		}

	}
}
