/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    Grgoire Dup (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.infra.role.ui.views;

import java.net.URI;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.ui.URIEditorInput;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.gmt.modisco.common.core.builder.AbstractMoDiscoCatalog.ModiscoCatalogChangeListener;
import org.eclipse.gmt.modisco.common.core.logging.MoDiscoLogger;
import org.eclipse.gmt.modisco.common.core.utils.ImageUtils;
import org.eclipse.gmt.modisco.infra.role.Role;
import org.eclipse.gmt.modisco.infra.role.RolePackage;
import org.eclipse.gmt.modisco.infra.role.RoleSet;
import org.eclipse.gmt.modisco.infra.role.core.RoleSetCatalog;
import org.eclipse.gmt.modisco.infra.role.ui.Activator;
import org.eclipse.gmt.modisco.infra.role.ui.Messages;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.IOpenListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.OpenEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.ViewPart;

/**
 * An Eclipse view that displays a list of available MoDisco role sets. Clicking
 * on a role set opens it in the role set editor.
 */
public class RoleView extends ViewPart implements ModiscoCatalogChangeListener {

	private static final int JOB_SCHEDULE_DELAY = 500;
	private static final int COLUMN_WIDTH = 200;
	private TreeViewer roleSetViewer;
	private TreeViewerColumn nameColumn;
	private TreeViewerColumn locationColumn;
	private Job refreshJob = null;
	private TreeViewerColumn defaultColumn;
	private TreeViewerColumn extendsColumn;

	public TreeViewer getRoleSetViewer() {
		return this.roleSetViewer;
	}

	public TreeViewerColumn getNameColumn() {
		return this.nameColumn;
	}

	@Override
	public void createPartControl(final Composite parent) {

		parent.setLayout(new FillLayout());

		this.roleSetViewer = new TreeViewer(parent, SWT.H_SCROLL | SWT.V_SCROLL | SWT.SINGLE
				| SWT.FULL_SELECTION);
		this.roleSetViewer.getTree().setLinesVisible(true);
		this.roleSetViewer.getTree().setHeaderVisible(true);

		this.roleSetViewer.addOpenListener(new IOpenListener() {
			public void open(final OpenEvent event) {
				IStructuredSelection selection = (IStructuredSelection) RoleView.this
						.getRoleSetViewer().getSelection();

				EObject modelElement = (EObject) selection.getFirstElement();
				EPackage roleSet = null;
				if (modelElement instanceof RoleSet) {
					roleSet = (RoleSet) modelElement;
				} else if (modelElement instanceof Role) {
					Role role = (Role) modelElement;
					roleSet = role.getEPackage();
				}
				if (roleSet != null) {
					// open role set in editor
					URI uri = null;
					try {
						String uriString = RoleSetCatalog.getSingleton().getURI(roleSet.getName())
								.toString();
						uri = URI.create(uriString);
						if ("file".equals(uri.getScheme())) { //$NON-NLS-1$
							IDE.openEditor(PlatformUI.getWorkbench().getActiveWorkbenchWindow()
									.getActivePage(), new URI(roleSet.eResource().getURI()
									.toString()), Activator.ROLE_EDITOR_ID, true);
						} else {
							URIEditorInput uriEditorInput = new URIEditorInput(
									org.eclipse.emf.common.util.URI.createURI(uri.toString()));
							IDE.openEditor(PlatformUI.getWorkbench().getActiveWorkbenchWindow()
									.getActivePage(), uriEditorInput, Activator.ROLE_EDITOR_ID,
									true);
						}

					} catch (Exception e) {
						MoDiscoLogger.logError(e, "Failed to open: " + uri, Activator.getDefault()); //$NON-NLS-1$
					}
				}
			}
		});

		createNameColumn();
		createExtendsColumn();
		createLocationColumn();

		createDefaultColumn();

		this.roleSetViewer.setContentProvider(new RoleSetContentProvider(RolePackage.eINSTANCE
				.getRoleStructuralFeature()));

		getSite().setSelectionProvider(this.roleSetViewer);
		refresh(false);

		RoleSetCatalog.getSingleton().addChangeListener(this);
	}

	private void createLocationColumn() {
		this.locationColumn = new TreeViewerColumn(this.roleSetViewer, SWT.NONE);
		this.locationColumn.getColumn().setText(Messages.RoleView_location);
		this.locationColumn.getColumn().setWidth(RoleView.COLUMN_WIDTH * 2);
		this.locationColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				if (!(element instanceof RoleSet)) {
					return ""; //$NON-NLS-1$
				}
				RoleSet roleSet = (RoleSet) element;
				return RoleSetCatalog.getSingleton().getURI(roleSet.getName()).toString();
			}
		});
		addSorter(this.locationColumn, new ViewerSorter() {
			@Override
			public int compare(final Viewer viewer, final Object e1, final Object e2) {
				RoleSet roleSet1 = (RoleSet) e1;
				RoleSet roleSet2 = (RoleSet) e2;
				String uri1 = RoleSetCatalog.getSingleton().getURI(roleSet1.getName()).toString();
				String uri2 = RoleSetCatalog.getSingleton().getURI(roleSet2.getName()).toString();
				int compare = uri1.compareTo(uri2);
				if (viewer instanceof TreeViewer) {
					TreeViewer structuredViewer = (TreeViewer) viewer;
					if (structuredViewer.getTree().getSortDirection() == SWT.DOWN) {
						compare = compare * -1;
					}
				}
				return compare;
			}
		});
	}

	private void createNameColumn() {
		this.nameColumn = new TreeViewerColumn(this.roleSetViewer, SWT.NONE);
		this.nameColumn.getColumn().setText(Messages.RoleView_name);
		this.nameColumn.getColumn().setWidth(RoleView.COLUMN_WIDTH);
		this.nameColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof RoleSet) {
					RoleSet roleSet = (RoleSet) element;
					return roleSet.getName();
				} else if (element instanceof Role) {
					Role role = (Role) element;
					return role.getName();
				} else if (element instanceof EStructuralFeature) {
					EStructuralFeature feature = (EStructuralFeature) element;
					return feature.getName();
				} else {
					return ""; //$NON-NLS-1$
				}

			}

			@Override
			public Image getImage(final Object element) {
				return ImageUtils.getImage(element);
			}
		});

		addSorter(this.nameColumn, new ViewerSorter() {
			@Override
			public int compare(final Viewer viewer, final Object e1, final Object e2) {
				RoleSet roleSet1 = (RoleSet) e1;
				RoleSet roleSet2 = (RoleSet) e2;
				int compare = roleSet1.getName().compareTo(roleSet2.getName());
				if (viewer instanceof TreeViewer) {
					TreeViewer structuredViewer = (TreeViewer) viewer;
					if (structuredViewer.getTree().getSortDirection() == SWT.DOWN) {
						compare = compare * -1;
					}
				}
				return compare;
			}
		});
	}

	private void createDefaultColumn() {
		this.defaultColumn = new TreeViewerColumn(this.roleSetViewer, SWT.NONE);
		this.defaultColumn.getColumn().setText(Messages.RoleView_1);
		this.defaultColumn.getColumn().setWidth(RoleView.COLUMN_WIDTH);
		this.defaultColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof RoleSet) {
					RoleSet roleSet = (RoleSet) element;
					EAnnotation annotation = roleSet.getEAnnotation("default");//$NON-NLS-1$
					if (annotation != null) {
						if (annotation.getReferences().size() == 1
								&& annotation.getReferences().get(0) instanceof EPackage) {
							EPackage ePackage = (EPackage) annotation.getReferences().get(0);
							return ePackage.getName() + " " + ePackage.getNsURI(); //$NON-NLS-1$
						}

					}
				}
				return ""; //$NON-NLS-1$
			}

		});

		addSorter(this.defaultColumn, new ViewerSorter() {
			@Override
			public int compare(final Viewer viewer, final Object e1, final Object e2) {
				RoleSet roleSet1 = (RoleSet) e1;
				RoleSet roleSet2 = (RoleSet) e2;
				int compare = roleSet1.getName().compareTo(roleSet2.getName());
				if (viewer instanceof TreeViewer) {
					TreeViewer structuredViewer = (TreeViewer) viewer;
					if (structuredViewer.getTree().getSortDirection() == SWT.DOWN) {
						compare = compare * -1;
					}
				}
				return compare;
			}
		});
	}

	private void createExtendsColumn() {
		this.extendsColumn = new TreeViewerColumn(this.roleSetViewer, SWT.NONE);
		this.extendsColumn.getColumn().setText("Extends");
		this.extendsColumn.getColumn().setWidth(RoleView.COLUMN_WIDTH);
		this.extendsColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof RoleSet) {
					RoleSet roleSet = (RoleSet) element;
					return roleSet.getExtendedPackage().getName() + " "
							+ roleSet.getExtendedPackage().getNsURI();
				}
				return ""; //$NON-NLS-1$
			}

		});

		addSorter(this.extendsColumn, new ViewerSorter() {
			@Override
			public int compare(final Viewer viewer, final Object e1, final Object e2) {
				RoleSet roleSet1 = (RoleSet) e1;
				RoleSet roleSet2 = (RoleSet) e2;
				int compare = roleSet1.getName().compareTo(roleSet2.getName());
				if (viewer instanceof TreeViewer) {
					TreeViewer structuredViewer = (TreeViewer) viewer;
					if (structuredViewer.getTree().getSortDirection() == SWT.DOWN) {
						compare = compare * -1;
					}
				}
				return compare;
			}
		});
	}

	private void addSorter(final TreeViewerColumn column, final ViewerSorter viewerSorter) {
		column.getColumn().addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				TreeViewer treeView = RoleView.this.getRoleSetViewer();
				treeView.setSorter(viewerSorter);
				treeView.getTree().setSortColumn(column.getColumn());
				if (treeView.getTree().getSortDirection() == SWT.UP) {
					treeView.getTree().setSortDirection(SWT.DOWN);
				} else if (treeView.getTree().getSortDirection() == SWT.DOWN) {
					treeView.getTree().setSortDirection(SWT.UP);
				} else {
					treeView.getTree().setSortDirection(SWT.UP);
				}
				treeView.refresh();
			}
		});

	}

	@Override
	public void setFocus() {
		this.roleSetViewer.getControl().setFocus();
	}

	@Override
	public void dispose() {
		super.dispose();
		RoleSetCatalog.getSingleton().removeChangeListener(this);
	}

	/** Called when a role set changes */
	public void changed(final EObject changedRoleSet, final IFile file) {
		refresh(true);
	}

	/** Called when a role set is added */
	public void added(final EObject addedRoleSet, final IFile file) {
		refresh(true);
	}

	/** Called when a role set is removed */
	public void removed(final IFile roleSetFile) {
		refresh(true);
	}

	/** Optionally delayed refresh */
	private void refresh(final boolean delayed) {
		if (this.refreshJob == null) {
			this.refreshJob = new Job(Messages.RoleView_JobName_refreshingRoleSetsView) {
				@Override
				protected IStatus run(final IProgressMonitor monitor) {
					Display.getDefault().syncExec(new Runnable() {
						public void run() {
							RoleView.this.getRoleSetViewer().setInput(
									RoleSetCatalog.getSingleton().getAllRoleSets().toArray());
							RoleView.this.getNameColumn().getColumn().pack();
							RoleView.this.getRoleSetViewer().refresh();
						}
					});
					return Status.OK_STATUS;
				}
			};
		}
		if (delayed) {
			// delayed until it stops changing
			this.refreshJob.cancel();
			this.refreshJob.setPriority(Job.DECORATE);
			this.refreshJob.schedule(RoleView.JOB_SCHEDULE_DELAY);
		} else {
			// force synchronous execution
			this.refreshJob.schedule();
			try {
				this.refreshJob.join();
			} catch (InterruptedException e) {
				MoDiscoLogger.logError(e, Activator.getDefault());
			}
		}
	}

}
