/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Nicolas Bros (Mia-Software)
 *******************************************************************************/
package org.eclipse.gmt.modisco.infra.query.ui.dialogs;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.gmt.modisco.common.core.utils.ImageUtils;
import org.eclipse.gmt.modisco.common.ui.util.ImageProvider;
import org.eclipse.gmt.modisco.infra.query.ModelQuery;
import org.eclipse.gmt.modisco.infra.query.ModelQuerySet;
import org.eclipse.gmt.modisco.infra.query.core.ModelQuerySetCatalog;
import org.eclipse.gmt.modisco.infra.query.ui.Activator;
import org.eclipse.gmt.modisco.infra.query.ui.Messages;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.IOpenListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.OpenEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.dialogs.FilteredTree;
import org.eclipse.ui.dialogs.PatternFilter;

/** A dialog to select a {@link ModelQuery} */
public class QuerySelectionDialog extends Dialog {

	private final Collection<ModelQuerySet> fQuerySets;
	private final ViewerFilter viewerFilter;
	private FilteredTree filteredTree;
	private final boolean multiSelection;
	private List<ModelQuery> selectedQueries;

	/**
	 * @param parentShell
	 *            the parent shell
	 * @param querySets
	 *            the query sets from which queries will be selectable. If
	 *            <code>null</code>, all query sets will be displayed.
	 * @param viewerFilter
	 *            an optional filter for selectable queries (can be
	 *            <code>null</code> if no filtering is required)
	 */
	public QuerySelectionDialog(final Shell parentShell, final boolean multiSelection,
			final Collection<ModelQuerySet> querySets, final ViewerFilter viewerFilter) {
		super(parentShell);
		this.multiSelection = multiSelection;
		if (querySets == null) {
			this.fQuerySets = ModelQuerySetCatalog.getSingleton().getAllModelQuerySets();
		} else {
			this.fQuerySets = querySets;
		}
		this.viewerFilter = viewerFilter;
	}

	@Override
	protected Control createDialogArea(final Composite parent) {
		Composite composite = (Composite) super.createDialogArea(parent);
		// composite.setLayout(new GridLayout());
		this.filteredTree = createFilteredTree(composite, this.fQuerySets, this.viewerFilter,
				this.multiSelection);
		// double-click or enter to choose the selected query
		this.filteredTree.getViewer().addOpenListener(new IOpenListener() {
			public void open(final OpenEvent event) {
				QuerySelectionDialog.this.okPressed();
			}
		});
		return composite;
	}

	@Override
	protected void configureShell(final Shell shell) {
		super.configureShell(shell);
		shell.setText(Messages.QuerySelectionDialog_selectQuery);
		shell.setImage(ImageProvider.getInstance().getModiscoLogo());
	}

	@Override
	protected boolean isResizable() {
		return true;
	}

	@Override
	protected IDialogSettings getDialogBoundsSettings() {
		final String sectionName = this.getClass().getName() + ".settings"; //$NON-NLS-1$
		IDialogSettings settings = Activator.getDefault().getDialogSettings();
		IDialogSettings section = settings.getSection(sectionName);
		if (section == null) {
			section = settings.addNewSection(sectionName);
		}
		return section;
	}

	@Override
	protected void okPressed() {
		this.selectedQueries = new ArrayList<ModelQuery>();
		ISelection selection = this.filteredTree.getViewer().getSelection();
		if (selection instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			Iterator<?> iterator = structuredSelection.iterator();
			while (iterator.hasNext()) {
				Object selectedElement = iterator.next();
				if (selectedElement instanceof ModelQuery) {
					this.selectedQueries.add((ModelQuery) selectedElement);
				}
			}
		}
		super.okPressed();
	}

	public List<ModelQuery> getSelectedQueries() {
		return this.selectedQueries;
	}

	/**
	 * Create a tree with a filter text field to select queries.
	 * 
	 * @param parent
	 *            the parent composite
	 * @param availableQuerySets
	 *            the query sets from which queries are selected.
	 *            <code>null</code> means all.
	 * @param viewerFilter
	 *            a filter on the displayed queries. Can be <code>null</code>.
	 * @return a filtered tree control
	 */
	public static FilteredTree createFilteredTree(final Composite parent,
			final Collection<ModelQuerySet> availableQuerySets, final ViewerFilter viewerFilter,
			final boolean multiSelection) {
		final Collection<ModelQuerySet> querySets;
		if (availableQuerySets == null) {
			querySets = ModelQuerySetCatalog.getSingleton().getAllModelQuerySets();
		} else {
			querySets = availableQuerySets;
		}

		final PatternFilter filter = new PatternFilter();
		final int selectionStyle;
		if (multiSelection) {
			selectionStyle = SWT.MULTI;
		} else {
			selectionStyle = SWT.SINGLE;
		}
		final FilteredTree tree = new FilteredTree(parent, selectionStyle | SWT.H_SCROLL
				| SWT.V_SCROLL | SWT.BORDER, filter, true);

		final TreeViewer viewer = tree.getViewer();
		viewer.setContentProvider(new ITreeContentProvider() {

			public Object[] getElements(final Object inputElement) {
				@SuppressWarnings("unchecked")
				final java.util.List<ModelQuerySet> querySets = (java.util.List<ModelQuerySet>) inputElement;
				return querySets.toArray();
			}

			public Object[] getChildren(final Object parentElement) {
				if (parentElement instanceof ModelQuerySet) {
					final ModelQuerySet querySet = (ModelQuerySet) parentElement;
					return querySet.getQueries().toArray();
				}
				return null;
			}

			public Object getParent(final Object element) {
				if (element instanceof ModelQuery) {
					final ModelQuery query = (ModelQuery) element;
					return query.getModelQuerySet();
				}
				return null;
			}

			public boolean hasChildren(final Object element) {
				if (element instanceof ModelQuerySet) {
					final ModelQuerySet querySet = (ModelQuerySet) element;
					return !querySet.getQueries().isEmpty();
				}
				return false;
			}

			public void inputChanged(final Viewer viewer, final Object oldInput,
					final Object newInput) {
			}

			public void dispose() {
			}
		});

		viewer.setLabelProvider(new LabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof ModelQuerySet) {
					final ModelQuerySet querySet = (ModelQuerySet) element;
					return querySet.getName();
				} else if (element instanceof ModelQuery) {
					final ModelQuery query = (ModelQuery) element;
					return query.getName();
				}
				return super.getText(element);
			}

			@Override
			public Image getImage(final Object element) {
				return ImageUtils.getImage(element);
			}
		});

		if (viewerFilter != null) {
			viewer.setFilters(new ViewerFilter[] { viewerFilter });
		}

		viewer.setInput(querySets);
		return tree;
	}
}
