/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    Grgoire Dup (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.infra.query.ui.views;

import java.net.URI;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.ui.URIEditorInput;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmt.modisco.common.core.builder.AbstractMoDiscoCatalog.ModiscoCatalogChangeListener;
import org.eclipse.gmt.modisco.common.core.logging.MoDiscoLogger;
import org.eclipse.gmt.modisco.common.core.utils.ImageUtils;
import org.eclipse.gmt.modisco.infra.query.ModelQuery;
import org.eclipse.gmt.modisco.infra.query.ModelQuerySet;
import org.eclipse.gmt.modisco.infra.query.core.ModelQuerySetCatalog;
import org.eclipse.gmt.modisco.infra.query.ui.Activator;
import org.eclipse.gmt.modisco.infra.query.ui.Messages;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.IOpenListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.OpenEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.ViewPart;

/**
 * An Eclipse view that displays a list of available MoDisco query sets.
 * Clicking on a query set opens it in the query set editor.
 */
public class ModelQueryView extends ViewPart implements ModiscoCatalogChangeListener {

	private static final int JOB_SCHEDULE_DELAY = 500;
	private static final int COLUMN_WIDTH = 200;
	private TreeViewer modelQuerySetViewer;
	private TreeViewerColumn nameColumn;
	private TreeViewerColumn locationColumn;
	private Job refreshJob = null;

	public TreeViewer getModelQuerySetViewer() {
		return this.modelQuerySetViewer;
	}

	public TreeViewerColumn getNameColumn() {
		return this.nameColumn;
	}

	@Override
	public void createPartControl(final Composite parent) {

		parent.setLayout(new FillLayout());

		this.modelQuerySetViewer = new TreeViewer(parent, SWT.H_SCROLL | SWT.V_SCROLL | SWT.SINGLE
				| SWT.FULL_SELECTION);
		this.modelQuerySetViewer.getTree().setLinesVisible(true);
		this.modelQuerySetViewer.getTree().setHeaderVisible(true);

		this.modelQuerySetViewer.addOpenListener(new IOpenListener() {
			public void open(final OpenEvent event) {
				IStructuredSelection selection = (IStructuredSelection) ModelQueryView.this
						.getModelQuerySetViewer().getSelection();

				EObject modelElement = (EObject) selection.getFirstElement();
				ModelQuerySet modelQuerySet = null;
				if (modelElement instanceof ModelQuerySet) {
					modelQuerySet = (ModelQuerySet) modelElement;
				} else if (modelElement instanceof ModelQuery) {
					ModelQuery modelQuery = (ModelQuery) modelElement;
					modelQuerySet = modelQuery.getModelQuerySet();
				}
				if (modelQuerySet != null) {
					// open query set in editor
					URI uri = null;
					try {
						String uriString = ModelQuerySetCatalog.getSingleton().getURI(
								modelQuerySet.getName()).toString();
						uri = URI.create(uriString);
						if ("file".equals(uri.getScheme())) { //$NON-NLS-1$
							IDE.openEditor(PlatformUI.getWorkbench().getActiveWorkbenchWindow()
									.getActivePage(), new URI(modelQuerySet.eResource().getURI()
									.toString()), Activator.QUERY_EDITOR_ID, true);
						} else {
							URIEditorInput uriEditorInput = new URIEditorInput(
									org.eclipse.emf.common.util.URI.createURI(uri.toString()));
							IDE.openEditor(PlatformUI.getWorkbench().getActiveWorkbenchWindow()
									.getActivePage(), uriEditorInput, Activator.QUERY_EDITOR_ID,
									true);
						}

					} catch (Exception e) {
						MoDiscoLogger.logError(e, "Failed to open: " + uri, Activator.getDefault()); //$NON-NLS-1$
					}
				}
			}
		});

		createNameColumn();

		createLocationColumn();

		this.modelQuerySetViewer.setContentProvider(new ITreeContentProvider() {
			public Object[] getElements(final Object inputElement) {
				if (!(inputElement instanceof Object[])) {
					throw new RuntimeException("Unexpected inputElement type: " //$NON-NLS-1$
							+ inputElement.getClass().getName());
				}
				return (Object[]) inputElement;
			}

			public void inputChanged(final Viewer viewer, final Object oldInput,
					final Object newInput) {
				// Nothing to do
			}

			public Object[] getChildren(final Object parentElement) {
				if (!(parentElement instanceof ModelQuerySet)) {
					throw new RuntimeException("Unexpected inputElement type: " //$NON-NLS-1$
							+ parentElement.getClass().getName());
				}
				ModelQuerySet modelQuerySet = (ModelQuerySet) parentElement;
				return modelQuerySet.getQueries().toArray();
			}

			public Object getParent(final Object element) {
				if (element instanceof ModelQuerySet) {
					return null;
				} else if (element instanceof ModelQuery) {
					ModelQuery modelQuery = (ModelQuery) element;
					return modelQuery.getModelQuerySet();
				} else {
					throw new RuntimeException("Unexpected inputElement type: " //$NON-NLS-1$
							+ element.getClass().getName());
				}
			}

			public boolean hasChildren(final Object element) {
				return (element instanceof ModelQuerySet);
			}

			public void dispose() {
				// Nothing to do
			}
		});

		getSite().setSelectionProvider(this.modelQuerySetViewer);
		refresh(false);

		ModelQuerySetCatalog.getSingleton().addChangeListener(this);
	}

	private void createLocationColumn() {
		this.locationColumn = new TreeViewerColumn(this.modelQuerySetViewer, SWT.NONE);
		this.locationColumn.getColumn().setText(Messages.ModelQueryView_location);
		this.locationColumn.getColumn().setWidth(ModelQueryView.COLUMN_WIDTH * 2);
		this.locationColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				if (!(element instanceof ModelQuerySet)) {
					return ""; //$NON-NLS-1$
				}
				ModelQuerySet modelQuerySet = (ModelQuerySet) element;
				return ModelQuerySetCatalog.getSingleton().getURI(modelQuerySet.getName())
						.toString();
			}
		});
		addSorter(this.locationColumn, new ViewerSorter() {
			@Override
			public int compare(final Viewer viewer, final Object e1, final Object e2) {
				ModelQuerySet modelQuerySet1 = (ModelQuerySet) e1;
				ModelQuerySet modelQuerySet2 = (ModelQuerySet) e2;
				String uri1 = ModelQuerySetCatalog.getSingleton().getURI(modelQuerySet1.getName())
						.toString();
				String uri2 = ModelQuerySetCatalog.getSingleton().getURI(modelQuerySet2.getName())
						.toString();
				int compare = uri1.compareTo(uri2);
				if (viewer instanceof TreeViewer) {
					TreeViewer structuredViewer = (TreeViewer) viewer;
					if (structuredViewer.getTree().getSortDirection() == SWT.DOWN) {
						compare = compare * -1;
					}
				}
				return compare;
			}
		});
	}

	private void createNameColumn() {
		this.nameColumn = new TreeViewerColumn(this.modelQuerySetViewer, SWT.NONE);
		this.nameColumn.getColumn().setText(Messages.ModelQueryView_name);
		this.nameColumn.getColumn().setWidth(ModelQueryView.COLUMN_WIDTH);
		this.nameColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof ModelQuerySet) {
					ModelQuerySet modelQuerySet = (ModelQuerySet) element;
					return modelQuerySet.getName();
				} else if (element instanceof ModelQuery) {
					ModelQuery modelQuery = (ModelQuery) element;
					return modelQuery.getName();
				} else {
					return ""; //$NON-NLS-1$
				}

			}

			@Override
			public Image getImage(final Object element) {
				return ImageUtils.getImage(element);
			}
		});

		addSorter(this.nameColumn, new ViewerSorter() {
			@Override
			public int compare(final Viewer viewer, final Object e1, final Object e2) {
				ModelQuerySet modelQuerySet1 = (ModelQuerySet) e1;
				ModelQuerySet modelQuerySet2 = (ModelQuerySet) e2;
				int compare = modelQuerySet1.getName().compareTo(modelQuerySet2.getName());
				if (viewer instanceof TreeViewer) {
					TreeViewer structuredViewer = (TreeViewer) viewer;
					if (structuredViewer.getTree().getSortDirection() == SWT.DOWN) {
						compare = compare * -1;
					}
				}
				return compare;
			}
		});
	}

	private void addSorter(final TreeViewerColumn column, final ViewerSorter viewerSorter) {
		column.getColumn().addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				TreeViewer treeView = ModelQueryView.this.getModelQuerySetViewer();
				treeView.setSorter(viewerSorter);
				treeView.getTree().setSortColumn(column.getColumn());
				if (treeView.getTree().getSortDirection() == SWT.UP) {
					treeView.getTree().setSortDirection(SWT.DOWN);
				} else if (treeView.getTree().getSortDirection() == SWT.DOWN) {
					treeView.getTree().setSortDirection(SWT.UP);
				} else {
					treeView.getTree().setSortDirection(SWT.UP);
				}
				treeView.refresh();
			}
		});

	}

	@Override
	public void setFocus() {
		this.modelQuerySetViewer.getControl().setFocus();
	}

	@Override
	public void dispose() {
		super.dispose();
		ModelQuerySetCatalog.getSingleton().removeChangeListener(this);
	}

	/** Called when a query set changes */
	public void changed(final EObject changedQuerySet, final IFile file) {
		refresh(true);
	}

	/** Called when a query set is added */
	public void added(final EObject addedQuerySet, final IFile file) {
		refresh(true);
	}

	/** Called when a query set is removed */
	public void removed(final IFile querySetFile) {
		refresh(true);
	}

	/** Optionally delayed refresh */
	private void refresh(final boolean delayed) {
		if (this.refreshJob == null) {
			this.refreshJob = new Job(Messages.refreshingQuerySetsView) {
				@Override
				protected IStatus run(final IProgressMonitor monitor) {
					Display.getDefault().syncExec(new Runnable() {
						public void run() {
							ModelQueryView.this.getModelQuerySetViewer().setInput(
									ModelQuerySetCatalog.getSingleton().getAllModelQuerySets()
											.toArray());
							ModelQueryView.this.getNameColumn().getColumn().pack();
							ModelQueryView.this.getModelQuerySetViewer().refresh();
						}
					});
					return Status.OK_STATUS;
				}
			};
		}
		if (delayed) {
			// delayed until it stops changing
			this.refreshJob.cancel();
			this.refreshJob.setPriority(Job.DECORATE);
			this.refreshJob.schedule(ModelQueryView.JOB_SCHEDULE_DELAY);
		} else {
			// force synchronous execution
			this.refreshJob.schedule();
			try {
				this.refreshJob.join();
			} catch (InterruptedException e) {
				MoDiscoLogger.logError(e, Activator.getDefault());
			}
		}
	}

}
