/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Grgoire Dup (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.ui.views;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EPackage.Registry;
import org.eclipse.gmt.modisco.common.core.builder.EcoreCatalog;
import org.eclipse.gmt.modisco.common.core.builder.AbstractMoDiscoCatalog.ModiscoCatalogChangeListener;
import org.eclipse.gmt.modisco.common.core.logging.MoDiscoLogger;
import org.eclipse.gmt.modisco.common.core.utils.ImageUtils;
import org.eclipse.gmt.modisco.common.ui.Messages;
import org.eclipse.gmt.modisco.common.ui.MoDiscoCommonUIPlugin;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.part.ViewPart;

/**
 * An Eclipse view that displays a list of available Ecore meta-models.
 */
public class PackageRegistryView extends ViewPart implements ModiscoCatalogChangeListener {

	private static final int JOB_SCHEDULE_DELAY = 500;
	private static final int COLUMN_WIDTH = 200;
	private TreeViewer ecoreViewer;
	private TreeViewerColumn nameColumn;
	private TreeViewerColumn locationColumn;
	private Job refreshJob = null;
	private TreeViewerColumn nsURIColumn;

	public TreeViewer getModelQuerySetViewer() {
		return this.ecoreViewer;
	}

	public TreeViewerColumn getNameColumn() {
		return this.nameColumn;
	}

	@Override
	public void createPartControl(final Composite parent) {

		parent.setLayout(new FillLayout());

		this.ecoreViewer = new TreeViewer(parent, SWT.H_SCROLL | SWT.V_SCROLL | SWT.SINGLE
				| SWT.FULL_SELECTION);
		this.ecoreViewer.getTree().setLinesVisible(true);
		this.ecoreViewer.getTree().setHeaderVisible(true);

		createNameColumn();
		createLocationColumn();
		createNsUriColumn();

		this.ecoreViewer.setContentProvider(new ITreeContentProvider() {
			public Object[] getElements(final Object inputElement) {
				if (inputElement instanceof EPackage.Registry) {
					EPackage.Registry registry = (EPackage.Registry) inputElement;
					List<EPackage> ePackages = new ArrayList<EPackage>();
					// avoid a ConcurrentModificationException
					List<String> keys = new ArrayList<String>();
					keys.addAll(registry.keySet());
					for (String nsURI : keys) {
						try {
							EPackage ePackage = registry.getEPackage(nsURI);
							if (ePackage == null) {
								throw new Exception("ePackage is null for : " //$NON-NLS-1$
										+ nsURI);
							}
							ePackages.add(ePackage);
						} catch (Exception e) {
							MoDiscoLogger.logWarning(e,
									"An error happened while loading an EPackage from Package.Registry.INSTANCE: " //$NON-NLS-1$
											+ nsURI, MoDiscoCommonUIPlugin.getDefault());
						}
					}
					return ePackages.toArray();
				} else if (inputElement == null) {
					return new Object[] {};
				} else if (inputElement instanceof String) {
					String message = (String) inputElement;
					return new Object[] { message };
				} else {
					throw new RuntimeException("Unexpected element type: " //$NON-NLS-1$
							+ inputElement.getClass().getName());
				}
			}

			public void inputChanged(final Viewer viewer, final Object oldInput,
					final Object newInput) {
				// Nothing to do
			}

			public Object[] getChildren(final Object parentElement) {
				if (parentElement instanceof EPackage.Registry) {
					EPackage.Registry registry = (EPackage.Registry) parentElement;
					return registry.values().toArray();
				} else if (parentElement == null || parentElement instanceof String) {
					return new Object[] {};
				} else {
					throw new RuntimeException("Unexpected element type: " //$NON-NLS-1$
							+ parentElement.getClass().getName());
				}
			}

			public Object getParent(final Object element) {
				return null;
			}

			public boolean hasChildren(final Object element) {
				if (element instanceof EPackage.Registry) {
					return true;
				}
				return false;
			}

			public void dispose() {
				// Nothing to do
			}
		});

		getSite().setSelectionProvider(this.ecoreViewer);
		refresh();

		EcoreCatalog.getSingleton().addChangeListener(this);
	}

	private void createNsUriColumn() {
		this.nsURIColumn = new TreeViewerColumn(this.ecoreViewer, SWT.NONE);
		this.nsURIColumn.getColumn().setText(Messages.PackageRegistryView_nsURI);
		this.nsURIColumn.getColumn().setWidth(PackageRegistryView.COLUMN_WIDTH * 2);
		this.nsURIColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				if (!(element instanceof EPackage)) {
					return ""; //$NON-NLS-1$
				}
				EPackage ePackage = (EPackage) element;
				return ePackage.getNsURI();
			}
		});
		addSorter(this.nsURIColumn, new ViewerSorter() {
			@Override
			public int compare(final Viewer viewer, final Object e1, final Object e2) {
				EPackage ePackage1 = (EPackage) e1;
				EPackage ePackage2 = (EPackage) e2;
				String uri1 = ePackage1.getNsURI();
				String uri2 = ePackage2.getNsURI();
				int compare = uri1.compareTo(uri2);
				if (viewer instanceof TreeViewer) {
					TreeViewer structuredViewer = (TreeViewer) viewer;
					if (structuredViewer.getTree().getSortDirection() == SWT.DOWN) {
						compare = compare * -1;
					}
				}
				return compare;
			}
		});
	}

	private void createLocationColumn() {
		this.locationColumn = new TreeViewerColumn(this.ecoreViewer, SWT.NONE);
		this.locationColumn.getColumn().setText(Messages.PackageRegistryView_Location);
		this.locationColumn.getColumn().setWidth(PackageRegistryView.COLUMN_WIDTH * 2);
		this.locationColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				if (!(element instanceof EPackage)) {
					return ""; //$NON-NLS-1$
				}
				EPackage ePackage = (EPackage) element;
				return EcoreCatalog.getSingleton().getOriginalLocation(ePackage);
			}
		});
		addSorter(this.locationColumn, new ViewerSorter() {
			@Override
			public int compare(final Viewer viewer, final Object e1, final Object e2) {
				EPackage ePackage1 = (EPackage) e1;
				EPackage ePackage2 = (EPackage) e2;
				String uri1 = EcoreCatalog.getSingleton().getOriginalLocation(ePackage1);
				String uri2 = EcoreCatalog.getSingleton().getOriginalLocation(ePackage2);
				int compare = uri1.compareTo(uri2);
				if (viewer instanceof TreeViewer) {
					TreeViewer structuredViewer = (TreeViewer) viewer;
					if (structuredViewer.getTree().getSortDirection() == SWT.DOWN) {
						compare = compare * -1;
					}
				}
				return compare;
			}
		});
	}

	private void createNameColumn() {
		this.nameColumn = new TreeViewerColumn(this.ecoreViewer, SWT.NONE);
		this.nameColumn.getColumn().setText(Messages.PackageRegistryView_Name);
		this.nameColumn.getColumn().setWidth(PackageRegistryView.COLUMN_WIDTH);
		this.nameColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof EPackage) {
					EPackage ePackage = (EPackage) element;
					return ePackage.getName();
				} else if (element instanceof String) {
					return (String) element;
				}
				return ""; //$NON-NLS-1$
			}

			@Override
			public Image getImage(final Object element) {
				return ImageUtils.getImage(element);
			}
		});

		addSorter(this.nameColumn, new ViewerSorter() {
			@Override
			public int compare(final Viewer viewer, final Object e1, final Object e2) {
				EPackage ePackage1 = (EPackage) e1;
				EPackage ePackage2 = (EPackage) e2;
				int compare = ePackage1.getName().compareTo(ePackage2.getName());
				if (viewer instanceof TreeViewer) {
					TreeViewer structuredViewer = (TreeViewer) viewer;
					if (structuredViewer.getTree().getSortDirection() == SWT.DOWN) {
						compare = compare * -1;
					}
				}
				return compare;
			}
		});
	}

	private void addSorter(final TreeViewerColumn column, final ViewerSorter viewerSorter) {
		column.getColumn().addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				TreeViewer treeView = PackageRegistryView.this.getModelQuerySetViewer();
				treeView.setSorter(viewerSorter);
				treeView.getTree().setSortColumn(column.getColumn());
				if (treeView.getTree().getSortDirection() == SWT.UP) {
					treeView.getTree().setSortDirection(SWT.DOWN);
				} else if (treeView.getTree().getSortDirection() == SWT.DOWN) {
					treeView.getTree().setSortDirection(SWT.UP);
				} else {
					treeView.getTree().setSortDirection(SWT.UP);
				}
				treeView.refresh();
			}
		});

	}

	@Override
	public void setFocus() {
		this.ecoreViewer.getControl().setFocus();
	}

	@Override
	public void dispose() {
		super.dispose();
		EcoreCatalog.getSingleton().removeChangeListener(this);
	}

	/** Optionally delayed refresh */
	private void refresh() {
		if (this.refreshJob == null) {
			this.refreshJob = new Job(Messages.PackageRegistryView_RefreshingPackageRegistryView) {
				@Override
				protected IStatus run(final IProgressMonitor monitor) {
					// display a message while initializing
					// (initialization can take a while)
					Display.getDefault().syncExec(new Runnable() {
						public void run() {
							getModelQuerySetViewer().setInput("Initializing...");
						}
					});
					final EPackage.Registry registry = EPackage.Registry.INSTANCE;
					// Read a first time in a non-UI thread so as to avoid
					// blocking. Then it will be cached for the UI thread.
					preloadRegistry(registry);
					Display.getDefault().syncExec(new Runnable() {
						public void run() {
							getModelQuerySetViewer().setInput(registry);
							getNameColumn().getColumn().pack();
							getModelQuerySetViewer().refresh();
						}
					});
					return Status.OK_STATUS;
				}
			};
		}
		// delayed until it stops changing
		this.refreshJob.cancel();
		this.refreshJob.setPriority(Job.DECORATE);
		this.refreshJob.schedule(PackageRegistryView.JOB_SCHEDULE_DELAY);
	}

	/**
	 * Reads the registry in a non-UI thread, so that the UI thread will be able
	 * to access it faster
	 * 
	 * @param registry
	 *            the registry to preload
	 */
	private void preloadRegistry(final Registry registry) {
		// avoid ConcurrentModificationException
		List<String> keys = new ArrayList<String>();
		keys.addAll(registry.keySet());
		for (String nsURI : keys) {
			try {
				registry.getEPackage(nsURI);
			} catch (Exception e) {
				MoDiscoLogger.logWarning(e,
						"An error happened while loading an EPackage from Package.Registry.INSTANCE: " //$NON-NLS-1$
								+ nsURI, MoDiscoCommonUIPlugin.getDefault());
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seeorg.eclipse.gmt.modisco.common.core.builder.AbstractMoDiscoCatalog.
	 * ModiscoCatalogChangeListener#added(org.eclipse.emf.ecore.EObject,
	 * org.eclipse.core.resources.IFile)
	 */
	public void added(final EObject eObject, final IFile file) {
		refresh();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seeorg.eclipse.gmt.modisco.common.core.builder.AbstractMoDiscoCatalog.
	 * ModiscoCatalogChangeListener#changed(org.eclipse.emf.ecore.EObject,
	 * org.eclipse.core.resources.IFile)
	 */
	public void changed(final EObject eObject, final IFile file) {
		refresh();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seeorg.eclipse.gmt.modisco.common.core.builder.AbstractMoDiscoCatalog.
	 * ModiscoCatalogChangeListener#removed(org.eclipse.core.resources.IFile)
	 */
	public void removed(final IFile file) {
		refresh();
	}

}
