/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Grgoire Dup (Mia-Software)
 *    Nicolas Bros (Mia-Software)
 *******************************************************************************/
package org.eclipse.gmt.modisco.common.core.resource;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.ContentHandler;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.URIConverter;
import org.eclipse.emf.ecore.resource.URIHandler;
import org.eclipse.emf.ecore.resource.impl.ExtensibleURIConverterImpl;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.resource.impl.URIHandlerImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.gmt.modisco.common.core.CommonModiscoActivator;
import org.eclipse.gmt.modisco.common.core.builder.AbstractMoDiscoCatalog;
import org.eclipse.gmt.modisco.common.core.preferences.PreferenceConstants;
import org.eclipse.gmt.modisco.common.core.protocol.ModiscoURIHandler;

/**
 * @author Grgoire Dup
 */
public final class MoDiscoResourceSet extends ResourceSetImpl {

	private static final String DEBUG_ID = "org.eclipse.gmt.modisco.common.core/debug/MoDiscoResourceSet/debug"; //$NON-NLS-1$
	public static final boolean DEBUG = CommonModiscoActivator.getDefault().isDebugging()
			&& new Boolean(Platform.getDebugOption(MoDiscoResourceSet.DEBUG_ID));

	private static MoDiscoResourceSet resourceSet = null;
	private final Map<String, MoDiscoResourceListenerGroup> listenerGroupMap = new HashMap<String, MoDiscoResourceListenerGroup>();

	private MoDiscoResourceSet() {
		super();
	}

	public static MoDiscoResourceSet getResourceSetSingleton() {
		if (MoDiscoResourceSet.resourceSet == null) {
			MoDiscoResourceSet.resourceSet = new MoDiscoResourceSet();
		}
		return MoDiscoResourceSet.resourceSet;
	}

	/**
	 * Return a loaded (or reloaded) EMF resource.
	 * 
	 * @param uri
	 *            the resource
	 * @param listener
	 *            a listener that will be notified of each update to the
	 *            returned resource.
	 * @return an EMF resource
	 * @throws IOException
	 */
	public Resource getResource(final URI uri, final IMoDiscoResourceListener listener)
			throws IOException {
		setPackageRegistry(EPackage.Registry.INSTANCE);
		if (MoDiscoResourceSet.DEBUG) {
			IStatus status = new Status(IStatus.INFO, CommonModiscoActivator.PLUGIN_ID, "[" //$NON-NLS-1$
					+ this.getClass().getSimpleName()
					+ ".getResource(URI,Listener)] " + uri.toString()); //$NON-NLS-1$
			CommonModiscoActivator.getDefault().getLog().log(status);
		}
		for (Resource existingResource : getResources()) {
			if (existingResource.getURI().equals(uri)) {
				existingResource.unload();
				break;
			}
		}
		Resource resource = super.getResource(uri, true);
		resource.load(Collections.EMPTY_MAP);
		addListenerToReferedResource(resource, listener);
		return resource;
	}

	private void addListenerToReferedResource(final Resource resource,
			final IMoDiscoResourceListener listener) {
		if (listener != null) {
			TreeIterator<EObject> treeIterator = resource.getAllContents();
			while (treeIterator.hasNext()) {
				EObject eObject = treeIterator.next();
				for (EStructuralFeature feature : eObject.eClass().getEAllStructuralFeatures()) {
					Object object = eObject.eGet(feature);
					if (object instanceof EObject) {
						readEObject(resource, listener, object);
					} else if (object instanceof EList<?>) {
						EList<?> list = (EList<?>) object;
						for (Object listObject : list) {
							if (object instanceof EObject) {
								readEObject(resource, listener, listObject);
							}
						}
					}
				}
			}
		}
	}

	private void readEObject(final Resource resource, final IMoDiscoResourceListener listener,
			final Object object) {
		EObject referedEObject = (EObject) object;
		Resource referedObjectResource = referedEObject.eResource();
		if (referedObjectResource != resource) {
			MoDiscoResourceListenerGroup listenerGroup;
			if (referedObjectResource == null) {
				if (referedEObject.eIsProxy()) {
					throw new RuntimeException("A proxy with a null resouce has been found: " //$NON-NLS-1$
							+ EcoreUtil.getURI(referedEObject).toString());
				}
			} else {
				listenerGroup = getListenerGroup(referedEObject.eResource().getURI());
				listenerGroup.addListener(listener, resource.getURI());
			}
		}
	}

	private MoDiscoResourceListenerGroup getListenerGroup(final URI watchedResourceUri) {
		MoDiscoResourceListenerGroup listenerGroup = this.listenerGroupMap.get(watchedResourceUri
				.toString());
		if (listenerGroup == null) {
			listenerGroup = new MoDiscoResourceListenerGroup(watchedResourceUri);
			this.listenerGroupMap.put(watchedResourceUri.toString(), listenerGroup);
		}
		return listenerGroup;
	}

	/**
	 * This method is used to notify the resource set that one of its resources
	 * has been reloaded.
	 * 
	 * This method is used by the
	 * {@link AbstractMoDiscoCatalog#addWSFile(IFile)} method.
	 * 
	 * @param resource
	 *            a reloaded resource
	 */
	public void aResourceHasBeenLoaded(final Resource resource) {
		URI uri = resource.getURI();
		if (MoDiscoResourceSet.DEBUG) {
			IStatus status = new Status(IStatus.INFO, CommonModiscoActivator.PLUGIN_ID, "[" //$NON-NLS-1$
					+ this.getClass().getSimpleName() + ".aResourceAsBeenLoaded(Resource)] " //$NON-NLS-1$
					+ uri.toString());
			CommonModiscoActivator.getDefault().getLog().log(status);
		}
		MoDiscoResourceListenerGroup listenerGroup = getListenerGroup(uri);
		listenerGroup.notifyChange();
	}

	@Override
	public URIConverter getURIConverter() {
		if (this.uriConverter == null) {
			this.uriConverter = createURIConverter();
		}
		return this.uriConverter;
	}

	/**
	 * Create a URI converter that includes the {@link ModiscoURIHandler}, and
	 * that excludes {@link URIHandlerImpl} if the user chose to disable it from
	 * the preferences ({@link PreferenceConstants#P_DISABLE_DEFAULT_URIHANDLER}
	 * )
	 * 
	 * @author nbros
	 */
	public static URIConverter createURIConverter() {
		final List<URIHandler> defaultHandlers = URIHandler.DEFAULT_HANDLERS;
		final List<URIHandler> uriHandlers = new ArrayList<URIHandler>();
		if (CommonModiscoActivator.getDefault().getPreferenceStore().getBoolean(
				PreferenceConstants.P_DISABLE_DEFAULT_URIHANDLER)) {
			uriHandlers.add(new ModiscoURIHandler());
			for (URIHandler uriHandler : defaultHandlers) {
				if (uriHandler.getClass() != URIHandlerImpl.class) {
					uriHandlers.add(uriHandler);
				}
			}
		} else {
			uriHandlers.add(new ModiscoURIHandler());
			uriHandlers.addAll(defaultHandlers);
		}
		return new ExtensibleURIConverterImpl(uriHandlers, ContentHandler.Registry.INSTANCE
				.contentHandlers());
	}
}
