/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Nicolas Bros (Mia-Software)
 *******************************************************************************/
package org.eclipse.gmt.modisco.infra.browser.actions;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.emf.common.util.EList;
import org.eclipse.gmt.modisco.infra.browser.Messages;
import org.eclipse.gmt.modisco.infra.browser.MoDiscoBrowserPlugin;
import org.eclipse.gmt.modisco.infra.browser.custom.MetamodelView;
import org.eclipse.gmt.modisco.infra.browser.custom.TypeView;
import org.eclipse.gmt.modisco.infra.browser.editors.EcoreBrowser;
import org.eclipse.gmt.modisco.infra.browser.ui.LoadCustomizationsDialog;
import org.eclipse.gmt.modisco.infra.browser.util.EMFUtil;
import org.eclipse.gmt.modisco.infra.browser.util.ImageProvider;
import org.eclipse.gmt.modisco.infra.role.Role;
import org.eclipse.gmt.modisco.infra.role.RoleSet;
import org.eclipse.gmt.modisco.infra.role.core.RoleSetCatalog;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.window.Window;
import org.eclipse.osgi.util.NLS;

public class LoadCustomizationsAction extends Action {

	private final IBrowserActionBarContributor browserActionBarContributor;

	public LoadCustomizationsAction(final IBrowserActionBarContributor browserActionBarContributor) {
		super();
		setToolTipText(Messages.BrowserActionBarContributor_loadCustomization);
		setImageDescriptor(ImageProvider.getInstance().getAddUiCustomIconDescriptor());
		this.browserActionBarContributor = browserActionBarContributor;
	}

	@Override
	public void run() {

		final EcoreBrowser browser = this.browserActionBarContributor.getBrowser();
		if (browser != null) {
			final List<MetamodelView> registeredCustomizations = browser.getBrowserConfiguration()
					.getCustomizationEngine().getRegisteredCustomizations();

			final LoadCustomizationsDialog loadCustomizationsDialog = new LoadCustomizationsDialog(
					browser.getSite().getShell(), registeredCustomizations);
			if (loadCustomizationsDialog.open() == Window.OK) {
				try {
					List<MetamodelView> selectedCustomizations = loadCustomizationsDialog
							.getSelectedCustomizations();

					if (loadCustomizationsDialog.isLoadRequiredRolesSelected()) {
						// load roles corresponding to customizations
						loadRolesForCustomizations(selectedCustomizations, browser);
					}

					browser.loadCustomizations(selectedCustomizations);

				} catch (final Exception e) {
					MoDiscoBrowserPlugin.logException(
							Messages.BrowserActionBarContributor_loadingCustomizationFailed, e);
				}
			}
		}
	}

	private void loadRolesForCustomizations(final List<MetamodelView> customizations,
			final EcoreBrowser browser) {
		final Set<Role> referencedRoles = new HashSet<Role>();
		final Collection<RoleSet> roleSets = RoleSetCatalog.getSingleton().getAllRoleSets();

		for (MetamodelView customization : customizations) {
			String metamodelURI = customization.getMetamodelURI();
			// find customized RoleSet
			RoleSet customizedRoleSet = null;
			if (metamodelURI != null) {
				for (RoleSet roleSet : roleSets) {
					if (metamodelURI.equals(roleSet.getNsURI())) {
						customizedRoleSet = roleSet;
						break;
					}
				}
			}
			if (customizedRoleSet == null) {
				continue;
			}

			// find customized Roles
			EList<TypeView> types = customization.getTypes();
			for (TypeView typeView : types) {
				String metaclassName = typeView.getMetaclassName();
				Role role = findRoleWithFullyQualifiedName(metaclassName, customizedRoleSet);
				if (role != null) {
					referencedRoles.add(role);
				} else {
					MoDiscoBrowserPlugin.logWarning(NLS.bind(
							Messages.BrowserActionBarContributor_missingRequiredRole, new Object[] {
									metaclassName, customizedRoleSet.getName(),
									customization.getName() }));
				}
			}

			for (Role referencedRole : referencedRoles) {
				browser.getBrowserConfiguration().loadRole(referencedRole);
			}
		}
		browser.reloadModel();
	}

	private Role findRoleWithFullyQualifiedName(final String metaclassName,
			final RoleSet customizedRoleSet) {
		EList<Role> roles = customizedRoleSet.getRoles();
		for (Role role : roles) {
			String roleName = EMFUtil.getMetaclassQualifiedName(role);
			if (metaclassName.equals(roleName)) {
				return role;
			}
		}
		return null;
	}

}
