/*******************************************************************************
 * Copyright (c) 2008, 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.infra.browser.core;

import java.util.ArrayList;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmt.modisco.infra.browser.Messages;
import org.eclipse.gmt.modisco.infra.browser.MoDiscoBrowserPlugin;
import org.eclipse.gmt.modisco.infra.browser.customization.CustomizationEngine;
import org.eclipse.gmt.modisco.infra.browser.editors.BrowserConfiguration;
import org.eclipse.gmt.modisco.infra.browser.util.EMFUtil;
import org.eclipse.gmt.modisco.infra.role.Role;
import org.eclipse.osgi.util.NLS;

/**
 * All the model elements for a metaclass. Each {@link InstancesForMetaclass}
 * corresponds to a metaclass, and has a list of all the instances of this
 * metaclass in the model.
 */
public class InstancesForMetaclass {

	/** All the model elements of the metaclass */
	private final ArrayList<EObject> elements = new ArrayList<EObject>();

	/** All the model elements defined on derived metaclasses */
	private final ArrayList<EObject> derivedElements = new ArrayList<EObject>();

	/** The list of subclasses of this metaclass */
	private final ArrayList<InstancesForMetaclass> subclasses = new ArrayList<InstancesForMetaclass>();

	/** The EClass corresponding to this metaclass */
	private final EClass eClass;

	/**
	 * The list of all metaclasses. Used to find the instance list of super
	 * metaclasses.
	 */
	private final InstancesForMetaclasses instancesForMetaclasses;

	private final BrowserConfiguration browserConfiguration;

	private ArrayList<EObject> cachedElements = null;
	private int cachedElementsModCount = -1;
	private ArrayList<EObject> cachedDerivedElements = null;
	private int cachedDerivedElementsModCount = -1;

	/**
	 * @param eClass
	 *            the {@link EClass} corresponding to the metaclass.
	 * @param browserConfiguration
	 */
	public InstancesForMetaclass(final EClass eClass,
			final InstancesForMetaclasses instancesForMetaclasses,
			final BrowserConfiguration browserConfiguration) {
		this.eClass = eClass;
		this.instancesForMetaclasses = instancesForMetaclasses;
		this.browserConfiguration = browserConfiguration;
	}

	public ArrayList<EObject> getElements() {
		// cache visible elements list (with modcount)
		if (this.cachedElements == null
				|| this.cachedElementsModCount != this.browserConfiguration.getModCount()) {
			final CustomizationEngine customizationEngine = this.browserConfiguration
					.getCustomizationEngine();
			this.cachedElements = customizationEngine.filterVisible(this.elements);
			this.cachedElementsModCount = this.browserConfiguration.getModCount();
		}
		return this.cachedElements;
	}

	public ArrayList<EObject> getDerivedElements() {
		// cache visible elements list (with modcount)
		if (this.cachedDerivedElements == null
				|| this.cachedDerivedElementsModCount != this.browserConfiguration.getModCount()) {
			final CustomizationEngine customizationEngine = this.browserConfiguration
					.getCustomizationEngine();
			this.cachedDerivedElements = customizationEngine.filterVisible(this.derivedElements);
			this.cachedDerivedElementsModCount = this.browserConfiguration.getModCount();
		}
		return this.cachedDerivedElements;
	}

	/**
	 * Add this instance to the list of instances of the metaclass
	 * 
	 * @param addDerived
	 *            whether to also add this object to super-classes
	 */
	public void add(final EObject object, final boolean addDerived) {
		this.elements.add(object);

		// also add this object to super-classes
		if (addDerived) {
			addDerived(object);
		}
	}

	/** Add this instance to the list of instances of the metaclass */
	public void roleAdd(final EObject object) {
		this.elements.add(object);
	}

	/**
	 * Add this instance to the list of instances of metaclasses this metaclass
	 * is derived from
	 */
	private void addDerived(final EObject object) {
		final EList<EClass> allSuperTypes = this.eClass.getEAllSuperTypes();
		for (final EClass superclass : allSuperTypes) {
			final InstancesForMetaclass instancesForSuperclass = this.instancesForMetaclasses
					.getInstancesForMetaclass(EMFUtil.getMetaclassQualifiedName(superclass));
			if (instancesForSuperclass != null) {
				instancesForSuperclass.derivedElements.add(object);
			} else {
				MoDiscoBrowserPlugin.logWarning(NLS.bind(Messages.InstancesForMetaclass_notFound,
						EMFUtil.getMetaclassQualifiedName(superclass)));
			}
		}
	}

	/** Remove this instance from the list of instances of the metaclass */
	public void remove(final EObject object) {
		object.eAdapters().remove(this);
		this.elements.remove(object);

		// also remove this object from super-classes
		removeDerived(object, false);
	}

	/**
	 * Remove this instance from the list of instances of metaclasses this
	 * metaclass is derived from
	 */
	private void removeDerived(final EObject object, final boolean derived) {
		if (derived) {
			this.derivedElements.remove(object);
		}

		final EList<EClass> allSuperTypes = this.eClass.getEAllSuperTypes();
		for (final EClass superclass : allSuperTypes) {
			final InstancesForMetaclass instancesForSuperclass = this.instancesForMetaclasses
					.getInstancesForMetaclass(EMFUtil.getMetaclassQualifiedName(superclass));
			instancesForSuperclass.removeDerived(object, true);
		}
	}

	/** Cached class qualified name */
	private String classQualifiedName = null;

	/**
	 * A parent for the object when it is in a tree (when metaclasses are
	 * grouped by packages)
	 */
	private Object parent;

	/** @return the number of instances of the metaclass */
	public int size() {
		return getElements().size();
	}

	/** @return the number of instances of this metaclass or its subclasses */
	public int totalSize() {
		return getElements().size() + getDerivedElements().size();
	}

	/** @return the EClass corresponding to this MetaclassListItemProvider */
	public EClass getEClass() {
		return this.eClass;
	}

	public String getClassQualifiedName() {
		if (this.classQualifiedName == null) {
			this.classQualifiedName = EMFUtil.getMetaclassQualifiedName(getEClass());
		}
		return this.classQualifiedName;
	}

	public void setParent(final Object parent) {
		this.parent = parent;
	}

	public Object getParent() {
		return this.parent;
	}

	/** Add this class to the subclasses of its parents */
	public void buildParentsSubclasses() {
		final EList<EClass> superTypes = this.eClass.getESuperTypes();
		for (final EClass superclass : superTypes) {
			final InstancesForMetaclass instancesForSuperclass = this.instancesForMetaclasses
					.getInstancesForMetaclass(EMFUtil.getMetaclassQualifiedName(superclass));
			if (instancesForSuperclass != null) {
				instancesForSuperclass.addSubclass(this);
			} else if (!(this.eClass instanceof Role)) {
				MoDiscoBrowserPlugin.logWarning(NLS.bind(Messages.InstancesForMetaclass_notFound,
						EMFUtil.getMetaclassQualifiedName(superclass)));
			}
		}
	}

	/** Clears the list of subclasses of this class */
	public void clearSubclasses() {
		this.subclasses.clear();
	}

	public InstancesForMetaclass[] getSubclasses() {
		return this.subclasses.toArray(new InstancesForMetaclass[this.subclasses.size()]);
	}

	/** Add the given class to the subclasses of this class */
	private void addSubclass(final InstancesForMetaclass instancesForMetaclass) {
		this.subclasses.add(instancesForMetaclass);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return getClassQualifiedName() + " " + this.elements.size(); //$NON-NLS-1$
	}
}
