/*******************************************************************************
 * Copyright (c) 2008, 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.infra.browser.editors;

import java.util.Iterator;

import org.eclipse.gmt.modisco.infra.browser.Messages;
import org.eclipse.gmt.modisco.infra.browser.actions.BrowseAction;
import org.eclipse.gmt.modisco.infra.browser.actions.DecreaseFontSizeAction;
import org.eclipse.gmt.modisco.infra.browser.actions.ExecuteQueryAction;
import org.eclipse.gmt.modisco.infra.browser.actions.ExecuteSelectedQueryAction;
import org.eclipse.gmt.modisco.infra.browser.actions.IBrowserActionBarContributor;
import org.eclipse.gmt.modisco.infra.browser.actions.IncreaseFontSizeAction;
import org.eclipse.gmt.modisco.infra.browser.actions.LoadCustomizationsAction;
import org.eclipse.gmt.modisco.infra.browser.actions.LoadRolesAction;
import org.eclipse.gmt.modisco.infra.browser.actions.OpenTableEditorAction;
import org.eclipse.gmt.modisco.infra.browser.actions.RefreshViewerAction;
import org.eclipse.gmt.modisco.infra.browser.actions.RemoveQueryAction;
import org.eclipse.gmt.modisco.infra.browser.actions.RestoreFontSizeAction;
import org.eclipse.gmt.modisco.infra.browser.actions.SelectQueryAction;
import org.eclipse.gmt.modisco.infra.browser.actions.ShowPropertiesViewAction;
import org.eclipse.gmt.modisco.infra.browser.core.ModelElementItem;
import org.eclipse.gmt.modisco.infra.browser.core.QueryItem;
import org.eclipse.gmt.modisco.infra.browser.editors.table.TableEditor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.part.EditorActionBarContributor;

/** Action bar contributor for the MoDisco Ecore model browser. */
public class BrowserActionBarContributor extends EditorActionBarContributor implements
		IBrowserSelectionChangedListener, IMenuListener, IBrowserActionBarContributor {

	/** This keeps track of the active editor. */
	private IEditorPart fActiveEditor;

	/** This keeps track of the current selection provider. */
	private ISelectionProvider selectionProvider;

	/** This action opens the "Properties" view. */
	private final IAction showPropertiesViewAction = new ShowPropertiesViewAction();

	public EcoreBrowser getBrowser() {
		return (EcoreBrowser) this.fActiveEditor;
	}

	/** This action opens a table editor on the current selection. */
	private final IAction openTableEditorAction = new OpenTableEditorAction(this);

	/**
	 * This action refreshes the viewer of the current editor if the editor
	 * implements {@link org.eclipse.emf.common.ui.viewer.IViewerProvider}.
	 */
	private final IAction refreshViewerAction = new RefreshViewerAction(this);

	/**
	 * This action makes it possible to select a query to be shown on each
	 * occurrence of the selected model element
	 */
	private final IAction selectQueryAction = new SelectQueryAction(this);

	/** Executes the selected query(-ies) */
	private final IAction executeSelectedQueryAction = new ExecuteSelectedQueryAction(this);

	/** Removes the selected query(-ies) */
	private final IAction removeQueryAction = new RemoveQueryAction(this);

	/** Opens the "Query Execution" view with the selected elements as context */
	private final IAction executeQueryAction = new ExecuteQueryAction(this);

	/**
	 * The current selection, updated when an event is received from the
	 * selection provider
	 */
	private ISelection fSelection;

	public ISelection getSelection() {
		return this.fSelection;
	}

	/**
	 * An action to show the selected element in the list of instances of its
	 * metaclass
	 */
	private final Action browseAction = new BrowseAction(this);

	/** An action to increase font size */
	private final Action increaseFontSizeAction = new IncreaseFontSizeAction(this);

	/** An action to decrease font size */
	private final Action decreaseFontSizeAction = new DecreaseFontSizeAction(this);

	/** An action to restore font size */
	private final Action restoreFontSizeAction = new RestoreFontSizeAction(this);

	private final Action loadCustomizationsAction = new LoadCustomizationsAction(this);

	private final Action loadRolesAction = new LoadRolesAction(this);

	/** This adds {@link Action}s to the tool bar. */
	@Override
	public void contributeToToolBar(final IToolBarManager toolBarManager) {
		toolBarManager.add(this.increaseFontSizeAction);
		toolBarManager.add(this.decreaseFontSizeAction);
		toolBarManager.add(this.restoreFontSizeAction);
		toolBarManager.add(this.loadCustomizationsAction);
		toolBarManager.add(this.loadRolesAction);
	}

	/**
	 * When the active editor changes, this remembers the change and registers
	 * with it as a selection provider.
	 */
	@Override
	public void setActiveEditor(final IEditorPart part) {
		super.setActiveEditor(part);
		this.fActiveEditor = part;

		// Switch to the new selection provider.
		if (this.selectionProvider != null) {
			this.selectionProvider.removeSelectionChangedListener(this);
		}

		if (part == null) {
			this.selectionProvider = null;
		} else {
			if (part instanceof EcoreBrowser) {
				final EcoreBrowser browser = (EcoreBrowser) part;
				this.selectionProvider = browser.getSite().getSelectionProvider();
				this.selectionProvider.addSelectionChangedListener(this);

				// Fake a selection changed event to update the menus.
				if (browser.getEditorSelection() != null) {
					selectionChanged(new SelectionChangedEvent(this.selectionProvider, browser
							.getEditorSelection()));
				}
			}
		}
	}

	/**
	 * This implements
	 * {@link org.eclipse.jface.viewers.ISelectionChangedListener}, memorizing
	 * {@link org.eclipse.jface.viewers.SelectionChangedEvent}s to be able to
	 * build the menu when they are about to be shown (see
	 * {@link BrowserActionBarContributor#menuAboutToShow(IMenuManager)}).
	 */
	public void selectionChanged(final SelectionChangedEvent event) {
		this.fSelection = event.getSelection();
	}

	/** Update actions state depending on the selection in the editor */
	private void updateActions(final ISelection selection) {
		this.browseAction.setEnabled(enableBrowse(selection));
		this.openTableEditorAction.setEnabled(TableEditor.canBeOpenedOnSelection(selection));
	}

	/**
	 * Whether to enable the browse action. The browse action is enabled only if
	 * the selected element is an EObject which is part of the first resource of
	 * the resource set.
	 */
	private boolean enableBrowse(final ISelection selection) {
		if (selection instanceof IStructuredSelection) {
			final IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			if (structuredSelection.size() != 1) {
				return false;
			}
			final Object element = structuredSelection.getFirstElement();
			if (element instanceof ModelElementItem) {
				return true;
				// return EMFUtil.isInFirstResource(((ModelElementItem)
				// element).getEObject());
			}
		}
		return false;
	}

	/** This populates the pop-up menu before it appears. */
	public void menuAboutToShow(final IMenuManager menuManager) {
		updateActions(getSelection());

		menuManager.add(this.browseAction);
		menuManager.add(this.openTableEditorAction);

		int nSelectedModelElementItems = selectionCount(ModelElementItem.class);
		if (nSelectedModelElementItems > 0) {
			menuManager.add(this.executeQueryAction);
			menuManager.add(this.selectQueryAction);
			if (nSelectedModelElementItems > 1) {
				this.selectQueryAction.setText(Messages.SelectQueryAction_addQueryOnElements);
			} else {
				this.selectQueryAction.setText(Messages.SelectQueryAction_addQueryOnElement);
			}
		}

		int nSelectedQueries = selectionCount(QueryItem.class);
		if (nSelectedQueries > 0) {
			menuManager.add(this.executeSelectedQueryAction);
			menuManager.add(this.removeQueryAction);
			if (nSelectedQueries > 1) {
				this.executeSelectedQueryAction.setText(Messages.ExecuteQueryAction_executeQueries);
				this.removeQueryAction.setText(Messages.RemoveQueryAction_removeQueries);
			} else {
				this.executeSelectedQueryAction
						.setText(Messages.ExecuteQueryAction_executeSelectedQuery);
				this.removeQueryAction.setText(Messages.RemoveQueryAction_removeQuery);
			}
		}
		menuManager.add(new Separator());
		menuManager.add(this.showPropertiesViewAction);
		menuManager.add(this.refreshViewerAction);
	}

	private int selectionCount(final Class<?> clazz) {
		ISelection selection = getSelection();
		int count = 0;
		if (selection instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			Iterator<?> iterator = structuredSelection.iterator();
			while (iterator.hasNext()) {
				Object element = iterator.next();
				if (clazz.isAssignableFrom(element.getClass())) {
					count++;
				}
			}
		}
		return count;
	}

	public IEditorPart getActiveEditor() {
		return this.fActiveEditor;
	}
}
