/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.infra.browser.editors;

import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.gmt.modisco.common.core.logging.MoDiscoLogger;
import org.eclipse.gmt.modisco.infra.browser.Messages;
import org.eclipse.gmt.modisco.infra.browser.MoDiscoBrowserPlugin;
import org.eclipse.gmt.modisco.infra.browser.core.AttributeItem;
import org.eclipse.gmt.modisco.infra.browser.core.LinkItem;
import org.eclipse.gmt.modisco.infra.browser.core.ModelElementItem;
import org.eclipse.gmt.modisco.infra.browser.customization.CustomizationEngine;
import org.eclipse.gmt.modisco.infra.browser.customization.OverlayIconImageInfo;
import org.eclipse.gmt.modisco.infra.query.core.exception.ModelQueryException;
import org.eclipse.gmt.modisco.infra.role.Role;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;

/**
 * Custom painter to add information on tree elements:
 * <ul>
 * <li>Draws a down-pointing arrow on ordered references when they are
 * effectively displayed in their original order, that is, when "sort instances"
 * is disabled.
 * <li>Underlined and struckthrough customizations
 * <li>Displays "stickers" for roles on the right of model elements that carry
 * one or more role(s) for which an icon has been provided through a
 * customization (uiCustom file)
 * </ul>
 */
public class CustomTreePainter {

	private static final int MAX_ALPHA = 255;
	private final BrowserConfiguration browserConfiguration;
	private final Tree fTree;

	private int mouseX;
	private int mouseY;
	private boolean hovering = false;

	public CustomTreePainter(final Tree tree, final BrowserConfiguration browserConfiguration) {
		this.fTree = tree;
		this.browserConfiguration = browserConfiguration;
		setupTreeCustomPaint(tree);

		tree.getHorizontalBar().addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				tree.redraw();
			}
		});
		tree.addMouseMoveListener(new MouseMoveListener() {
			public void mouseMove(final MouseEvent e) {
				CustomTreePainter.this.mouseX = e.x;
				CustomTreePainter.this.mouseY = e.y;
				if (CustomTreePainter.this.hovering) {
					CustomTreePainter.this.hovering = false;
					tree.redraw();
				}
			}
		});

		tree.addListener(SWT.MouseExit, new Listener() {
			public void handleEvent(final Event event) {
				if (CustomTreePainter.this.hovering) {
					CustomTreePainter.this.mouseX = 0;
					CustomTreePainter.this.mouseY = 0;
					CustomTreePainter.this.hovering = false;
					tree.redraw();
				}
			}
		});
		tree.addListener(SWT.MouseHover, new Listener() {
			public void handleEvent(final Event event) {
				CustomTreePainter.this.hovering = true;
				tree.redraw();
			}
		});
	}

	private boolean isOrderedReference(final Object object) {
		// when instances are sorted, references are not ordered anymore
		if (this.browserConfiguration.isSortInstances()) {
			return false;
		}

		if (object instanceof LinkItem) {
			final LinkItem linkItemProvider = (LinkItem) object;
			final EReference reference = linkItemProvider.getReference();
			return reference.isMany() && reference.isOrdered();
		}
		return false;
	}

	private boolean isOrderingEnabled() {
		return this.browserConfiguration.isShowOrdering();
	}

	public static final int ROLE_ICON_SIZE = 16;
	public static final int MARGIN = 8;
	public static final int SPACING = 5;

	private void handleMeasureItem(final Event event) {
		final TreeItem item = (TreeItem) event.item;
		final Object data = item.getData();

		if (isOrderingEnabled()) {
			if (isOrderedReference(data)) {
				final int leftMargin = 5;
				event.width += leftMargin + event.height / 2 + 1;
			}
		}

		if (data instanceof ModelElementItem) {
			ModelElementItem modelElementItem = (ModelElementItem) data;
			final EObject eObject = modelElementItem.getEObject();
			int maxX = this.fTree.getClientArea().width
					+ this.fTree.getHorizontalBar().getSelection();
			List<RoleToPaint> rolesToPaint = getRolesToPaintFor(eObject, event.x, event.width,
					event.y, event.height, maxX);
			if (!rolesToPaint.isEmpty()) {
				Rectangle lastRoleBounds = rolesToPaint.get(rolesToPaint.size() - 1).getBounds();
				event.width = lastRoleBounds.x + lastRoleBounds.width;
			}
		}
	}

	private void handlePaintItem(final Event event) {
		final TreeItem item = (TreeItem) event.item;
		final Object data = item.getData();
		if (isOrderingEnabled()) {
			if (isOrderedReference(data)) {
				paintOrderArrow(event);
			}
		}
		paintCustomization(data, event);
		paintRoleIcons(data, event);
	}

	public class RoleToPaint {
		private static final int INITIAL_ALPHA = 255;
		private final Rectangle bounds;
		private final Role role;
		private int alpha = CustomTreePainter.RoleToPaint.INITIAL_ALPHA;
		private final Rectangle itemBounds;
		private final Image image;
		private final boolean overlay;

		public RoleToPaint(final Rectangle bounds, final Rectangle itemBounds, final Role role,
				final Image image, final boolean overlay) {
			this.bounds = bounds;
			this.itemBounds = itemBounds;
			this.role = role;
			this.image = image;
			this.overlay = overlay;
		}

		public Rectangle getBounds() {
			return this.bounds;
		}

		public Rectangle getItemBounds() {
			return this.itemBounds;
		}

		public Role getRole() {
			return this.role;
		}

		public Image getImage() {
			return this.image;
		}

		public boolean isOverlay() {
			return this.overlay;
		}

		public void setAlpha(final int alpha) {
			this.alpha = alpha;
		}

		public int getAlpha() {
			return this.alpha;
		}
	}

	/**
	 * @param eObject
	 *            the model element for which roles are to be painted
	 * @param posX
	 *            the horizontal offset of the tree element relative to the tree
	 * @param width
	 *            the width of the tree element
	 * @param posY
	 *            the horizontal offset of the tree element relative to the tree
	 * @param height
	 *            the height of the tree element
	 * @param maxX
	 *            the maximum visible offset in the tree
	 * @return a list of roles to paint
	 */
	public List<CustomTreePainter.RoleToPaint> getRolesToPaintFor(final EObject eObject,
			final int posX, final int width, final int posY, final int height, final int maxX) {
		final int rightX = posX + width;
		final int margin = 8;
		int offset = margin;
		final int spacing = 5;
		final int iconWidth = 16;
		final int iconHeight = 16;
		final int overlayIconWidth = 8;
		final int overlayIconHeight = 8;

		List<CustomTreePainter.RoleToPaint> rolesToPaint = new ArrayList<CustomTreePainter.RoleToPaint>();

		int lastX = 0;

		try {
			final CustomizationEngine customizationEngine = this.browserConfiguration
					.getCustomizationEngine();
			Rectangle itemBounds = new Rectangle(posX, posY, width, height);
			for (final Role role : this.browserConfiguration.getRoleContext().getRoles(eObject)) {
				OverlayIconImageInfo roleOverlayIcon = customizationEngine.getRoleOverlayIcon(
						eObject, role);
				if (roleOverlayIcon != null) {
					Point overlayIconOffset = getOverlayIconOffset(roleOverlayIcon);
					Rectangle targetBounds = new Rectangle(posX + overlayIconOffset.x, posY
							+ overlayIconOffset.y, overlayIconWidth, overlayIconHeight);
					rolesToPaint.add(new RoleToPaint(targetBounds, itemBounds, role,
							roleOverlayIcon.getImage(), true));
				} else {
					Image mainIcon = customizationEngine.getRoleMainIcon(eObject, role);
					if (mainIcon != null) {
						// if the role icon is already displayed on the
						// main icon, don't display it again as a sticker
						continue;
					}
					Image customizedIcon = customizationEngine.getTypeIcon(eObject, role);
					// don't show role sticker when no icon is provided
					if (customizedIcon == null) {
						continue;
					}

					Rectangle targetBounds = new Rectangle(rightX + offset, posY, iconWidth,
							iconHeight);
					offset += iconWidth + spacing;
					rolesToPaint.add(new RoleToPaint(targetBounds, itemBounds, role,
							customizedIcon, false));
					lastX = targetBounds.x + targetBounds.width;
				}
			}
		} catch (ModelQueryException e) {
			MoDiscoBrowserPlugin.logException(Messages.CustomTreePainter_errorDrawingRoleIcon, e);
		}

		if (lastX > maxX) {
			int shift = lastX - maxX;
			ListIterator<CustomTreePainter.RoleToPaint> rolesToPaintIterator = rolesToPaint
					.listIterator();
			while (rolesToPaintIterator.hasNext()) {
				CustomTreePainter.RoleToPaint roleToPaint = rolesToPaintIterator.next();
				if (!roleToPaint.isOverlay()) {
					roleToPaint.getBounds().x -= shift;
					final int minVisibleText = 50;
					if (roleToPaint.getBounds().x < posX + minVisibleText) {
						rolesToPaintIterator.remove();
						continue;
					}
					int overlapWithText = rightX - roleToPaint.getBounds().x;
					if (overlapWithText > 0) {
						final int minAlpha = 128;
						final int multiplicator = 3;
						// the more it overlaps with text, the more
						// transparent it gets
						int alpha = Math.max(CustomTreePainter.MAX_ALPHA - overlapWithText
								* multiplicator, minAlpha);
						roleToPaint.setAlpha(alpha);
					}
				}
			}
		}

		return rolesToPaint;
	}

	private Point getOverlayIconOffset(final OverlayIconImageInfo roleOverlayIcon) {
		final int step = 8;
		switch (roleOverlayIcon.getIconPosition()) {
		case TopLeft:
			return new Point(0, 0);
		case TopMiddle:
			return new Point(step, 0);
		case TopRight:
			return new Point(step * 2, 0);
		case BottomLeft:
			return new Point(0, step);
		case BottomMiddle:
			return new Point(step, step);
		case BottomRight:
			return new Point(step * 2, step);
		default:
			MoDiscoLogger.logError("Unhandled overlay icon position", MoDiscoBrowserPlugin //$NON-NLS-1$
					.getPlugin());
		}
		return null;
	}

	private void paintRoleIcons(final Object data, final Event event) {
		if (data instanceof ModelElementItem) {
			ModelElementItem modelElementItem = (ModelElementItem) data;
			final EObject eObject = modelElementItem.getEObject();
			int maxX = this.fTree.getClientArea().width
					+ this.fTree.getHorizontalBar().getSelection();
			List<CustomTreePainter.RoleToPaint> rolesToPaint = getRolesToPaintFor(eObject, event.x,
					event.width, event.y, event.height, maxX);

			if (rolesToPaint.isEmpty()) {
				// enable default tooltip
				this.fTree.setToolTipText(null);
				return;
			} else {
				// disable default tooltip
				this.fTree.setToolTipText(""); //$NON-NLS-1$
			}

			// hide role stickers when hovering on the element text
			boolean showStickers = !(this.hovering && hoveringOnTextLeftOfRoles(rolesToPaint));

			for (RoleToPaint roleToPaint : rolesToPaint) {
				Image customizedIcon = roleToPaint.getImage();
				if (customizedIcon != null) {
					Rectangle bounds = customizedIcon.getBounds();
					Rectangle target = roleToPaint.getBounds();

					if (!roleToPaint.isOverlay()) {
						if (!showStickers) {
							continue;
						}
						boolean hoveringOverRole = this.hovering
								&& roleToPaint.getBounds().contains(this.mouseX, this.mouseY);

						int alpha = roleToPaint.getAlpha();
						if (hoveringOverRole) {
							alpha = CustomTreePainter.MAX_ALPHA;
						}
						event.gc.setAlpha(alpha);
					}

					event.gc.drawImage(customizedIcon, 0, 0, bounds.width, bounds.height, target.x,
							target.y, target.width, target.height);
				}
			}
		}
	}

	private boolean hoveringOnTextLeftOfRoles(final List<CustomTreePainter.RoleToPaint> rolesToPaint) {
		if (rolesToPaint.size() == 0) {
			return false;
		}
		if (rolesToPaint.get(0).getAlpha() == CustomTreePainter.MAX_ALPHA) {
			return false;
		}
		RoleToPaint first = rolesToPaint.get(0);
		return first.getItemBounds().contains(this.mouseX, this.mouseY)
				&& this.mouseX < first.getBounds().x;
	}

	private void paintCustomization(final Object element, final Event event) {
		final CustomizationEngine customizationEngine = this.browserConfiguration
				.getCustomizationEngine();
		boolean underlined = false;
		boolean struckthrough = false;

		if (element instanceof ModelElementItem) {
			final ModelElementItem elementItem = (ModelElementItem) element;
			final EObject eObject = elementItem.getEObject();
			underlined = customizationEngine.isTypeUnderlined(eObject.eClass(), eObject);
			struckthrough = customizationEngine.isTypeStruckthrough(eObject.eClass(), eObject);
		} else if (element instanceof AttributeItem) {
			final AttributeItem attributeItem = (AttributeItem) element;
			final EObject parent = attributeItem.getParent();
			underlined = customizationEngine.isAttributeUnderlined(attributeItem
					.roleOrParentClass(), attributeItem.getAttribute().getName(), parent);
			struckthrough = customizationEngine.isAttributeStruckthrough(attributeItem
					.roleOrParentClass(), attributeItem.getAttribute().getName(), parent);
		} else if (element instanceof LinkItem) {
			final LinkItem linkItem = (LinkItem) element;
			final EObject parent = linkItem.getParent();
			underlined = customizationEngine.isReferenceUnderlined(linkItem.roleOrParentClass(),
					linkItem.getReference().getName(), parent);
			struckthrough = customizationEngine.isReferenceStruckthrough(linkItem
					.roleOrParentClass(), linkItem.getReference().getName(), parent);
		}

		final int leftMargin = 20;
		final int rightMargin = 5;
		if (underlined) {
			final int y = event.y + event.height - 2;
			event.gc.drawLine(event.x + leftMargin, y, event.x + event.width - rightMargin, y);
		}

		if (struckthrough) {
			final int y = event.y + event.height / 2 + 1;
			event.gc.drawLine(event.x + leftMargin, y, event.x + event.width - rightMargin, y);
		}

	}

	private void paintOrderArrow(final Event event) {
		final int horizontalDivs = 4;
		final int leftMargin = 5;
		// to pass checkstyle...
		final int three = 3;

		int arrowSize = event.height / 2;
		// so that the pixels are always aligned on integer boundaries, to avoid
		// jaggies
		arrowSize -= arrowSize % horizontalDivs;
		int top = event.y + event.height / horizontalDivs;
		int left = event.x + event.width + leftMargin;

		final int x0 = left;
		final int x1 = left + arrowSize / horizontalDivs;
		final int x2 = left + arrowSize / 2;
		final int x3 = left + (arrowSize / horizontalDivs) * three;
		final int x4 = left + arrowSize;

		final int y0 = top;
		final int y1 = top + (arrowSize / 2);
		final int y2 = top + arrowSize;

		// draws an arrow, starting from the top left corner, clockwise
		event.gc.drawPolygon(new int[] { x1, y0, x3, y0, x3, y1, x4, y1, x2, y2, x0, y1, x1, y1 });
	}

	private void setupTreeCustomPaint(final Tree tree) {
		tree.addListener(SWT.MeasureItem, new Listener() {
			public void handleEvent(final Event event) {
				handleMeasureItem(event);
			}

		});
		tree.addListener(SWT.PaintItem, new Listener() {
			public void handleEvent(final Event event) {
				handlePaintItem(event);
			}
		});
	}

	public void dispose() {
	}
}
