/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.infra.browser.editors.table;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.ui.URIEditorInput;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory.Descriptor.Registry;
import org.eclipse.gmt.modisco.common.core.utils.StringUtils;
import org.eclipse.gmt.modisco.infra.browser.Messages;
import org.eclipse.gmt.modisco.infra.browser.MoDiscoBrowserPlugin;
import org.eclipse.gmt.modisco.infra.browser.core.LinkItem;
import org.eclipse.gmt.modisco.infra.browser.core.ModelElementItem;
import org.eclipse.gmt.modisco.infra.browser.editors.BrowserConfiguration;
import org.eclipse.gmt.modisco.infra.browser.editors.EcoreBrowser;
import org.eclipse.gmt.modisco.infra.browser.editors.table.ColumnDescription.Type;
import org.eclipse.gmt.modisco.infra.browser.editors.table.TableEditorInput.ElementsDescription;
import org.eclipse.gmt.modisco.infra.browser.util.ImageProvider;
import org.eclipse.gmt.modisco.infra.query.runtime.ModelQueryResult;
import org.eclipse.gmt.modisco.infra.role.RoleStructuralFeature;
import org.eclipse.gmt.modisco.infra.role.core.RoleContext;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.MenuDetectEvent;
import org.eclipse.swt.events.MenuDetectListener;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.EditorPart;
import org.eclipse.ui.part.FileEditorInput;

public class TableEditor extends EditorPart {

	private static final int LABEL_COLUMN_WIDTH = 125;
	private static final int DEFAULT_COLUMN_WIDTH = 100;
	private static final int MAX_INITIAL_COLUMN_WIDTH = 300;
	public static final String EDITOR_ID = "org.eclipse.gmt.modisco.infra.browser.table.editorID"; //$NON-NLS-1$
	/**
	 * The number of items above which a virtual table is used. The virtual
	 * style is not always used because it prevents correct packing of columns
	 * (since the elements' labels are not computed in advance).
	 */
	private static final int USE_VIRTUAL_THRESHOLD = 1000;

	/** The data key for columns that are hidden because they are empty */
	private static final String KEY_EMPTY_HIDDEN = "emptyHidden"; //$NON-NLS-1$

	/**
	 * The data key for columns that are hidden because the feature (attribute
	 * or reference) they represent is not common to all the elements
	 */
	private static final String KEY_NOT_COMMON_HIDDEN = "notCommonHidden"; //$NON-NLS-1$

	private TableViewer tableViewer;
	private TableEditorInput tableEditorInput;
	private TableEditorLabelProvider tableEditorLabelProvider;
	private List<ColumnDescription> columnsConfiguration;
	private TableEditorToolBar toolBar;
	private TableEditorContentProvider tableEditorContentProvider;

	@Override
	public void init(final IEditorSite site, final IEditorInput input) throws PartInitException {
		if (input instanceof TableEditorInput) {
			this.tableEditorInput = (TableEditorInput) input;
			setSite(site);
			setInput(this.tableEditorInput);
			setPartName(Messages.TableEditor_tableViewer);
		} else {
			throw new PartInitException("Input should be of type TableEditorInput"); //$NON-NLS-1$
		}
	}

	@Override
	public void createPartControl(final Composite parent) {

		final Composite editorComposite = new Composite(parent, SWT.NONE);
		final GridLayout gridLayout = new GridLayout(1, false);
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 2;
		editorComposite.setLayout(gridLayout);

		createToolBar(editorComposite);
		this.toolBar.setLabelText(this.tableEditorInput.getDescription());

		final GridData tableGridData = new GridData();
		tableGridData.grabExcessHorizontalSpace = true;
		tableGridData.grabExcessVerticalSpace = true;
		tableGridData.horizontalAlignment = SWT.FILL;
		tableGridData.verticalAlignment = SWT.FILL;

		int virtualFlag = 0;
		if (this.tableEditorInput.getElements().size() > TableEditor.USE_VIRTUAL_THRESHOLD) {
			virtualFlag = SWT.VIRTUAL;
		}

		this.tableViewer = new TableViewer(editorComposite, SWT.SINGLE | SWT.H_SCROLL
				| SWT.V_SCROLL | SWT.FULL_SELECTION | virtualFlag);
		this.tableViewer.getTable().setLayoutData(tableGridData);

		// use the same font size as in the main editor
		this.tableViewer.getTable().setFont(
				this.tableEditorInput.getBrowserConfiguration().getCustomFont());
		this.tableViewer.getTable().setLinesVisible(true);
		this.tableViewer.getTable().setHeaderVisible(true);

		this.tableEditorContentProvider = new TableEditorContentProvider();
		this.columnsConfiguration = createColumnConfiguration(false);

		this.tableEditorLabelProvider = new TableEditorLabelProvider(this.columnsConfiguration,
				this.tableEditorInput.getBrowserConfiguration(), this.tableEditorInput
						.getContextMap());

		createColumns(this.columnsConfiguration);

		this.tableViewer.setContentProvider(this.tableEditorContentProvider);
		this.tableViewer.setLabelProvider(this.tableEditorLabelProvider);
		this.tableViewer.setInput(this.tableEditorInput);

		createContextMenu();
		registerOpenListeners();
		registerCloseWhenMainEditorClosed();

		packColumns();
	}

	private void createToolBar(final Composite parent) {
		final Composite toolBarComposite = new Composite(parent, SWT.NONE);
		this.toolBar = new TableEditorToolBar(toolBarComposite, this);
	}

	private void registerCloseWhenMainEditorClosed() {
		final EcoreBrowser editor = this.tableEditorInput.getBrowserConfiguration().getEditor();
		if (editor == null) {
			return;
		}

		editor.getEditorSite().getPage().addPartListener(new IPartListener() {

			public void partClosed(final IWorkbenchPart part) {
				final EcoreBrowser editor = TableEditor.this.tableEditorInput
						.getBrowserConfiguration().getEditor();
				if (editor != null && part == editor) {
					TableEditor.this.getSite().getPage().closeEditor(TableEditor.this, false);
				}
			}

			public void partOpened(final IWorkbenchPart part) {
			}

			public void partDeactivated(final IWorkbenchPart part) {
			}

			public void partBroughtToTop(final IWorkbenchPart part) {
			}

			public void partActivated(final IWorkbenchPart part) {
			}
		});
	}

	private void createContextMenu() {
		final Menu menu = new Menu(this.tableViewer.getTable());

		this.tableViewer.getTable().addMenuDetectListener(new MenuDetectListener() {
			public void menuDetected(final MenuDetectEvent e) {
				final Point clickedPoint = new Point(e.x, e.y);

				final Point widgetRelativeClickedPoint = Display.getDefault().map(null,
						TableEditor.this.tableViewer.getTable(), clickedPoint);

				final ViewerCell cell = TableEditor.this.tableViewer
						.getCell(widgetRelativeClickedPoint);

				final int column;
				if (cell == null) {
					column = 0;
				} else {
					column = cell.getColumnIndex();
				}

				// clear dynamic items
				final MenuItem[] menuItems = menu.getItems();
				for (final MenuItem menuItem : menuItems) {
					menuItem.dispose();
				}

				final ColumnDescription columnDescription = TableEditor.this.columnsConfiguration
						.get(column);
				EObject element = null;
				final Object mainElement = getSelectedElement();
				final EObject mainModelElement;
				if (mainElement instanceof EObject) {
					mainModelElement = (EObject) mainElement;
				} else {
					return;
				}

				if (columnDescription.getType() == Type.REFERENCE) {
					boolean oneElement = true;
					if (columnDescription.getReference().isMany()) {
						// test whether the many-valued list has only 1 element
						@SuppressWarnings("unchecked")
						final EList<EObject> list = (EList<EObject>) getStructuralFeatureValue(
								mainModelElement, columnDescription.getReference());
						oneElement = list.size() == 1;
					}

					if (oneElement) {
						element = findElementToOpenInModelBrowser(column);
						if (element != null) {
							createBrowseMenuItemFor(element);
						}
					} else {
						createViewInTableMenuItemFor(mainModelElement, columnDescription
								.getReference());
					}
				} else if (columnDescription.getType() == Type.ECONTAINER) {
					element = mainModelElement.eContainer();
					if (element != null) {
						createBrowseMenuItemFor(element);
					}
				}

				if (element != mainModelElement) {
					createBrowseMenuItemFor(mainModelElement);
				}

				menu.setLocation(clickedPoint);
				menu.setVisible(true);
			}

			private void createBrowseMenuItemFor(final EObject element) {
				if (element == null) {
					return;
				}

				final String text = TableEditor.this.tableEditorLabelProvider.getTextFor(element);
				// Image image = tableEditorLabelProvider.getImageFor(element);

				final MenuItem browseMenuItem = new MenuItem(menu, SWT.PUSH);
				browseMenuItem.setText(NLS.bind(Messages.TableEditor_browse, StringUtils
						.truncateBeforeNewline(text)));
				// browseMenuItem.setImage(image);

				browseMenuItem.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(final SelectionEvent e) {
						openElementInModelBrowser(element);
					}
				});
			}

			private void createViewInTableMenuItemFor(final EObject parentElement,
					final EReference reference) {
				@SuppressWarnings("unchecked")
				final EList<EObject> elements = (EList<EObject>) getStructuralFeatureValue(
						parentElement, reference);

				final MenuItem menuItem = new MenuItem(menu, SWT.PUSH);
				menuItem.setText(Messages.openTableEditorOnSelection);

				if (elements.size() == 0) {
					menuItem.setEnabled(false);
				}

				menuItem.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(final SelectionEvent e) {
						final BrowserConfiguration browserConfiguration = TableEditor.this.tableEditorInput
								.getBrowserConfiguration();
						final String description = getEditorDescriptionFor(parentElement,
								reference, browserConfiguration);
						openOn(TableEditor.this.tableEditorInput.getBrowserConfiguration(),
								elements, description);
					}
				});
			}
		});
	}

	/**
	 * Registers listeners used to open an element in the main editor by
	 * double-click or &lt;Enter&gt;
	 */
	private void registerOpenListeners() {
		// double-click
		this.tableViewer.getTable().addMouseListener(new MouseAdapter() {
			@Override
			public void mouseDoubleClick(final MouseEvent e) {
				final ViewerCell cell = TableEditor.this.tableViewer.getCell(new Point(e.x, e.y));
				int column;
				if (cell == null) {
					column = 0;
				} else {
					column = cell.getColumnIndex();
				}

				openSelectionInModelBrowser(column);
			}
		});

		// enter
		this.tableViewer.getTable().addKeyListener(new KeyAdapter() {
			@Override
			public void keyPressed(final KeyEvent e) {
				if (e.keyCode == SWT.CR) {
					openSelectionInModelBrowser(0);
				}
			}
		});
	}

	/** pack the table's columns with a limit on max width */
	private void packColumns() {
		final TableColumn[] tableColumns = this.tableViewer.getTable().getColumns();
		for (final TableColumn tableColumn : tableColumns) {
			tableColumn.pack();
			if (tableColumn.getWidth() > TableEditor.MAX_INITIAL_COLUMN_WIDTH) {
				tableColumn.setWidth(TableEditor.MAX_INITIAL_COLUMN_WIDTH);
			}
		}
	}

	/**
	 * Create a configuration about what to display in columns
	 * 
	 * @param hideEmptyColumns
	 *            whether to hide empty columns
	 */
	private List<ColumnDescription> createColumnConfiguration(final boolean hideEmptyColumns) {
		// TODO: hideEmptyColumns unused
		ElementsDescription elementsDescription = this.tableEditorInput.getElementsDescription();

		final ArrayList<ColumnDescription> columns = new ArrayList<ColumnDescription>();
		columns.add(new ColumnDescription(ColumnDescription.Type.DEFAULT_LABEL));
		columns.add(new ColumnDescription(ColumnDescription.Type.METACLASS_NAME));
		if (elementsDescription.containsEObjects()) {
			columns.add(new ColumnDescription(ColumnDescription.Type.ECONTAINER));
		}
		if (this.tableEditorInput.getContextMap() != null) {
			// context, for query results
			columns.add(new ColumnDescription(ColumnDescription.Type.CONTEXT));
		}

		for (final EAttribute attribute : this.tableEditorInput.getAttributes()) {
			if (!hideEmptyColumns || !isEmpty(attribute)) {
				columns.add(new ColumnDescription(attribute));
			}
		}

		// first, non-derived references
		for (final EReference reference : this.tableEditorInput.getReferences()) {
			if (!reference.isDerived()) {
				if (!hideEmptyColumns || !isEmpty(reference)) {
					columns.add(new ColumnDescription(reference));
				}
			}
		}

		// then derived references
		for (final EReference reference : this.tableEditorInput.getReferences()) {
			if (reference.isDerived()) {
				if (!hideEmptyColumns || !isEmpty(reference)) {
					columns.add(new ColumnDescription(reference));
				}
			}
		}

		return columns;
	}

	/** Whether no element has this reference set */
	private boolean isEmpty(final EReference reference) {
		final Object[] elements = this.tableEditorContentProvider
				.getElements(this.tableEditorInput);
		for (final Object element : elements) {
			if (element instanceof EObject) {
				final EObject eObject = (EObject) element;
				if (hasStructuralFeature(eObject, reference)) {
					final Object value = getStructuralFeatureValue(eObject, reference);
					if (reference.isMany()) {
						@SuppressWarnings("unchecked")
						final EList<EObject> list = (EList<EObject>) value;
						if (!list.isEmpty()) {
							return false;
						}
					} else {
						if (value != null) {
							return false;
						}
					}
				}
			}
		}
		return true;
	}

	/** Whether no element has this attribute set */
	private boolean isEmpty(final EAttribute attribute) {
		final Object[] elements = this.tableEditorContentProvider
				.getElements(this.tableEditorInput);
		for (final Object element : elements) {
			if (element instanceof EObject) {
				final EObject eObject = (EObject) element;
				if (hasStructuralFeature(eObject, attribute)) {
					final Object value = getStructuralFeatureValue(eObject, attribute);
					if (attribute.isMany()) {
						@SuppressWarnings("unchecked")
						final EList<EObject> list = (EList<EObject>) value;
						if (!list.isEmpty()) {
							return false;
						}
					} else {
						if (value != null) {
							return false;
						}
					}
				}
			}
		}
		return true;
	}

	private void createColumns(final List<ColumnDescription> columns) {
		int columnIndex = 0;
		for (final ColumnDescription columnDescription : columns) {
			final TableViewerColumn column = new TableViewerColumn(this.tableViewer, SWT.NONE);
			final TableColumn tableColumn = column.getColumn();
			tableColumn.setWidth(TableEditor.DEFAULT_COLUMN_WIDTH);
			tableColumn.setResizable(true);
			tableColumn.setMoveable(true);
			column.getColumn().setData(columnDescription);

			switch (columnDescription.getType()) {
			// shows the same thing as in model browser tree
			case DEFAULT_LABEL:
				tableColumn.setText(Messages.TableEditor_columnName_Label);
				tableColumn.setWidth(TableEditor.LABEL_COLUMN_WIDTH);
				break;
			case METACLASS_NAME:
				tableColumn.setText(Messages.TableEditor_columnName_Metaclass);
				break;
			case CONTEXT:
				tableColumn.setText(Messages.TableEditor_ColumnName_queryContext);
				break;
			case ECONTAINER:
				tableColumn.setText("/eContainer"); //$NON-NLS-1$
				break;
			case ATTRIBUTE:
				final EAttribute attribute = columnDescription.getAttribute();
				if (attribute.isDerived()) {
					tableColumn.setText("/" + attribute.getName()); //$NON-NLS-1$
				} else {
					tableColumn.setText(attribute.getName());
				}
				tableColumn.setImage(ImageProvider.getInstance().getAttributeIcon());
				break;
			case REFERENCE:
				final EReference reference = columnDescription.getReference();
				final String multiplicity = LinkItem.getMultiplicity(reference);
				if (reference.isDerived()) {
					tableColumn.setText("/" + reference.getName() + multiplicity); //$NON-NLS-1$
				} else {
					tableColumn.setText(reference.getName() + multiplicity);
				}

				tableColumn.setImage(LinkItem.getImageFor(reference));
				// tableColumn.setImage(ImageProvider.getInstance().getLinkIcon());
				break;
			default:
				MoDiscoBrowserPlugin.logWarning("Unhandled column description type"); //$NON-NLS-1$
			}

			final int fColumnIndex = columnIndex;
			new TableSorter(this.tableViewer, column) {
				@Override
				protected int doCompare(final Viewer viewer, final Object e1, final Object e2) {
					// reuse the label provider to get the name to compare
					final String label1 = TableEditor.this.tableEditorLabelProvider.getColumnText(
							e1, fColumnIndex);
					final String label2 = TableEditor.this.tableEditorLabelProvider.getColumnText(
							e2, fColumnIndex);
					return label1.compareToIgnoreCase(label2);
				}
			};

			columnIndex++;
		}
	}

	@Override
	public void setFocus() {
		if (this.tableViewer != null) {
			this.tableViewer.getTable().setFocus();
		}
	}

	/**
	 * Whether the table editor can be opened on the given selection.
	 * <code>true</code> if the selection is a link or a collection of model
	 * objects.
	 */
	public static boolean canBeOpenedOnSelection(final ISelection selection) {
		if (selection instanceof IStructuredSelection) {
			final IStructuredSelection structuredSelection = (IStructuredSelection) selection;

			if (structuredSelection.size() == 1
					&& structuredSelection.getFirstElement() instanceof LinkItem) {
				final LinkItem linkItem = (LinkItem) structuredSelection.getFirstElement();
				return linkItem.getChildren().size() > 0;
			} else {
				boolean empty = true;
				final Iterator<?> iterator = structuredSelection.iterator();
				while (iterator.hasNext()) {
					final Object selectedElement = iterator.next();
					if (selectedElement instanceof ModelElementItem) {
						empty = false;
					} else {
						return false;
					}
				}
				return !empty;
			}
		}
		return false;
	}

	/**
	 * Open a new table view editor on the given list of elements.
	 * 
	 * @param browserConfiguration
	 *            the main editor configuration
	 * @param elements
	 *            the elements to show as rows in the table
	 * @param description
	 *            a descriptive String of what will be displayed in the table
	 *            view editor
	 */
	public static void openOn(final BrowserConfiguration browserConfiguration,
			final List<? extends Object> elements, final String description) {
		final TableEditorInput input = new TableEditorInput(browserConfiguration, elements,
				description);
		openOn(input);
	}

	private static void openOn(final TableEditorInput input) {
		final IWorkbenchPage activePage = PlatformUI.getWorkbench().getActiveWorkbenchWindow()
				.getActivePage();
		try {
			IDE.openEditor(activePage, input, TableEditor.EDITOR_ID, true);
		} catch (final PartInitException e) {
			MoDiscoBrowserPlugin.logException(e);
		}
	}

	/**
	 * Open a new table view editor on the given query result.
	 * 
	 * @param result
	 *            the results of a query to display in the table
	 */
	public static void openOn(final List<ModelQueryResult> result) {
		List<Object> elements = new ArrayList<Object>();
		final Map<Object, EObject> contextMap = new HashMap<Object, EObject>();

		for (ModelQueryResult modelQueryResult : result) {
			Object value = modelQueryResult.getValue();
			flattenToList(value, elements, modelQueryResult.getSource(), contextMap);
		}

		// TODO: query name?
		final String description = NLS.bind(Messages.TableEditor_Description_modelQueryResults,
				elements.size());

		BrowserConfiguration browserConfiguration = new BrowserConfiguration();
		final Registry registry = ComposedAdapterFactory.Descriptor.Registry.INSTANCE;
		ComposedAdapterFactory adapterFactoryWithRegistry = new ComposedAdapterFactory(registry);
		browserConfiguration.setAdapterFactoryWithRegistry(adapterFactoryWithRegistry);

		final TableEditorInput input = new TableEditorInput(browserConfiguration, elements,
				description, contextMap);
		openOn(input);
	}

	/**
	 * Flattens the contents of the given value (if it is {@link Iterable}) into
	 * <code>elements</code>, or add it directly to this list if it is not
	 * iterable. In addition, add in the context map the binding between each
	 * element and the given context.
	 */
	private static void flattenToList(final Object value, final List<Object> elements,
			final EObject context, final Map<Object, EObject> contextMap) {
		if (value instanceof Iterable<?>) {
			Iterable<?> iterable = (Iterable<?>) value;
			Iterator<?> iterator = iterable.iterator();
			while (iterator.hasNext()) {
				Object element = iterator.next();
				flattenToList(element, elements, context, contextMap);
			}
		} else {
			elements.add(value);
			contextMap.put(value, context);
		}
	}

	private void openSelectionInModelBrowser(final int column) {
		final EObject element = findElementToOpenInModelBrowser(column);
		if (element != null) {
			openElementInModelBrowser(element);
		}
	}

	/** @return the first selected element or <code>null</code> if none */
	private Object getSelectedElement() {
		final ISelection selection = this.tableViewer.getSelection();
		if (selection instanceof IStructuredSelection) {
			final IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			return structuredSelection.getFirstElement();
		}
		return null;
	}

	/**
	 * Find and return the element that should be opened when the given column
	 * is clicked
	 */
	private EObject findElementToOpenInModelBrowser(final int column) {
		final Object mainElement = getSelectedElement();
		if (mainElement == null) {
			return null;
		}
		final EObject mainModelElement;
		if (mainElement instanceof EObject) {
			mainModelElement = (EObject) mainElement;
		} else {
			mainModelElement = null;
		}

		EObject targetObject = null;
		if (column > 0) {
			final ColumnDescription columnDescription = this.columnsConfiguration.get(column);
			final Object value;
			switch (columnDescription.getType()) {
			case ATTRIBUTE:
				if (mainModelElement != null) {
					value = getStructuralFeatureValue(mainModelElement, columnDescription
							.getAttribute());
					if (value instanceof EObject) {
						targetObject = (EObject) value;
					}
				}
				break;
			case REFERENCE:
				if (mainModelElement != null) {
					value = getStructuralFeatureValue(mainModelElement, columnDescription
							.getReference());
					if (columnDescription.getReference().isMany()) {
						@SuppressWarnings("unchecked")
						final EList<EObject> list = (EList<EObject>) value;
						if (list.size() == 1) {
							targetObject = list.get(0);
						}
					} else {
						targetObject = (EObject) value;
					}
				}
				break;
			case ECONTAINER:
				if (mainModelElement != null) {
					value = mainModelElement.eContainer();
					if (value instanceof EObject) {
						targetObject = (EObject) value;
					}
				}
				break;
			case CONTEXT:
				Map<Object, EObject> contextMap = this.tableEditorInput.getContextMap();
				if (contextMap != null) {
					targetObject = contextMap.get(mainElement);
				}
				break;
			default:
				targetObject = null;
				break;
			}
		}

		if (targetObject != null) {
			return targetObject;
		} else {
			return mainModelElement;
		}
	}

	private void openElementInModelBrowser(final EObject element) {
		final EcoreBrowser editor = TableEditor.this.tableEditorInput.getBrowserConfiguration()
				.getEditor();
		if (editor != null) {
			editor.getSite().getPage().activate(editor);
			editor.browseTo(element);
		} else {
			// try to re-open the element in the same browser it came from
			Resource resource = element.eResource();
			if (resource != null) {
				ResourceSet resourceSet = resource.getResourceSet();
				IEditorReference[] editorReferences = getSite().getPage().getEditorReferences();
				for (IEditorReference editorReference : editorReferences) {
					IEditorPart anEditor = editorReference.getEditor(false);
					if (anEditor instanceof EcoreBrowser) {
						EcoreBrowser browser = (EcoreBrowser) anEditor;
						if (browser.getResourceSet() == resourceSet) {
							browser.browseTo(element);
							browser.getSite().getPage().activate(browser);
							return;
						}
					}
				}
			}

			// if that failed (the editor is now closed maybe), then open a new
			// model browser and select the element
			final IWorkbenchPage activePage = PlatformUI.getWorkbench().getActiveWorkbenchWindow()
					.getActivePage();
			try {
				final URI elementURI = EcoreUtil.getURI(element);
				URI resourceURI = elementURI.trimFragment();
				final IEditorInput editorInput;
				if (resourceURI.isPlatformResource()) {
					String platformString = resourceURI.toPlatformString(true);
					IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember(
							platformString);
					if (res instanceof IFile) {
						IFile file = (IFile) res;
						editorInput = new FileEditorInput(file);
					} else {
						MoDiscoBrowserPlugin.logError("Cannot open model: " + resourceURI); //$NON-NLS-1$
						return;
					}
				} else {
					editorInput = new URIEditorInput(resourceURI);
				}

				IEditorPart newEditor = IDE.openEditor(activePage, editorInput,
						EcoreBrowser.EDITOR_ID, true);
				if (newEditor instanceof EcoreBrowser) {
					EcoreBrowser browser = (EcoreBrowser) newEditor;
					browser.browseToByURI(elementURI);
				}
			} catch (final PartInitException e) {
				MoDiscoBrowserPlugin.logException(e);
			}
		}
	}

	@Override
	public boolean isDirty() {
		return false;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	@Override
	public void doSave(final IProgressMonitor monitor) {
	}

	@Override
	public void doSaveAs() {
	}

	/** Hide columns which don't contain any elements */
	public void setHideEmptyColumns(final boolean hideEmptyColumns) {
		try {
			this.tableViewer.getControl().setRedraw(false);
			for (final ColumnDescription columnDescription : this.columnsConfiguration) {
				boolean bEmpty = false;
				if (columnDescription.getType() == Type.ATTRIBUTE) {
					bEmpty = isEmpty(columnDescription.getAttribute());
				} else if (columnDescription.getType() == Type.REFERENCE) {
					bEmpty = isEmpty(columnDescription.getReference());
				}

				final TableColumn column = findColumn(columnDescription);
				if (hideEmptyColumns && bEmpty) {
					if (column != null) {
						// indicates that the column is hidden because it is
						// empty
						column.setData(TableEditor.KEY_EMPTY_HIDDEN, Boolean.TRUE);
						hideColumn(column);
					}
				} else if (!hideEmptyColumns
						&& Boolean.TRUE == column.getData(TableEditor.KEY_EMPTY_HIDDEN)) {
					column.setData(TableEditor.KEY_EMPTY_HIDDEN, Boolean.FALSE);
					showEmptyColumn(column);
				}
			}
		} finally {
			this.tableViewer.getControl().setRedraw(true);
		}
	}

	/**
	 * Whether to show columns which are specific to a few elements (false), or
	 * only columns which are common to all the elements (true)
	 */
	public void setOnlyShowCommonColumns(final boolean onlyShowCommonColumns) {
		try {
			this.tableViewer.getControl().setRedraw(false);
			final EClass[] metaclasses = this.tableEditorInput.getMetaclasses();

			for (final ColumnDescription columnDescription : this.columnsConfiguration) {
				boolean common = true;
				if (columnDescription.getType() == Type.ATTRIBUTE) {
					common = isCommon(columnDescription.getAttribute(), metaclasses);
				} else if (columnDescription.getType() == Type.REFERENCE) {
					common = isCommon(columnDescription.getReference(), metaclasses);
				}

				final TableColumn column = findColumn(columnDescription);
				if (onlyShowCommonColumns && !common) {
					if (column != null) {
						// indicates that the column is hidden because its
						// feature
						// is not common to all the elements
						column.setData(TableEditor.KEY_NOT_COMMON_HIDDEN, Boolean.TRUE);
						hideColumn(column);
					}
				} else if (!onlyShowCommonColumns
						&& Boolean.TRUE == column.getData(TableEditor.KEY_NOT_COMMON_HIDDEN)) {
					column.setData(TableEditor.KEY_NOT_COMMON_HIDDEN, Boolean.FALSE);
					showEmptyColumn(column);
				}
			}
		} finally {
			this.tableViewer.getControl().setRedraw(true);
		}
	}

	/** Hide the given table column */
	private void hideColumn(final TableColumn column) {
		column.setWidth(0);
		column.setResizable(false);
	}

	/** Show the given column after checking that it shouldn't be hidden anymore */
	private void showEmptyColumn(final TableColumn column) {
		if (Boolean.TRUE != column.getData(TableEditor.KEY_EMPTY_HIDDEN)
				&& Boolean.TRUE != column.getData(TableEditor.KEY_NOT_COMMON_HIDDEN)) {
			column.pack();
			column.setResizable(true);
		}
	}

	/** @return whether the given feature is common to all the given metaclasses */
	private boolean isCommon(final EStructuralFeature feature, final EClass[] metaclasses) {
		for (final EClass metaclass : metaclasses) {
			if (!metaclass.getEAllStructuralFeatures().contains(feature)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Find the column with the given {@link ColumnDescription}
	 * 
	 * @return the column
	 */
	private TableColumn findColumn(final ColumnDescription columnDescription) {
		final TableColumn[] columns = this.tableViewer.getTable().getColumns();
		for (final TableColumn tableColumn : columns) {
			if (tableColumn.getData() == columnDescription) {
				return tableColumn;
			}
		}
		return null;
	}

	/**
	 * Get a descriptive label that will be shown when a table editor is opened
	 * on the given reference
	 */
	public static String getEditorDescriptionFor(final EObject parent, final EReference reference,
			final BrowserConfiguration browserConfiguration) {

		final String parentLabel = new ModelElementItem(parent, null, browserConfiguration)
				.getText();
		return NLS.bind(Messages.TableEditor_contentsDescription, parentLabel, reference.getName());
	}

	private Object getStructuralFeatureValue(final EObject eObject,
			final EStructuralFeature structuralFeature) {
		if (structuralFeature instanceof RoleStructuralFeature) {
			try {
				RoleContext roleContext = this.tableEditorInput.getBrowserConfiguration()
						.getRoleContext();
				return roleContext.get(eObject, structuralFeature);
			} catch (Exception e) {
				MoDiscoBrowserPlugin.logException(e);
				return null;
			}
		} else {
			return eObject.eGet(structuralFeature);
		}
	}

	private boolean hasStructuralFeature(final EObject eObject,
			final EStructuralFeature structuralFeature) {
		final EClass eClass = eObject.eClass();
		RoleContext roleContext = this.tableEditorInput.getBrowserConfiguration().getRoleContext();
		if (roleContext.getRoleFeatures(eObject).contains(structuralFeature)) {
			return true;
		}
		return eClass.getEAllStructuralFeatures().contains(structuralFeature);
	}
}
