/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.infra.browser.editors.table;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.gmt.modisco.infra.browser.Messages;
import org.eclipse.gmt.modisco.infra.browser.MoDiscoBrowserPlugin;
import org.eclipse.gmt.modisco.infra.browser.core.InstancesForMetaclass;
import org.eclipse.gmt.modisco.infra.browser.editors.BrowserConfiguration;
import org.eclipse.gmt.modisco.infra.browser.util.EMFUtil;
import org.eclipse.gmt.modisco.infra.query.core.exception.ModelQueryException;
import org.eclipse.gmt.modisco.infra.role.Role;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPersistableElement;

public class TableEditorInput implements IEditorInput {

	/** The elements that are displayed in the table */
	private final List<? extends Object> elements;
	/** Union of all the elements' attributes */
	private final List<EAttribute> attributes;
	/** Union of all the elements' references */
	private final List<EReference> references;

	/** The main browser's configuration */
	private final BrowserConfiguration browserConfiguration;
	/** A textual description of what is shown in the browser */
	private final String description;
	/** All the metaclasses of the elements */
	private HashSet<EClass> metaclasses;
	/**
	 * Associations between elements and the context they came from (in the case
	 * of a model query)
	 */
	private Map<Object, EObject> contextMap;

	/**
	 * Create an input for all the given elements
	 * 
	 * @param browserConfiguration
	 *            the main browser's configuration
	 * @param elements
	 *            the elements to show as rows in the table
	 * @param description
	 *            a descriptive String of what will be displayed in the table
	 *            view editor
	 * */
	public TableEditorInput(final BrowserConfiguration browserConfiguration,
			final List<? extends Object> elements, final String description) {
		this.browserConfiguration = browserConfiguration;
		this.elements = elements;
		this.description = description;

		this.metaclasses = new HashSet<EClass>();
		for (final Object element : elements) {
			if (element instanceof EObject) {
				EObject eObject = (EObject) element;
				this.metaclasses.add(eObject.eClass());
				try {
					List<Role> roles = this.browserConfiguration.getRoleContext().getRoles(eObject);
					this.metaclasses.addAll(roles);
				} catch (ModelQueryException e) {
					MoDiscoBrowserPlugin.logException(e);
				}
			}
		}

		// all the attributes of the metaclasses of the elements
		final HashSet<EAttribute> attributeSet = new HashSet<EAttribute>();
		for (final EClass eClass : this.metaclasses) {
			attributeSet.addAll(eClass.getEAllAttributes());
		}

		// all the references of the metaclasses of the elements
		final HashSet<EReference> referenceSet = new HashSet<EReference>();
		for (final EClass eClass : this.metaclasses) {
			referenceSet.addAll(eClass.getEAllReferences());
		}

		this.attributes = new ArrayList<EAttribute>(attributeSet);
		this.references = new ArrayList<EReference>(referenceSet);
	}

	/**
	 * Create an input for all the given elements (in the case of a model query)
	 * 
	 * @param browserConfiguration
	 *            the main browser's configuration
	 * @param elements
	 *            the elements to show as rows in the table
	 * @param description
	 *            a descriptive String of what will be displayed in the table
	 *            view editor
	 * @param contextMap
	 *            associations between elements and the context that what used
	 *            in the query to find them
	 * */
	public TableEditorInput(final BrowserConfiguration browserConfiguration,
			final List<? extends Object> elements, final String description,
			final Map<Object, EObject> contextMap) {
		this(browserConfiguration, elements, description);
		this.contextMap = contextMap;
	}

	/**
	 * Create an input for all elements of the given class
	 * 
	 * @param browserConfiguration
	 *            the main editor configuration
	 * @param eClass
	 *            the metaclass of the elements to show
	 * @param description
	 *            a descriptive String of what will be displayed in the table
	 *            view editor
	 * */
	public TableEditorInput(final BrowserConfiguration browserConfiguration, final EClass eClass,
			final String description) {
		this.browserConfiguration = browserConfiguration;
		this.description = description;
		final InstancesForMetaclass instances = this.browserConfiguration
				.getInstancesForMetaclasses().getInstancesForMetaclass(
						EMFUtil.getMetaclassQualifiedName(eClass));
		this.elements = new ArrayList<EObject>(instances.getElements());
		this.attributes = new ArrayList<EAttribute>(eClass.getEAllAttributes());
		this.references = new ArrayList<EReference>(eClass.getEAllReferences());
	}

	public List<? extends Object> getElements() {
		return this.elements;
	}

	public List<EAttribute> getAttributes() {
		return this.attributes;
	}

	public List<EReference> getReferences() {
		return this.references;
	}

	public String getDescription() {
		return this.description;
	}

	public BrowserConfiguration getBrowserConfiguration() {
		return this.browserConfiguration;
	}

	public boolean exists() {
		return false;
	}

	public ImageDescriptor getImageDescriptor() {
		return null;
	}

	public String getName() {
		return Messages.TableEditorInput_editorName_TableViewer;
	}

	public IPersistableElement getPersistable() {
		return null;
	}

	public String getToolTipText() {
		return Messages.TableEditorInput_tooltip_TableViewer;
	}

	@SuppressWarnings("unchecked")
	public Object getAdapter(final Class adapter) {
		return null;
	}

	public EClass[] getMetaclasses() {
		return this.metaclasses.toArray(new EClass[this.metaclasses.size()]);
	}

	public class ElementsDescription {
		private boolean containsEObjects;
		private boolean containsPrimitiveTypes;

		public boolean containsEObjects() {
			return this.containsEObjects;
		}

		public boolean containsPrimitiveTypes() {
			return this.containsPrimitiveTypes;
		}
	}

	public ElementsDescription getElementsDescription() {
		ElementsDescription elementsDescription = new ElementsDescription();
		for (Object object : this.elements) {
			if (object instanceof EObject) {
				elementsDescription.containsEObjects = true;
			} else {
				elementsDescription.containsPrimitiveTypes = true;
			}
		}
		return elementsDescription;
	}

	public Map<Object, EObject> getContextMap() {
		return this.contextMap;
	}
}
