/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.infra.browser.editors.table;

import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.gmt.modisco.common.core.utils.StringUtils;
import org.eclipse.gmt.modisco.infra.browser.Messages;
import org.eclipse.gmt.modisco.infra.browser.MoDiscoBrowserPlugin;
import org.eclipse.gmt.modisco.infra.browser.core.ModelElementItem;
import org.eclipse.gmt.modisco.infra.browser.editors.BrowserConfiguration;
import org.eclipse.gmt.modisco.infra.role.RoleAttribute;
import org.eclipse.gmt.modisco.infra.role.RoleReference;
import org.eclipse.gmt.modisco.infra.role.core.RoleContext;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.graphics.Image;

public class TableEditorLabelProvider extends LabelProvider implements ITableLabelProvider {

	private final BrowserConfiguration browserConfiguration;
	private final List<ColumnDescription> columns;
	private final Map<Object, EObject> contextMap;

	public TableEditorLabelProvider(final List<ColumnDescription> columns,
			final BrowserConfiguration browserConfiguration, final Map<Object, EObject> contextMap) {
		this.columns = columns;
		this.browserConfiguration = browserConfiguration;
		this.contextMap = contextMap;
	}

	public String getTextFor(final Object element) {
		if (element == null) {
			return "null"; //$NON-NLS-1$
		}
		if (element instanceof EObject) {
			final EObject eObject = (EObject) element;
			return new ModelElementItem(eObject, null, this.browserConfiguration).getText();
		} else {
			// primitive DataType
			return element.toString();
		}
	}

	public Image getImageFor(final Object element) {
		if (element == null) {
			return null;
		}

		if (element instanceof EObject) {
			final EObject eObject = (EObject) element;
			return new ModelElementItem(eObject, null, this.browserConfiguration).getImage();
		}

		return null;
	}

	public Image getColumnImage(final Object element, final int columnIndex) {
		if (columnIndex < this.columns.size()) {
			final ColumnDescription columnDescription = this.columns.get(columnIndex);

			switch (columnDescription.getType()) {
			case DEFAULT_LABEL:
				return getImageFor(element);
			case REFERENCE:
				final EReference reference = columnDescription.getReference();
				final EObject eObject = (EObject) element;

				final Object refValue;
				if (reference instanceof RoleReference) {
					try {
						RoleContext roleContext = this.browserConfiguration.getRoleContext();
						if (roleContext.getReferences(eObject).contains(reference)) {
							refValue = roleContext.get(eObject, reference);
						} else {
							return null;
						}
					} catch (Exception e) {
						MoDiscoBrowserPlugin.logException(e);
						return null;
					}
				} else {
					if (!eObject.eClass().getEAllReferences().contains(reference)) {
						return null;
					} else {
						refValue = eObject.eGet(reference);
					}
				}

				if (reference.isMany()) {
					@SuppressWarnings("unchecked")
					final EList<EObject> list = (EList<EObject>) refValue;

					// when only 1 element, then display its icon directly
					if (list.size() == 1) {
						return getImageFor(list.get(0));
					} else {
						return null;
					}
				} else {
					if (refValue != null) {
						return getImageFor(refValue);
					}
				}
			case ECONTAINER:
				return getImageFor(((EObject) element).eContainer());
			case CONTEXT:
				return getContextColumnImage(element);
			default:
				return null;
			}
		}

		return null;
	}

	public String getColumnText(final Object element, final int columnIndex) {
		if (columnIndex >= this.columns.size()) {
			return Messages.TableEditorLabelProvider_indexOutOfBounds;
		}

		final ColumnDescription columnDescription = this.columns.get(columnIndex);
		switch (columnDescription.getType()) {
		case DEFAULT_LABEL:
			return getTextFor(element);
		case METACLASS_NAME:
			return getMetaclassColumnText(element);
		case CONTEXT:
			return getContextColumnText(element);
		case ATTRIBUTE:
			return getAttributeColumnText(element, columnDescription);
		case REFERENCE:
			return getReferenceColumnText(element, columnDescription);
		case ECONTAINER:
			return getTextFor(((EObject) element).eContainer());
		default:
			return Messages.TableEditorLabelProvider_unexpectedColumnType;
		}
	}

	private String getMetaclassColumnText(final Object element) {
		if (element instanceof EObject) {
			EObject eObject = (EObject) element;
			final EClass eClass = eObject.eClass();
			if (eClass != null) {
				return eClass.getName();
			} else {
				return Messages.TableEditorLabelProvider_none;
			}
		} else {
			// primitive DataType
			return element.getClass().getSimpleName();
		}
	}

	private String getContextColumnText(final Object element) {
		if (this.contextMap != null) {
			EObject eObject = this.contextMap.get(element);
			return getTextFor(eObject);
		}
		return Messages.TableEditorLabelProvider_nullContext;
	}

	private Image getContextColumnImage(final Object element) {
		if (this.contextMap != null) {
			EObject eObject = this.contextMap.get(element);
			return getImageFor(eObject);
		}
		return null;
	}

	private String getAttributeColumnText(final Object element,
			final ColumnDescription columnDescription) {
		final EObject eObject = (EObject) element;
		final EAttribute attribute = columnDescription.getAttribute();

		final Object attrValue;
		if (attribute instanceof RoleAttribute) {
			try {
				RoleContext roleContext = this.browserConfiguration.getRoleContext();
				if (roleContext.getAttributes(eObject).contains(attribute)) {
					attrValue = roleContext.get(eObject, attribute);
				} else {
					return Messages.TableEditorLabelProvider_NA;
				}
			} catch (Exception e) {
				MoDiscoBrowserPlugin.logException(e);
				return Messages.TableEditorLabelProvider_error;
			}
		} else {
			if (!eObject.eClass().getEAllAttributes().contains(attribute)) {
				return Messages.TableEditorLabelProvider_NA;
			} else {
				attrValue = eObject.eGet(attribute);
			}
		}

		if (attrValue == null) {
			return "null"; //$NON-NLS-1$
		} else {
			return StringUtils.truncateBeforeNewline(attrValue.toString());
		}
	}

	private String getReferenceColumnText(final Object element,
			final ColumnDescription columnDescription) {
		final EObject eObject = (EObject) element;
		final EReference reference = columnDescription.getReference();
		final Object refValue;

		if (reference instanceof RoleReference) {
			try {
				RoleContext roleContext = this.browserConfiguration.getRoleContext();
				if (roleContext.getReferences(eObject).contains(reference)) {
					refValue = roleContext.get(eObject, reference);
				} else {
					return Messages.TableEditorLabelProvider_NA;
				}
			} catch (Exception e) {
				MoDiscoBrowserPlugin.logException(e);
				return Messages.TableEditorLabelProvider_error;
			}
		} else {
			if (!eObject.eClass().getEAllReferences().contains(reference)) {
				return Messages.TableEditorLabelProvider_NA;
			} else {
				refValue = eObject.eGet(reference);
			}
		}

		if (refValue == null) {
			return "null"; //$NON-NLS-1$
		} else if (reference.isMany()) {
			@SuppressWarnings("unchecked")
			final EList<EObject> list = (EList<EObject>) refValue;

			// when only 1 element, then display it directly
			if (list.size() == 1) {
				return getTextFor(list.get(0));
			}

			return "(" + list.size() + ")"; //$NON-NLS-1$ //$NON-NLS-2$
		} else {
			return getTextFor(refValue);
		}
	}
}
